package edu.jhu.ece.iacl.plugins.dti.tractography;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import javax.vecmath.Point2i;

import edu.jhu.bme.smile.commons.math.StatisticsDouble;
import edu.jhu.ece.iacl.algorithms.dti.tractography.FiberStatistics;
import edu.jhu.ece.iacl.algorithms.manual_label.LabelImage;
import edu.jhu.ece.iacl.algorithms.manual_label.ROI;
import edu.jhu.ece.iacl.jist.io.*;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.fiber.FiberCollection;
import edu.jhu.ece.iacl.jist.structures.geom.GridPt;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;


public class MedicAlgorithmApplyFiberROIStats extends ProcessingAlgorithm {
	//input param
	private ParamObject<FiberCollection> fiberset;
	private ParamVolume roi;
	private ParamVolume statvolume;
	private ParamOption outputtype;
//	private ParamOption statistic;
	private ParamBoolean outputFiberSets;
//	private ParamBoolean outputConnectionStatTables;
//	private ParamBoolean splitFibers;
	
	//output param
	private ParamFileCollection fibersubset;
	private ParamFileCollection connStatTables;
	private ParamObject<double[][]> labelList;
	private ParamFileCollection connMatrices;
	
	private FiberCollectionReaderWriter fcrw = FiberCollectionReaderWriter.getInstance();
	private CurveVtkReaderWriter cvrw = CurveVtkReaderWriter.getInstance();
	private ArrayDoubleTxtReaderWriter arw = ArrayDoubleTxtReaderWriter.getInstance();
	
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Gets a subset of the input fibers based on the input volumetric ROI.";
	private static final String longDescription = "";


	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(fiberset = new ParamObject<FiberCollection>("Fiber Set", new FiberCollectionReaderWriter()));
		inputParams.add(roi = new ParamVolume("ROI"));
		inputParams.add(statvolume = new ParamVolume("Volume"));
		statvolume.setMandatory(false);
		inputParams.add(outputtype=new ParamOption("Output Type",new String[]{"DtiStudio dat","Vtk"}));
		//inputParams.add(statistic=new ParamOption("Statistic",new String[]{"Fiber Length","Volume Mean","Volume Median","Volume Min","Volume Max"}));
		inputParams.add(outputFiberSets = new ParamBoolean("Output Fiber SubSets",false));
		//inputParams.add(outputConnectionStatTables = new ParamBoolean("Output Fiber Data Tables",false));
		//inputParams.add(splitFibers = new ParamBoolean("Split Fibers",false));
		
		inputParams.setPackage("IACL");
		inputParams.setCategory("DTI.Fiber");
		inputParams.setLabel("Apply Fiber ROI Stats");
		inputParams.setName("Apply_Fiber_ROI Stats");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("");
		info.add(new AlgorithmAuthor("John Bogovic", "bogovic@jhu.edu", "http://putter.ece.jhu.edu/John"));
		info.setAffiliation("Johns Hopkins University, Departments of Electrical and Biomedical Engineering");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.UNKNOWN);
	}


	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(fibersubset = new ParamFileCollection("Fiber Sub-Sets",new FileExtensionFilter(new String[]{"dat","vtk"})));
		fibersubset.setMandatory(false);
		outputParams.add(labelList = new ParamObject<double[][]>("Labels found", new ArrayDoubleTxtReaderWriter()));
		labelList.setMandatory(false);
		outputParams.add(connStatTables = new ParamFileCollection("Connection Statistics",new FileExtensionFilter(new String[]{"txt","csv"})));
		connStatTables.setMandatory(false);
		outputParams.add(connMatrices = new ParamFileCollection("Connectivity Matrices",new FileExtensionFilter(new String[]{"txt","csv"})));
		connMatrices.setMandatory(false);
	//	connStatTables
	}


	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		File dir = new File(this.getOutputDirectory()+File.separator+edu.jhu.ece.iacl.jist.utility.FileUtil.forceSafeFilename(this.getAlgorithmName()));
		try{
			if(!dir.isDirectory()){
				(new File(dir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){
			e.printStackTrace();
		}
		
		// THE ROI OBJECT DOES MOST OF THE WORK
		ROI mask = new  ROI(roi.getImageData());
		
		// list of the unique labels in the roi
		ArrayList<Integer> labellist = mask.getLabelList();
		Collections.sort(labellist);
		System.out.println("Found Labels");
		
		//write labels
		try{
			// file with sorted label list
			arw.write(toArrayInt(labellist), new File(dir.getCanonicalPath()+File.separator+"Labels.csv"));
		}catch(IOException e){
			e.printStackTrace();
		}
		
		/*
		 * REPRESENT A "CONNECTION" AS A TUPLE 
		 * - contains the both labels being connected.
		 * - convention: label w/ lower numeric value comes first
		 *  (a,b)  a<b 
		 *  is the connection between labels a and b
		 * 
		 */
		
		
		/* connsToFibs lets you access all the fibers belonging to connection c with:
		 *  conntsToFibs.get(c)  */
		HashMap<Point2i, FiberCollection> connsToFibs;
		
		
		/* connsToPts lets you access all the VOXELS belonging to connection c with:
		 *  connsToPts.get(c) */
		HashMap<Point2i, HashSet<GridPt>> connsToPts;
		
		
		//write new output
		HashMap<Point2i, Float>  meanVolData;  	// stores the mean 'FA' for each connection
		HashMap<Point2i, Integer> fbrCountData;	// stores the number of fibers for each connection
		
		connsToFibs = mask.fiberroiChopMean(fiberset.getObject(), labellist, statvolume.getImageData());
	    meanVolData = mask.meanVolData;
		fbrCountData = mask.fbrCountData;
		
		
//		System.out.println()
		//write output
		if(outputFiberSets.getValue()){
			System.out.println("\nKey Set: "+connsToFibs.keySet());
			System.out.println("Writing Fiber subsets");
			ArrayList<File> fiberfilelist = new ArrayList<File>();
			for(Point2i key : connsToFibs.keySet()){
				System.out.println("This key: " +key);
				System.out.println("This fiberColl: " +connsToFibs.get(key));
				File f = writeFibers(connsToFibs.get(key),dir);
				fiberfilelist.add(f);
			}
			fibersubset.setValue(fiberfilelist);
		}
		
		Set<Point2i> hashIdx = meanVolData.keySet();
		
		double[][] fcount;
		double[][] meanconnmtx;
		fcount = new double[labellist.size()][labellist.size()];
		meanconnmtx = new double[labellist.size()][labellist.size()];
		
		
		for(Point2i key:hashIdx){
		
			meanconnmtx[labellist.indexOf(key.x)][labellist.indexOf(key.y)] = meanVolData.get(key); 
			fcount[labellist.indexOf(key.x)][labellist.indexOf(key.y)] = fbrCountData.get(key);	
			
		}
		
			
			try{
			connStatTables.add(arw.write(meanconnmtx, 
					new File(dir.getCanonicalFile()+File.separator+roi.getImageData().getName()+"_meanFAConnMtx.csv")));

			connStatTables.add(arw.write(fcount, 
					new File(dir.getCanonicalFile()+File.separator+roi.getImageData().getName()+"_fbrCountConnMtx.csv")));

		}catch(IOException e){
			e.printStackTrace();
		}
		
		
	}

	private File writeFibers(FiberCollection fibers, File dir){
		File out = null;
		if(outputtype.getIndex()==0){
			out = fcrw.write(fibers, dir);
		}else{
			out = cvrw.write(fibers.toCurveCollection(), dir);
		}
		return out;
	}

	private double[][] toArrayInt(ArrayList<Integer> in){
		double[][] out = new double[in.size()][1];
		for(int i=0; i<in.size(); i++){
			out[i][0]=in.get(i);
		}
		return out;
	}
	
}
