package edu.vanderbilt.VUIIS.plugins;


import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.VUIIS.algorithms.CESTCalc;


public class CESTRCalculation extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume maskVol;	// 3-D Volume containing Mask Data [0 or 1]
	private ParamVolume CESTVol;	// 4-D Volume containing CEST data with a range of offsets
	private ParamFile freqOffset;	// .txt File containing Frequency Offset Values (1 per Line)
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume CESTRVol; // An Image Volume containing CEST Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "CESTR Calculation: Calculates the CESTR value for each voxel in a 4-D CEST Volume.";
	private static final String longDescription = "\nInput 1: 4-D CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies] " +
	"\n\tNote 1: Volume must contain at least one S0 volume for CEST calculation" +
	"\nInput 2: List of Offset Frequencies (.txt)" +
	"\n\tLabel the 'offset frequencies' associated with the S_0 images as 'NaN'." +
	"\nInput 3: 3-D Mask Volume (x,y,z):" +
	" [rows,columns,slices]"+
	"\n\tNote 1: The Mask Voxel should be [0 or 1] indicating whether the voxel is included or not"+
	"\n\tNote 2: If no Mask Volume is defined a mask of all 1's (all true) will be used."+
	"\nOutput 1: 4-D CESTR Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies]" +
	"\n\tWhere number of offset frequencies matches the number of S volumes in the data set (TOTAL-S0)."+
	"\nOutput 2: Execution Time [sec]; " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("CEST");
		inputParams.setLabel("CESTR Calculation");
		inputParams.setName("CESTR Calculation");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(CESTVol = new ParamVolume("CEST Volume (4D)",null,-1,-1,-1,-1));
		inputParams.add(freqOffset = new ParamFile("Offset Frequencies (.txt)",new FileExtensionFilter(new String[] {"txt"})));
		inputParams.add(maskVol = new ParamVolume("Mask Volume Map (3D)",null,-1,-1,-1,1));
		maskVol.setMandatory(false);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(CESTRVol = new ParamVolume("CESTR Data (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}


	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			this.setLabel("PARSING INPUTS");
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUT DATA");
			this.setLabel("PARSING INPUTS");
			ImageData cest = new ImageDataFloat(CESTVol.getImageData());	// Extract CEST Data from Volume
			int rcest = cest.getRows(); int ccest = cest.getCols(); int scest = cest.getSlices(); int tcest = cest.getComponents();	// Assign Variables for Dimensions of Volume
			
			ImageData mask = null; // Initialize Mask Volume
			int rmask = 0; int cmask = 0; int smask = 0;	// Initialize Mask Dimension Variables
			
			if(maskVol.getImageData() != null){	// Only give Mask Volume is not Empty
				mask = new ImageDataFloat(maskVol.getImageData());	// Extract Mask data from Volume
				rmask = mask.getRows(); cmask = mask.getCols(); smask = mask.getSlices();	// Assign Mask Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "MASK VOLUMES ADDED");
				// Check if cest and mask sizes match
				if(rcest != rmask || ccest != cmask || scest != smask){
					JistLogger.logError(JistLogger.SEVERE,"Image and Mask Volume Sizes do not match. Aborting.");
					return;
				}
				
			}

			Vector<Boolean> baseline = new Vector<Boolean>();	// Declare a Boolean Vector to Determine Baseline Values
			Vector<Double> offsetVal = new Vector<Double>();	// Declare a Double Vector to Store Read Values
			int ignoreCnt=0;	// Initialize the Counter for Ignored Values
			try{
				BufferedReader rdr = new BufferedReader(new FileReader(freqOffset.getValue()));	// Creates a FileReader from Input File
				String thisline = " "; // Initialize Line Variable
				thisline = rdr.readLine(); // Reads Next Line to Line Variable
				while(thisline!=null && !thisline.isEmpty()){	// Reads all Lines until End
					Double val = Double.valueOf(thisline);
					if(val==null)
						val = Double.NaN;
					if(val.isNaN() || val.isInfinite()) {	// Adds True Ignore Vector and Ignore Counter if NaN
						val=Double.NaN;
						baseline.add(true);
						ignoreCnt++;
					} 
					else	// Adds False to Ignore Vector (Doesn't Add to Ignore Counter)
						baseline.add(false);
					offsetVal.add(val);	// Adds value to Read Values Vector
					JistLogger.logOutput(JistLogger.CONFIG,"Shift: "+val);
					thisline = rdr.readLine(); // Reads Next Line and Repeats Loop
				}
			}
			catch(IOException e){
				JistLogger.logError(JistLogger.SEVERE, "Cannot parse input shift file");	
			}
			JistLogger.logOutput(JistLogger.CONFIG, "OFFSET FREQUENCIES ADDED");
			
			boolean []baselineArray= new boolean[baseline.size()];	// Declares Ignore Array
			double[]offsetArray= new double[baseline.size()-ignoreCnt];	// Declares Array of Frequency Offset Values
			int idx =0;
			
			for(int i=0;i<baselineArray.length;i++) {	// Creates Ignore Array from Ignore Vector
				baselineArray[i] = baseline.get(i);
				if(!baselineArray[i]) {	// Assigns Values for Offsets if not Ignored
					offsetArray[idx]=offsetVal.get(i);
					idx++;
				}
			}
			JistLogger.logOutput(JistLogger.CONFIG, "ARRAYS CREATED");
			
			CESTCalc core = new CESTCalc();	// Create variable to allows for baseline and offset volume separation
			double[][][] ref = new double[rcest][ccest][scest];	// Initialize baseline variable
			ref = core.getS0avg(cest,baselineArray);	// Calculate baseline (Average of all baselines)
			// Check if t and shiftArray.length are the same
			JistLogger.logOutput(JistLogger.CONFIG, "Number of Image Volumes: "+tcest+"("+ignoreCnt+")"+"\nNumber of Offset Frequencies found in file: "+offsetArray.length );
			if (offsetArray.length != (tcest-ignoreCnt)){
				JistLogger.logError(JistLogger.SEVERE, "The number of Image Volumes and Offset Frequencies do not match. Aborting.");
				return;
			}
				
			
			JistLogger.logOutput(JistLogger.CONFIG, "INPUTS CHECKED");
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData CESTRestimate = new ImageDataFloat(rcest,ccest,scest,tcest-ignoreCnt);	// Create ImageData variable for CEST (Only components for Offset Volumes)
			
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			this.setLabel("CEST CALC");
			this.setTotalUnits(rcest);
			JistLogger.logOutput(JistLogger.INFO, "RUNNING CORE ALGORITHMS");
			boolean maskval = true;
			for(int i=0;i<rcest;i++){	// Calculates CEST for all voxels
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){
					for(int k=0;k<scest;k++){
						if(mask != null){
							maskval = mask.getBoolean(i,j,k);
						}
						if(maskval){	// Calculate if mask volume is empty or voxel is in mask
								idx = 0;
								for(int m=0;m<tcest;m++){	// Check for all volumes
									if(!baselineArray[m]){	// Only calculated for offset volumes
										CESTRestimate.set(i,j,k,idx,cest.getDouble(i,j,k,m)/ref[i][j][k]);	// Put CEST value in ImageData
										if(Double.isNaN(cest.getDouble(i,j,k,m)/ref[i][j][k]))	// Set NaN to 0
											CESTRestimate.set(i,j,k,idx,0);
										if(Double.isInfinite(cest.getDouble(i,j,k,m)/ref[i][j][k]))	// Set Infinity to 1
											CESTRestimate.set(i,j,k,idx,1);
										if(cest.getDouble(i,j,k,m)/ref[i][j][k] > 1){
											CESTRestimate.set(i,j,k,idx,1);
										}
										if(cest.getDouble(i,j,k,m)/ref[i][j][k] < 0){
											CESTRestimate.set(i,j,k,idx,0);
										}
										idx++;
									}
								}
							}
							else{	// If not in mask set to 0
								idx = 0;
								for(int m=0;m<tcest;m++){
									if(!baselineArray[m]){	// Still only for offset volumes
										CESTRestimate.set(i,j,k,idx,0);
										idx++;
									}
								}
							}
						}
					}
				}

			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			CESTRestimate.setName(cest.getName()+"_cestr");	// Set CEST volume Name
			CESTRestimate.setHeader(cest.getHeader());	// Set CEST volume Header
			CESTRVol.setValue(CESTRestimate);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
