package edu.vanderbilt.VUIIS.plugins;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;


import edu.jhu.bme.smile.commons.math.Spline;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.VUIIS.algorithms.CESTCalc;



public class CESTShift extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume shiftVol;	// 4-D Volume containing B0 shift data for each voxel
	private ParamFile freqOffset; 	// .txt File Containing Frequency Offset Values (1 per Line)
	private ParamVolume maskVol;	// 3-D Volume containing Mask Data [0 or 1]
	private ParamVolume imageVol;	// 4-D Volume containing Image Data with a range of offsets
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume shiftedVolume;	// Volume of Shifted CEST Data Based on Shift Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Shift CEST Volume: Shifts the CEST curves to have a minimum at zero offset for all voxels.";
	private static final String longDescription = "\nInput 1: 4-D CEST Volume (x,y,z,t): " +
			" [rows,columns,slices,offset frequencies]" +
			"\nInput 2: List of Offset Frequencies (.txt)" +
			"\n\tNote 1: if the 4-D CEST Volume contains reference (S_0) images, then" +
			"\n\tlabel the 'offset frequency' associated with the S_0 images as 'NaN'." +
			"\n\tNote 2: Offset Frequencies should be one per line." +
			"\nInput 3: 3-D Mask Volume [0 or 1] (x,y,z):" +
			" [rows,columns,slices]"+
			"\n\tNote 1: The Mask Voxel should be [0 or 1] indicating whether the voxel is included or not"+
			"\n\tNote 2: If no Mask Volume is defined a mask of all 1's (all true) will be used."+
			"\nInput 4: 3-D B0 Shift Map (or Similar Shift Volume) [Hz] (x,y,z):"+
			" [rows,columns,slices]" +
			"\nOutput 1: 4-D Shifted CEST Volume (x,y,z,t):" +
			" [rows,columns,slices,offset frequencies]" +
			"\n\tWhere number of offset frequencies matches the number of S volumes in the data set (TOTAL-S0)."+
			"\nOutput 1: 4-D S0 Volume (x,y,z,t):" +
			" [rows,columns,slices,offset frequencies]" +
			"\n\tReference volumes that were extracted to perform the shifting operation" +
			"\n\tWhere number of offset frequencies matches the number of S0 volumes in the data set."+
			"\nOutput 3: Execution Time [sec]; " +
			"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("CEST");
		inputParams.setLabel("Shift CEST Volume");
		inputParams.setName("Shift CEST Volume");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt Univeristy");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);
		
		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(imageVol = new ParamVolume("CEST Volume (4D)",null,-1,-1,-1,-1));
		inputParams.add(freqOffset=new ParamFile("List Offset Frequencies",new FileExtensionFilter(new String[]{"txt"})));
		inputParams.add(shiftVol=new ParamVolume("B0 Shift Map (4D)",null,-1,-1,-1,-1));
		inputParams.add(maskVol = new ParamVolume("Mask Volume Map (3D)",null,-1,-1,-1,1));
		maskVol.setMandatory(false);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(shiftedVolume = new ParamVolume("Shift-Corrected CEST Volume",VoxelType.FLOAT,-1,-1,-1,-1));

	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}


	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			this.setLabel("PARSING INPUTS");
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUTS");
			ImageData cest=new ImageDataFloat(imageVol.getImageData());	// Extract Image Data from Volume
			int rcest=cest.getRows(), ccest=cest.getCols(), scest=cest.getSlices(), tcest = cest.getComponents();	// Assign Variables for Dimensions of Volume
			
			ImageData shift=new ImageDataFloat(shiftVol.getImageData());	// Extract Shift Data from Volume
			int rshift=shift.getRows(), cshift=shift.getCols(), sshift=shift.getSlices();	// Assign Variables for Dimensions of Volume
			// Check if cest and shift sizes match
			if(rcest != rshift || ccest != cshift || scest != sshift){
				JistLogger.logError(JistLogger.SEVERE,"Image and Shift Volume Sizes do not match. Aborting.");
				return;
			}
			
			ImageData mask = null; // Initialize Mask Volume
			int rmask = 0; int cmask = 0; int smask = 0;	// Initialize Mask Dimension Variables
			
			if(maskVol.getImageData() != null){	// Only extract mask data if Mask Volume is not Empty
				mask = new ImageDataFloat(maskVol.getImageData());	// Extract Mask data from Volume
				rmask = mask.getRows(); cmask = mask.getCols(); smask = mask.getSlices();	// Assign Mask Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "MASK VOLUMES ADDED");
				// Check if cest and mask sizes match
				if(rcest != rmask || ccest != cmask || scest != smask){
					JistLogger.logError(JistLogger.SEVERE,"Image and Mask Volume Sizes do not match. Aborting.");
					return;
				}
				// Check if shift and mask sizes match
				if(rshift != rmask || cshift != cmask || sshift != smask){
					JistLogger.logError(JistLogger.SEVERE,"Image and Mask Volume Sizes do not match. Aborting.");
					return;
				}
			}
			
			Vector<Boolean> baseline = new Vector<Boolean>();	// Declare a Boolean Vector to Determine Baseline Values
			Vector<Double> offsetVal = new Vector<Double>();	// Declare a Double Vector to Store Read Values
			int ignoreCnt=0;	// Initialize the Counter for Ignored Values
			try{
				BufferedReader rdr = new BufferedReader(new FileReader(freqOffset.getValue()));	// Creates a FileReader from Input File
				String thisline = " "; // Initialize Line Variable
				thisline = rdr.readLine(); // Reads Next Line to Line Variable
				while(thisline!=null && !thisline.isEmpty()){	// Reads all Lines until End
					Double val = Double.valueOf(thisline);
					if(val==null)
						val = Double.NaN;
					if(val.isNaN() || val.isInfinite()) {	// Adds True Ignore Vector and Ignore Counter if NaN
						val=Double.NaN;
						baseline.add(true);
						ignoreCnt++;
					} 
					else	// Adds False to Ignore Vector (Doesn't Add to Ignore Counter)
						baseline.add(false);
					offsetVal.add(val);	// Adds value to Read Values Vector
					JistLogger.logOutput(JistLogger.CONFIG,"Shift: "+val);
					thisline = rdr.readLine(); // Reads Next Line and Repeats Loop
				}
			}
			catch(IOException e){
				JistLogger.logError(JistLogger.SEVERE, "Cannot parse input shift file");	// Displays Error if Cannot Parse		
			}
			JistLogger.logOutput(JistLogger.CONFIG, "OFFSET FREQUENCIES ADDED");
			
			boolean[] baselineArray= new boolean[baseline.size()];	// Declares Ignore Array
			double[] offsetArray= new double[baseline.size()-ignoreCnt];	// Declares Array of Frequency Offset Values
			int idx =0;
			double[] splineArray = new double[offsetArray.length+2];
			for(int i=0;i<baselineArray.length;i++) {	// Creates Ignore Array from Ignore Vector
				baselineArray[i] = baseline.get(i);
				if(!baselineArray[i]) {	// Assigns Values for Offsets if not Ignored
					offsetArray[idx]=offsetVal.get(i);
					splineArray[idx+1] = offsetVal.get(i);
					idx++;
				}
			}
			splineArray[0] = offsetArray[0]*10;
			splineArray[offsetArray.length+1] = offsetArray[offsetArray.length-1]*10;
			
			JistLogger.logOutput(JistLogger.CONFIG, "ARRAYS CREATED");
			// Check if t and shiftArray.length are the same
			JistLogger.logOutput(JistLogger.CONFIG, "Number of Image Volumes: "+tcest+"("+ignoreCnt+")"+"\nNumber of Offset Frequencies found in file: "+offsetArray.length );
			if (offsetArray.length != (tcest-ignoreCnt)){
				JistLogger.logError(JistLogger.SEVERE, "The number of Image Volumes and Offset Frequencies do not match. Aborting.");
				return;
			}
					
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData shiftVol = new ImageDataFloat(rcest,ccest,scest,tcest-ignoreCnt);	// Declares Shifted Volume
			ImageData ref = new ImageDataFloat(rcest,ccest,scest,ignoreCnt); // Declares Baseline Image Volume
			ImageData totalVol = new ImageDataFloat(rcest,ccest,scest,tcest); // Declares Total Output Volume
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			this.setLabel("IMAGE SHIFT");
			JistLogger.logOutput(JistLogger.INFO, "RUNNING CORE ALGORITHMS");
			JistLogger.logOutput(JistLogger.INFO, "SHIFTING IMAGE DATA");
			this.setTotalUnits(rcest);
			boolean maskval = true;
			for(int i=0;i<rcest;i++){	// For all voxels
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){
					for(int k=0;k<scest;k++){
						if(mask != null){
							maskval = mask.getBoolean(i,j,k);
						}
						if(maskval){	// Shift data if within mask
							idx = 0;
							int index = 0;
							double[] y = new double[tcest-ignoreCnt];
							double[] spliney = new double[y.length+2];
							for(int m=0;m<tcest;m++){	// Sets y equal to the intensity for each offset
								if(!baselineArray[m]){
									y[idx] = cest.getDouble(i,j,k,m);
									spliney[idx+1] = cest.getDouble(i,j,k,m);
									idx++;
								}
								else{
									ref.set(i,j,k,index,cest.getDouble(i,j,k,m));
									index++;
								}
							}
							spliney[0] = y[0];
							spliney[y.length+1] = y[y.length-1];
							double[] shiftFreq = new double[offsetArray.length+2]; // Creates array to hold shifted frequency
							for(int m=0;m<splineArray.length;m++){	// Subtracts shift value from offset values to get shifted frequency
									shiftFreq[m] = splineArray[m] - shift.getFloat(i,j,k);
							}
							Spline func = new Spline(shiftFreq,spliney);	// Creates a spline based on shifted values (extended at each end to account for shifted values outside of original domain)
							for(int m=0;m<tcest-ignoreCnt;m++){	// Sets intensity in shifted volume according to new spline
									shiftVol.set(i,j,k,m,func.spline_value(offsetArray[m]));
							}
						}
						else{
							for(int m=0;m<tcest-ignoreCnt;m++){	// Sets intensity to 0 if outside mask
								shiftVol.set(i,j,k,m,0);
							}
							for(int m=0;m<ignoreCnt;m++){	// Sets intensity in the reference volumes to 0 if outside mask
								ref.set(i,j,k,m,0);
							}
						}							
						int idx1 = 0;
						int idx2 = 0;
						for(int m=0;m<tcest;m++){ // Recombines reference with shifted volumes in original order
							if(baselineArray[m]){	// If reference at the beginning set next reference volume
								totalVol.set(i,j,k,m,ref.getFloat(i,j,k,idx1));
								idx1++;
							}
							else{	// Else set as next shifted volume
								totalVol.set(i,j,k,m,shiftVol.getFloat(i,j,k,idx2));
								idx2++;
							}
						}
					}
				}
			}
			
			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			totalVol.setName(cest.getName()+"_shifted");	// Sets filename for shiftVol
			totalVol.setHeader(cest.getHeader());	// Sets file header for shiftVol
			shiftedVolume.setValue(totalVol);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
