package edu.vanderbilt.VUIIS.plugins;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collections;
import java.util.Vector;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.PipeAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parser.MipavPluginParser;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.jhu.ece.iacl.plugins.registration.MedicAlgorithmFLIRT;
import edu.vanderbilt.VUIIS.algorithms.GaussianFunction;
import edu.vanderbilt.VUIIS.algorithms.MSWASSRMasked;
import gov.nih.mipav.view.dialogs.ActionDiscovery;


public class FLIRTRegCEST extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume CESTvol;	// 4-D Volume containing CEST data with a range of offsets
	private ParamVolume anatVol;
	private ParamOption regType;
	
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume RegVol; // An Image Volume containing CEST Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "FLIRT Registration for CEST Volumes";
	private static final String longDescription = "\nInput 1: 4-D CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies] " +
	"\nInput 2: 3-D Anatomical Volume (x,y,z):" +
	" [rows,columns,slices]" +
	"\nOutput 1: 4-D Registered CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies]" +
	"\nOutput 2: Execution Time [sec]; " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("Registration");
		inputParams.setLabel("CEST FLIRT Registration");
		inputParams.setName("CEST FLIRT Registration");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(CESTvol = new ParamVolume("CEST Volume (4D)",null,-1,-1,-1,-1));
		inputParams.add(anatVol = new ParamVolume("Anatomical Volume (3D)",null,-1,-1,-1,1));
		anatVol.setMandatory(false);
		inputParams.add(regType = new ParamOption("Registration Type",new String[]{"All to Anatomical","First to Anatomical"}));
		
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(RegVol = new ParamVolume("Registered CEST Volume (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}
	

	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute(ProcessingAlgorithm parent) {
			this.setLabel("PARSING INPUTS");
			
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUT DATA");
			this.setLabel("PARSING INPUTS");
			
			ImageData cest = CESTvol.getImageData();	// Extract CEST Data from Volume
			int rcest = cest.getRows(); int ccest = cest.getCols(); int scest = cest.getSlices(); int tcest = cest.getComponents();	// Assign Variables for Dimensions of Volume
		
			ImageData anat = null;	// Initialize ANAT variable
			int ranat = 0; int canat = 0; int sanat = 0;	// Initialize ANAT dimension variables
			
			if(anatVol.getImageData() != null){	// Only set ANAT variable if ANAT is not empty
				anat = new ImageDataFloat(anatVol.getImageData());	// Extract ANAT Data from volume
				ranat = anat.getRows(); canat = anat.getCols(); sanat = anat.getSlices();	// Assign ANAT Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "ANAT VOLUMES ADDED");
				// Check if img and anat sizes match
				if(rcest != ranat || ccest != canat || scest != sanat){
					JistLogger.logError(JistLogger.SEVERE,"Image and anat Volume Sizes do not match. Aborting.");
					return;
				}
			}
			
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData flatData = new ImageDataFloat(rcest,ccest,scest,tcest);
			ImageData changeData = new ImageDataFloat(rcest,ccest,scest,tcest);
			ImageData regData = new ImageDataFloat(rcest,ccest,scest,tcest);
			ImageData finalData = new ImageDataFloat(rcest,ccest,scest,tcest);
			ImageData source = null;
			ImageData target = null;
			ImageData output = null;
			
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "FLATTENING IMAGE");
			
			this.setLabel("FLATTEN IMAGE");
			this.setTotalUnits(rcest);
			
			double[] anat_tot = new double[sanat];
			double[] cest_tot = new double[scest];
			double[] anat_avg = new double[sanat];
			double[] cest_avg = new double[scest];
			double[] slice_delta = new double[scest];
			int num = rcest/4;
			double avg_count = 0;
			for(int k=0;k<scest;k++){
				for(int i=num;i<3*num;i++){
					for(int j=num;j<3*num;j++){	
						anat_tot[k] = anat.getDouble(i,j,k) + anat_tot[k];
						cest_tot[k] = cest.getDouble(i,j,k,1) + cest_tot[k];
						avg_count++;
					}	
				}
				anat_avg[k] = anat_tot[k]/avg_count;
				cest_avg[k] = cest_tot[k]/avg_count;
				slice_delta[k] = anat_avg[k]/cest_avg[k];
			}
			
			
			
			for(int i=0;i<rcest;i++){
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){	
					for(int k=0;k<scest;k++){
						Vector<Double> list = new Vector<Double>();
						double[] y = new double[tcest];
						for(int m=0;m<tcest;m++){
							list.add(cest.getDouble(i,j,k,m));
							y[m] = cest.getDouble(i,j,k,m);
						}
						double median = 0;
						Collections.sort(list);
						if (list.size() % 2 == 1)
							median = list.get((list.size()+1)/2-1);
						else{
							double lower = list.get(list.size()/2-1);
							double upper = list.get(list.size()/2);
							median = (lower + upper) / 2.0;
						}
						for(int m=0;m<tcest;m++){
							double delta = 0;
							if(y[m]!=0){
								if(y[m]>1.25*median || y[m]<0.75*median){
									delta = (median/y[m])*slice_delta[k];
									flatData.set(i,j,k,m,delta*y[m]);
									changeData.set(i,j,k,m,delta);
								}
								else{
									flatData.set(i,j,k,m,y[m]);
									changeData.set(i,j,k,m,1);
								}
							}
							else{
								flatData.set(i,j,k,m,y[m]);
								changeData.set(i,j,k,m,1);
							}
						}
					}
				}
			}
			
			JistLogger.logOutput(JistLogger.INFO, "FLIRT REGISTRATION");
			this.setLabel("FLIRT CALC");
			this.setTotalUnits(tcest);
			for(int m=0;m<tcest;m++){
				JistLogger.logOutput(JistLogger.INFO, "FLIRT: " + m);
				this.setCompletedUnits(m);
				if(m==0 && anatVol.getImageData() != null){
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								regData.set(i,j,k,0,flatData.getFloat(i,j,k,0));
							}
						}
					}
				}
				else{
				target = new ImageDataFloat(rcest,ccest,scest);
				target.setHeader(cest.getHeader());
				target.setName(cest.getName()+"_"+m+"_target");
				source = new ImageDataFloat(rcest,ccest,scest);
				source.setHeader(cest.getHeader());
				source.setName(cest.getName()+"_"+m+"_source");
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								if(regType.getIndex() == 0){
									target.set(i,j,k,anat.getFloat(i,j,k));
								}
								else{
									if(m==0){
										target.set(i,j,k,anat.getFloat(i,j,k));
									}
									else{
										target.set(i,j,k,regData.getFloat(i,j,k,m-1));
									}	
								}
								source.set(i,j,k,flatData.getFloat(i,j,k,m));
							}
						}
					}
					File dir = new File(getOutputDirectory()+"/flirt_"+m);
					dir.mkdir();
					
					MedicAlgorithmFLIRT flirt = new MedicAlgorithmFLIRT();
					flirt.target.setValue(target);
					flirt.source.setValue(source);
					flirt.dof.setValue(2);
					flirt.costFunction.setValue(3);
					flirt.run();
					output = flirt.registered.getImageData();
					
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								regData.set(i,j,k,m,output.getFloat(i,j,k));
							}
						}
					}
					dir.delete();
					flirt.finalize();
					flirt.target.dispose();
					flirt.source.dispose();
					flirt.registered.dispose();
					flirt = null;
					output = null;
					source = null;
					target = null;
				}
			}
		
			JistLogger.logOutput(JistLogger.INFO, "RESTORING IMAGE");
			Double temp;
			this.setTotalUnits(rcest);
			for(int i=0;i<rcest;i++){
				this.setCompletedUnits(i);
				for(int j=0;j<ccest;j++){
					for(int k=0;k<scest;k++){
						for(int m=0;m<tcest;m++){
							temp = (double)(regData.getFloat(i,j,k,m)/changeData.getFloat(i,j,k,m));
							if(temp.isInfinite() || temp.isNaN())
								temp = 0.0;
							finalData.set(i,j,k,m,temp);
						}
					}
				}
			}

			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			finalData.setName(cest.getName()+"_flirt");	// Set CEST volume Name
			finalData.setHeader(cest.getHeader());	// Set CEST volume Header
			RegVol.setValue(finalData);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
