package edu.vanderbilt.VUIIS.plugins;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;


import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.VUIIS.algorithms.InterpMinCalc;
import edu.vanderbilt.VUIIS.algorithms.MSWASSRMasked;


public class InterpMin extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume cestVolume;	// 4-D Volume containing CEST data with a range of offsets	
	private ParamFile offsetList; 	// .txt File Containing Frequency Offset Values
	private ParamInteger interpnum;	// Number of points for interpolation
	private ParamVolume maskVol;	// Optional Mask Volume
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume shiftMapVolume;	// Volume of Shift Estimates Based on WASSR Data

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Shift Value by Interpolation";
	private static final String longDescription = ": Finds the shift in the minimum by interpolation" +
	"\nInput 1: 4-D CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies]" +
	"\nInput 2: List of Offset Frequencies" +
	"\n\tNote: if the 4-D CEST Volume contains reference (S_0) images, then" +
	"\n\tlabel the 'offset frequency' associated with the S_0 images as 'NaN'." +
	"\nInput 3: Interpolation Number" +
	"\n\tNumber of points to use for interpolation" +
	"\nInput 3: 3-D Mask Volume (x,y,z):" +
	" [rows,columns,slices]"+
	"\n\tNote 1: The Mask Voxel should be [0 or 1] indicating whether the voxel is included or not"+
	"\n\tNote 2: If no Mask Volume is defined a mask of all 1's (all true) will be used."+
	"\nOutput 1: 3-D B0 Shift Map (x,y,z): " +
	" [rows,columns,slices]" +
	"\n\tA 3-D image volume detailing shift in center frequency per voxel." +
	"\nOutput 2: Execution Time [sec] " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("CEST");
		inputParams.setLabel("Interpolation Shift Calculation");
		inputParams.setName("Interpolation Shift Calculation");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt Univeristy");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		inputParams.add(cestVolume=new ParamVolume("CEST Data (4D)",null,-1,-1,-1,-1));
		inputParams.add(offsetList=new ParamFile("List of Offset Frequencies (.txt)",new FileExtensionFilter(new String[]{"txt"})));
		inputParams.add(interpnum = new ParamInteger("Number of Interpolation Points",1000));
		inputParams.add(maskVol = new ParamVolume("Mask Volume Map (3D)",null,-1,-1,-1,1));
		maskVol.setMandatory(false);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(shiftMapVolume = new ParamVolume("Estimated Frequency Shift (Hz)",VoxelType.FLOAT,-1,-1,-1,1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}


	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			this.setLabel("PARSING INPUTS");
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUTS");
			ImageData scalarVol=new ImageDataFloat(cestVolume.getImageData());	// Extract Image Data from Volume
			int r=scalarVol.getRows(), c=scalarVol.getCols(), s=scalarVol.getSlices(), t = scalarVol.getComponents();	// Assign Variables for Dimensions of Volume
			this.setTotalUnits(s);
			
			ImageData mask = null; // Initialize Mask Volume
			int rmask = 0; int cmask = 0; int smask = 0;	// Initialize Mask Dimension Variables
			
			if(maskVol.getImageData() != null){	// Only extract mask data if Mask Volume is not Empty
				mask = new ImageDataFloat(maskVol.getImageData());	// Extract Mask data from Volume
				rmask = mask.getRows(); cmask = mask.getCols(); smask = mask.getSlices();	// Assign Mask Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "MASK VOLUMES ADDED");
				// Check if cest and mask sizes match
				if(r != rmask || c != cmask || s != smask){
					JistLogger.logError(JistLogger.SEVERE,"Image and Mask Volume Sizes do not match. Aborting.");
					return;
				}

			}
			
			Vector<Boolean> ignore = new Vector<Boolean>();	// Declare a Boolean Vector to Determine Ignored Values
			Vector<Double> offset = new Vector<Double>();	// Declare a Double Vector to Store Read Values
			int ignoreCnt=0;	// Initialize the Counter for Ignored Values
			try{
				BufferedReader rdr = new BufferedReader(new FileReader(offsetList.getValue()));	// Creates a FileReader from Input File
				String thisline = " "; // Initialize Line Variable
				thisline = rdr.readLine(); // Reads Next Line to Line Variable
				while(thisline!=null && !thisline.isEmpty()){	// Reads all Lines until End
					Double val = Double.valueOf(thisline);
					if(val==null)
						val = Double.NaN;
					if(val.isNaN() || val.isInfinite()) {	// Adds True Ignore Vector and Ignore Counter if NaN
						val=Double.NaN;
						ignore.add(true);
						ignoreCnt++;
					} 
					else	// Adds False to Ignore Vector (Doesn't Add to Ignore Counter)
						ignore.add(false);
					offset.add(val);	// Adds value to Read Values Vector
					JistLogger.logOutput(JistLogger.CONFIG,"Shift: "+val);
					thisline = rdr.readLine(); // Reads Next Line and Repeats Loop
				}
			}
			catch(IOException e){
				JistLogger.logError(JistLogger.SEVERE, "Cannot parse input shift file");	// Displays Error if Cannot Parse		
			}
			
			boolean []ignoreArray= new boolean[ignore.size()];	// Declares Ignore Array
			double[]offsetArray= new double[ignore.size()-ignoreCnt];	// Declares Array of Frequency Offset Values
			int idx =0;
			for(int i=0;i<ignoreArray.length;i++) {	// Creates Ignore Array from Ignore Vector
				ignoreArray[i] = ignore.get(i);
				if(!ignoreArray[i]) {	// Assigns Values for Offsets if not Ignored
					offsetArray[idx]=offset.get(i);
					idx++;
				}
			}
			
			// Check if t and shiftArray.length are the same
			JistLogger.logOutput(JistLogger.CONFIG, "Number of MTW volumes: "+t+"("+ignoreCnt+")"+"\nNumber of offset frequencies found in file: "+offsetArray.length );
			if (offsetArray.length != (t-ignoreCnt)){
				JistLogger.logError(JistLogger.SEVERE, "The number of MTW volumes and shift values do not match, aborting.");
				return;
			}
					
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData SHIFTestimate = new ImageDataFloat(r,c,s,1);	// Declares WASSR Shift Data Variable
			
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/	
			this.setLabel("SHIFT CALC");
			JistLogger.logOutput(JistLogger.INFO, "RUNNING CORE ALGORITHMS");
			InterpMinCalc interp = new InterpMinCalc();	// Creates Algorithm Variable
			JistLogger.logOutput(JistLogger.INFO, "CALCULATING B0 SHIFT");
			double []y= new double[offsetArray.length];	// Declares Signal Values Array
			boolean maskval = true;
			for(int k=0;k<s;k++) {
				this.setCompletedUnits(k);
				for(int i=0;i<r;i++){	// Calculates WASSR Shift for each Voxel
					for(int j=0;j<c;j++) {
						if(mask != null){
							maskval = mask.getBoolean(i,j,k);
						}
						if((maskval)){ // Only checks against threshold if mask is selected or If value is above threshold
							idx=0;
							for(int m=0;m<t;m++) {
								if(!ignoreArray[m]) {
									y[idx]=scalarVol.getDouble(i, j,k,m);	// Gets Signal Values for the Voxel
									idx++;
								}
							}
							SHIFTestimate.set(i,j,k,(float)(interp.interpmin(offsetArray,y,interpnum.getInt())));	// Calculation Algorithm
						}
						else{ // If values is below threshold and mask is selected
							SHIFTestimate.set(i,j,k,0);	// Sets WASSR Shift to 0 for Voxels outside of the mask
						}
					}
				}
			}

			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			SHIFTestimate.setName(scalarVol.getName()+"_interpmin");	// Sets filename for WASSRestimate
			SHIFTestimate.setHeader(scalarVol.getHeader());	// Sets file header for WASSRestimate
			shiftMapVolume.setValue(SHIFTestimate);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
