package edu.vanderbilt.masi.algorithms.external;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import edu.jhu.ece.iacl.jist.utility.JistLogger;

public class JSONValidate {
	
	//programInfo Variables
	String programName="";					//String which contains the program name (e.g., FSL, Solar, dcmtk etc)
	String executableName="";
	String programPath="";					//String that contains the full path to the program bin file (e.g., /usr/local/fsl5/bin/fsl, /usr/local/bin/solar)
	String functionName="";					//String that contains the function to run with ONLY the inputArgs (e.g., polygenic -p -testrhop
	String command ="";						//String that should be executed;
	String delimiter = "";
	
	//Default Jist Module Parameters
	String packageName = "";
	String pluginClassName = "";
	String shortDescription ="";
	String longDescription = "";
	String pluginPackage ="";
	String pluginCategory = "";
	String pluginLabel = "";
	String pluginName = "";
	String pluginDevelopmentStatus="ALPHA";
	String pluginEditableStatus="true";
	String pluginAlgorithmAuthor="unknown";
	String pluginAlogirthmEmail="unknown";
	String pluginAlgorithmWebsite="unknown";
	String pluginAlgorithmAuthorAffiliation="unknown";
	
	//inputArgs passed
	ArrayList<String> inputObjs;		//The arguments specified in the inputParams object
	ArrayList<String> outputObjs;		//The output arguments
	ArrayList<String> paramImports = new ArrayList<String>();
	
	//The json object to walk through
	JSONObject jsonobj;
	
	//The input and output arrays
	JSONArray inputJSONArray;
	JSONArray outputJSONArray;
	
	//Ordering of the pins

	
	//Observation information
	File[] preList;							//List of files and folders before execution as to ensure files are not incorrectly moved.
	File postRunFiles;							//List of the files and folders after execution to make sure that everything gets to the output directory
	File outputDir;								//The output directory
		
	
	//Default constructor initialize all the variables.
	public JSONValidate(File inFile){
		readJSONFileToObject(inFile);
		init();	
		System.out.println(getPluginClassName());
	}
	
	//**This section for testing
	public String getProgramName(){return this.programName;}
	public String getProgramPath(){return this.programPath;}
	
	// Return the default plugin params
	public String getPackageName(){return this.packageName;}
	public String getPluginClassName(){return setPluginClassName();}
	public String getShortDescription(){return this.shortDescription;}
	public String getLongDescription(){return this.longDescription;}
	public String getPluginPackage(){return this.pluginPackage;}
	public String getPluginCategory(){return this.pluginCategory;}
	public String getPluginLabel(){return this.pluginLabel;}
	public String getPluginDevelopmentStatus(){return this.pluginDevelopmentStatus;}
	
	public ArrayList<String> getInputObjs(){return this.inputObjs;}
	//Return the inputparamType
	public String getInputType(String obj){
		String res="";
		
		res =getInputValueByKey(obj,"type");

		return res;
	}
	
	public String getOutputType(String obj){
		String res="";
		
		res =getOutputValueByKey(obj,"type");

		return res;
	}

	
	//Initialize all the variables
	public void init(){
		setPackageName();
		setPluginClassName();
		setShortDescription();
		setLongDescription();
		setPluginPackage();
		setPluginCategory();
		setPluginLabel();
		pluginName = getPluginName();
		setPluginDevelopmentStatus();
		pluginEditableStatus = getPluginEditableStatus();
		pluginAlgorithmAuthor = getPluginAlgorithmAuthor();
		getPluginAlgorithmEmail();
		pluginAlgorithmWebsite = getPluginAlgorithmWebsite();
		pluginAlgorithmAuthorAffiliation = getPluginAlgorithmAuthorAffiliation();
		setInputObjs();
		setOutputObjs();
		getRequiredParamImports();
		
		
	}
	//Check two objects down. Only keys should be value,type,order,extension
	public String getInputValueByKey(String objectArg,String key){
		String result ="";
		try {
			result = jsonobj.getJSONObject("inputParams").getJSONObject(objectArg).getString(key);
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: unable to find specified json object "+objectArg+" in inputParams.");
			e.printStackTrace();
		}
		return result;
	}
	
	//Check two objects down. Only keys should be value,type,order,extension
		public String getOutputValueByKey(String objectArg,String key){
			String result ="";
			try {
				result = jsonobj.getJSONObject("outputParams").getJSONObject(objectArg).getString(key);
			} catch (JSONException e) {
				JistLogger.logError(JistLogger.SEVERE, "ERROR: unable to find specified json object "+objectArg+" in outParams.");
				e.printStackTrace();
			}
			return result;
		}
	
	public Boolean isRequired(String objectArg, String key){
		String res = getInputValueByKey( objectArg,  key);
		if (res.equals("true") || res.equals("True")){
			return true;
		}else
			return false;
	}
	

	
	//get an input arg value based on the key (i.e., the arg flag like -p or --print)
	public String getValueFromKey(String jsonObjString,String key) throws JSONException{
		return jsonobj.getJSONObject(jsonObjString).getString(key);
	}
	
	
	//Read a json file and return as a flat string
	public void readJSONFileToObject(File inputFile){
		//Reads a line from the text file
		BufferedReader reader;
		String output = "";
		String line = "";
		try {
			reader = new BufferedReader(new FileReader (inputFile));
			line = reader.readLine();
			while (line !=null){
				output+=line;
				
				line = reader.readLine();	
			}
			reader.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			JistLogger.logError(JistLogger.SEVERE, "JSONValidate: File "+inputFile.getName()+"not found.");
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		JSONObject myobj=null;
		try {
			myobj = new JSONObject(output);
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Could not read json file to JSONObject");
			e.printStackTrace();
		}
		this.jsonobj=myobj;
	}
	
	
	//Set the jsonobject
	public void setJSONObject(JSONObject jsonobj){
		this.jsonobj = jsonobj;
	}
	
	//Getter for the program name (base program, like dcmtk or Solar etc)
	public String setProgramName(){
		String thisProgramName="";
		try {
			thisProgramName = jsonobj.getJSONObject("programInfo").getString("name");
		} catch (JSONException e) {
			
			e.printStackTrace();
		}
		return thisProgramName;
	}
	
	//Getter for the full path to the binary file
	public String setProgramPath(){
		String thisProgramPath = "";
		try {
			thisProgramPath = jsonobj.getJSONObject("programInfo").getString("path");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the path value is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisProgramPath;
	}
	
	//***********************************Begin default module setters*****************************************************************
	public String setPackageName(){
		String thisPackageName = "";
		try {
			thisPackageName = jsonobj.getJSONObject("programInfo").getString("packageName");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the packageName is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPackageName;
	}
	
	//Attempt to set the plugin class name. Error if not in the file
	public String setPluginClassName(){
		String thisPluginClassName = "";
		try {
			thisPluginClassName = jsonobj.getJSONObject("programInfo").getString("pluginClassName");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginClassName is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginClassName;		
	}
	
	//Attempt to set the short description. Error if not in the file
	public String setShortDescription(){
		String thisShortDescription = "";
		try {
			thisShortDescription = jsonobj.getJSONObject("programInfo").getString("shortDescription");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the shortDescription is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisShortDescription;
	}
	
	//Attempt to set the long description. Error if not in the file
	public String setLongDescription(){
		String thisLongDescription = "";
		try {
			thisLongDescription = jsonobj.getJSONObject("programInfo").getString("longDescription");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the longDescription is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisLongDescription;
	}
	
	//Attempt to set the plugin package. Error if not in the file
	public String setPluginPackage(){
		String thisPluginPackage = "";
		try {
			thisPluginPackage = jsonobj.getJSONObject("programInfo").getString("pluginPackage");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginPackage is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginPackage;
	}
	
	//Attempt to set the plugin category. Error if not in the file
	public String setPluginCategory(){
		String thisPluginCategory = "";
		try {
			thisPluginCategory = jsonobj.getJSONObject("programInfo").getString("pluginCategory");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginCategory is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginCategory;
	}
	
	//Attempt to set the plugin label. Error if not in the file
	public String setPluginLabel(){
		String thisPluginLabel = "";
		try {
			thisPluginLabel = jsonobj.getJSONObject("programInfo").getString("pluginLabel");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginLabel is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginLabel;
	}
	
	//Attempt to set the plugin name. Error if not in the file
	public String getPluginName(){
		String thisPluginName = "";
		try {
			thisPluginName = jsonobj.getJSONObject("programInfo").getString("pluginName");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginName is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginName;
	}
	
	//Attempt to set the plugin development status. Error if not in file
	public String setPluginDevelopmentStatus(){
		String thisPluginDevelopmentStatus = "";
		try {
			thisPluginDevelopmentStatus = jsonobj.getJSONObject("programInfo").getString("pluginDevelopmentStatus");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginDevelopmentStatus is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginDevelopmentStatus;
	}
	
	//Attempt to set the editable status (true/false) of the plugin. Error if not in file
	public String getPluginEditableStatus(){
		String thisPluginEditableStatus = "";
		try {
			thisPluginEditableStatus = jsonobj.getJSONObject("programInfo").getString("pluginEditableStatus");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginEditableStatus is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		 
		return thisPluginEditableStatus;
	}
	
	//Attempt to set the algorithm author. Error if not in file
	public String getPluginAlgorithmAuthor(){
		String thisPluginAlgorithmAuthor = "";
		try {
			thisPluginAlgorithmAuthor = jsonobj.getJSONObject("programInfo").getString("pluginAlgorithmAuthor");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginAlgorithmAuthor is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisPluginAlgorithmAuthor;

	}
	
	//Attempt to parse the algorithm authors email. Error if not in file
	public String getPluginAlgorithmEmail(){
		String thisPluginAlogirthmEmail = "";
		try {
			thisPluginAlogirthmEmail = jsonobj.getJSONObject("programInfo").getString("pluginAlogirthmEmail");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginAlogirthmEmail is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}

			return thisPluginAlogirthmEmail;
		}
	
	//Attempt to parse the plugin autor algorithm website. Error if not in the file.
	public String getPluginAlgorithmWebsite(){
		String thisPluginAlgorithmWebsite = "";
		try {
			thisPluginAlgorithmWebsite = jsonobj.getJSONObject("programInfo").getString("pluginAlgorithmWebsite");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginAlgorithmWebsite is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		
		return thisPluginAlgorithmWebsite;
		
	}
	
	//Attempt to parse the plugin author algorithm affiliation. Error if not in the file.
		public String getPluginAlgorithmAuthorAffiliation(){
			String thisPluginAlgorithmAuthorAffiliation="";
			try {
				thisPluginAlgorithmAuthorAffiliation = jsonobj.getJSONObject("programInfo").getString("pluginAlgorithmAuthorAffiliation");
			} catch (JSONException e) {
				JistLogger.logError(JistLogger.SEVERE, "ERROR: the pluginAlgorithmAuthorAffiliation is not in the programInfo object. Please check input file.");
				e.printStackTrace();
			}
			

			return thisPluginAlgorithmAuthorAffiliation;

		}
	//***********************************END default module setters*****************************************************************
	
	public String getExecutable(){
		String thisName = "";
		try {
			thisName = jsonobj.getJSONObject("programInfo").getString("executableName");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: the path value is not in the programInfo object. Please check input file.");
			e.printStackTrace();
		}
		return thisName;
	}
	
	//Getter for the function call with inputArgs. 
	public String getFunctionName(){
		String thisFunctionName = "";
		try {
			thisFunctionName = jsonobj.getJSONObject("programInfo").getString("function");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Unable to locate the function name in the programInfo object");
			e.printStackTrace();
		}
		return thisFunctionName;
	}
	

	
	//Get the input paramaters and set to a json array. Validate the required args for each
	public JSONArray setInputObjs(){
		JSONArray inputs = null;
		try {
			inputs = jsonobj.getJSONArray("inputParams");
			JistLogger.logOutput(JistLogger.FINE, "Found "+String.valueOf(inputs.length())+ " input paramater(s).");
			for (int i=0;i<inputs.length();i++){
				JSONObject thisObject = inputs.getJSONObject(i);
				if (hasRequired(thisObject)){
					JistLogger.logOutput(JistLogger.FINE,"	Validated object "+String.valueOf(i+1)+" of "+String.valueOf(inputs.length()));
					
					//Add the type to the required imports only if it isn't alreay there
					if (!paramImports.contains(thisObject.getString("type"))){
						paramImports.add(thisObject.getString("type"));
					}
				}else{
					throw new RuntimeException("FATAL Error. inputParam "+String.valueOf(i)+" is missing required key. Please verify your input file");
				}	
			}
		} catch (JSONException e) {
			// TODO Auto-generated catch block
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Unable to locate the inputParam array in the JSON file");
			e.printStackTrace();
		}
		this.inputJSONArray = inputs;
		return inputs;
			
	}
	
	//Init the output objects and set to a json array. Validate the required args for each
	public JSONArray setOutputObjs(){
		JSONArray outputs = null;
		try {
			outputs = jsonobj.getJSONArray("outputParams");
			JistLogger.logOutput(JistLogger.FINE, "Found "+String.valueOf(outputs.length())+ " output paramater(s).");
			for (int i=0;i<outputs.length();i++){
				JSONObject thisObject = outputs.getJSONObject(i);
				if (hasRequired(thisObject)){
					JistLogger.logOutput(JistLogger.FINE,"	Validated object "+String.valueOf(i+1)+" of "+String.valueOf(outputs.length()));
					
					//Add the type to the required imports only if it isn't alreay there
					if (!paramImports.contains(thisObject.getString("type"))){
						paramImports.add(thisObject.getString("type"));
					}
				}else{
					throw new RuntimeException("FATAL Error. outputParam "+String.valueOf(i)+" is missing required key. Please verify your input file");
				}	
			}
		} catch (JSONException e) {
			// TODO Auto-generated catch block
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Unable to locate the outputParam array in the JSON file");
			e.printStackTrace();
		}
		this.outputJSONArray = outputs;
		return outputs;
	}
	
	//Check to see that each JSON Object inside the inputparams or output params array has at minimum the required keys.
	public Boolean hasRequired(JSONObject obj){
		Boolean result =true;
		try {
			obj.getString("arg");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "Error: inputParam is missing required key arg");
			e.printStackTrace();
			result = false;
		}
		
		try {
			obj.getString("required");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "Error: inputParam is missing required key required");
			e.printStackTrace();
			result = false;
		}
		
		try {
			obj.getString("name");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "Error: inputParam is missing required key name");
			e.printStackTrace();
			result = false;
		}
		
		try {
			obj.getString("type");
		} catch (JSONException e) {
			JistLogger.logError(JistLogger.SEVERE, "Error: inputParam is missing required key type");
			e.printStackTrace();
			result = false;
		}
		return result;
	}
	
	
	//The current implemented JIST Params
	public enum ParamType{
		ParamFloat,ParamInteger,ParamLong,ParamDouble,ParamString,ParamFile,ParamVolume,ParamBoolean
	}
	
	//The file types
	public enum DialogType{
		FILE,DIRECTORTY
	}
	
	//Check based on the input type, if there are optional args (like max and min value for a float).
	public Boolean hasOptional(ParamType type, JSONObject obj){
		
		switch(type){
			case ParamFloat: {
				
				Float min=null;	//min value
				Float max=null;	//max value
				Float def=null;	//default value
				
				try {
					min = (float) obj.getDouble("min");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: min value undefined for "+type);
					e.printStackTrace();
				}
				
				try {
					max = (float) obj.getDouble("max");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: max value undefined for "+type);
					e.printStackTrace();
				}
				
				try {
					def = (float) obj.getDouble("default");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: default value undefined for "+type);
					e.printStackTrace();
				}
			}
				
			case ParamInteger: {
				int min=(Integer) null;	//min value
				int max=(Integer) null;	//max value
				int def=(Integer) null;	//default value
				
				try {
					min = (int) obj.getInt("min");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: min value undefined for "+type);
				}
				
				try {
					max = (int) obj.getInt("max");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: max value undefined for "+type);
				}
				
				try {
					def = (int) obj.getInt("default");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: default value undefined for "+type);
				}
			}
			case ParamLong: {
				Long min = null;
				Long max = null;
				Long def = null;
				
				try {
					min = obj.getLong("min");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: min value undefined for "+type);
				}
				
				try {
					max = obj.getLong("max");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: max value undefined for "+type);
				}
				
				try {
					def = obj.getLong("default");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: default value undefined for "+type);
				}
				
			}
			case ParamDouble: {
				double min = (Double) null;
				double max = (Double) null;
				double def = (Double) null;
				
				try {
					min = obj.getDouble("min");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: min value undefined for "+type);
				}
				
				try {
					max = obj.getDouble("max");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: max value undefined for "+type);
				}
				
				try {
					def = obj.getDouble("default");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: default value undefined for "+type);
				}
				
			}
			case ParamString: {
				//Reserved. Currently nothing to check since a name and a string are required.
			}
			case ParamFile: {
				String extensions =null;
				try {
					extensions = obj.getString("extension");
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: extension value undefined for "+type);
				}
				
				URI uri=null;
				
				try {
					uri = URI.create(obj.getString("uri"));
				} catch (JSONException e) {
					JistLogger.logOutput(JistLogger.FINE, "WARNING: extension value undefined for "+type);
				}
			}
			case ParamVolume: {
				
			}
			case ParamBoolean: {
				//Reserved. Nothing to do here since a name is required. 
			}
			default:
				JistLogger.logError(JistLogger.SEVERE, "Error: input param "+type+" is not currently implemented.");
		}

				
		return false;
	}

	//Execute the command
	public int execute(){
		Process p;
		int exitStatus =0;
		try {
			p = Runtime.getRuntime().exec(command);
			p.waitFor();
			exitStatus = p.exitValue();
			InputStream stderr = p.getErrorStream();
			OutputStream stdout = p.getOutputStream();
			
		} catch (IOException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Unable to run external command as built:\n"+command);
			e.printStackTrace();
		} catch (InterruptedException e) {
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Unable to run external command as built:\n"+command);
			e.printStackTrace();
		}
		return exitStatus;
	}
	
	//Based on the reuqired inputs, make sure that we import everything that we need. 
	public ArrayList<String> getRequiredParamImports(){
		ArrayList<String> req = new ArrayList<String>();
		ParamType type = null;
		for(String st :paramImports){
			ParamType theType = type.valueOf(st); //Check to make sure that this is a valid type in the enum
			req.add("import edu.jhu.ece.iacl.jist.pipeline.parameter."+st+";");
		}
		
		return req;
	}
	
}
