package edu.vanderbilt.masi.plugins.classification;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.json.JSONException;
import org.json.JSONObject;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.plugins.classification.FeatureCalculatorPlugin.ExecuteWrapper;

public class AddTrueLabelPlugin extends ProcessingAlgorithm {

	// input parameters
	public ParamVolumeCollection true_vol;
	public ParamVolumeCollection est_vol;
	public ParamFileCollection feature_file;

	// output parameters
	public ParamFileCollection feature_file_output;

	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Add True Labels to Feature File.";
	private static final String longDescription = "";
	
	public ImageData labels;
	public ImageData est_labels;

	@Override
	protected void createInputParameters(ParamCollection arg0) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		inputParams.setPackage("MASI");
		inputParams.setCategory("Feature Calculation");
		inputParams.setLabel("True Label Added");
		inputParams.setName("True_Label_Added");

		inputParams.add(true_vol = new ParamVolumeCollection("True Labels"));
		true_vol.setLoadAndSaveOnValidate(false);
		true_vol.setMandatory(true);

		inputParams.add(est_vol = new ParamVolumeCollection("Estimated Labels"));
		est_vol.setLoadAndSaveOnValidate(false);
		est_vol.setMandatory(false);
		
		inputParams.add(feature_file = new ParamFileCollection("Calculated Features"));
		feature_file.setLoadAndSaveOnValidate(false);
		feature_file.setMandatory(true);
		

	}

	@Override
	protected void createOutputParameters(ParamCollection arg0) {
		outputParams.add(feature_file_output = new ParamFileCollection("Output Features"));
	}

	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		ExecuteWrapper wrapper=new ExecuteWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);


	}

	protected class ExecuteWrapper extends AbstractCalculation {
		protected void execute(ProcessingAlgorithm alg)
				throws AlgorithmRuntimeException {
			File outdir = new File(
					alg.getOutputDirectory() +
					File.separator +
					FileUtil.forceSafeFilename(alg.getAlgorithmName()) +
					File.separator);
			outdir.mkdirs();
			for(int i =0; i< true_vol.size();i++){
				labels = true_vol.getParamVolume(i).getImageData();
				est_labels = est_vol.getParamVolume(i).getImageData();
				File f = feature_file.getValue(i);
				String name = f.getName();
				File out = new File(outdir, name);
				feature_file_output.add(out);
				JistLogger.logOutput(JistLogger.INFO,"On file "+name);
				FileOutputStream fstream;
				try {
					fstream = new FileOutputStream(out);
					OutputStreamWriter ow;
					if(out.getAbsolutePath().endsWith("gz"))	ow = new OutputStreamWriter(new GZIPOutputStream(fstream),"utf-8");
					else  ow = new OutputStreamWriter(fstream);
					InputStreamReader ir;
					if(f.getAbsolutePath().endsWith("gz")) ir = new InputStreamReader(new GZIPInputStream(new FileInputStream(f)));
					else ir = new InputStreamReader(new FileInputStream(f));
					BufferedWriter bw = new BufferedWriter(ow);
					BufferedReader br = new BufferedReader(ir);
					String line;
					line = br.readLine();
					bw.write(line);
					bw.write("\n");
					processFiles(br,bw);
					br.close();
					bw.close();
					ir.close();
					ow.close();
					fstream.close();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}

		}
		private void processFiles(BufferedReader br, BufferedWriter bw) throws IOException{
			String line;
			while((line=br.readLine())!=null){
				try {
					JSONObject obj = new JSONObject(line);
					int x = obj.getInt("x");
					int y = obj.getInt("y");
					int z = obj.getInt("z");
					int label = labels.getInt(x, y, z);
					int est_label = est_labels.getInt(x,y,z);
					obj.put("true_label", label);
					obj.put("label",est_label);
					bw.write(obj.toString());
					bw.write("\n");
				} catch (JSONException e) {
					e.printStackTrace();
				}
			}
		}
	}
}
