package edu.vanderbilt.masi.plugins.hardi;

import Jama.EigenvalueDecomposition;
import Jama.Matrix;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.io.ModelImageReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

public class ComputeGeneralizedFA extends ProcessingAlgorithm {
	/****************************************************
	 * Input Parameters
	 ****************************************************/
	private ParamVolume sphericalHarmonicCoeffsReal;	// A 4D volume with the SH real coefficients
	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamVolume gfaVolume;	// 3-D VOlume with GFA values
	
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Generalized Fractional Anisotropy from Qball estimation";
	private static final String longDescription = "Computes the GFA from the real data fromt the Qball estimation step. GFA = std(psi)/rms(psi)";
	
	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Step 1. Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("MASI");
		inputParams.setCategory("Modeling.Diffusion.QBall");
		inputParams.setLabel("Q-Ball: GFA");
		inputParams.setName("Q-Ball:_GFA");


		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.nitrc.org/projects/jist/");
		info.add(new AlgorithmAuthor("Stephen Damon", "stephen.m.damon@vanderbilt.edu", "masi.vuse.vanderbilt.edu"));
		info.setAffiliation("Vanderbilt University, Department of Electrical Engineering");
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);


		/****************************************************
		 * Step 2. Add input parameters to control system
		 ****************************************************/
		inputParams.add(sphericalHarmonicCoeffsReal=new ParamVolume("Spherical Harmonic Coefficients: Real (4-D)",null,-1,-1,-1,6));
		
	}

	
	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system  
		 ****************************************************/
		// Outputs
		gfaVolume = new ParamVolume("GFA",VoxelType.FLOAT,-1,-1,-1,1);
		gfaVolume.setName("GFA");
		outputParams.add(gfaVolume);

		
	}


	protected void execute(CalculationMonitor arg0)
			throws AlgorithmRuntimeException {
		/****************************************************
		 * Step 1. Indicate that the plugin has started.
		 * 		 	Tip: Use limited System.out.println statements
		 * 			to allow end users to monitor the status of
		 * 			your program and report potential problems/bugs
		 * 			along with information that will allow you to 
		 * 			know when the bug happened.  
		 ****************************************************/
		System.out.println(getClass().getCanonicalName()+"\t"+"ComputeGeneralized: Start");
		
		/****************************************************
		 * Step 2. Parse the input data 
		 ****************************************************/
		ImageData shReal=sphericalHarmonicCoeffsReal.getImageData();
		ImageDataFloat vol=new ImageDataFloat(shReal);
		float[][][][] shCoeffs = vol.toArray4d();
		
		
		int r=vol.getRows(), c=vol.getCols(), s=vol.getSlices(), t = vol.getComponents();
		
		/****************************************************
		 * Step 3. Setup memory for the computed volumes
		 ****************************************************/
		System.out.println(getClass().getCanonicalName()+"\t"+"ComputeGeneralizedFA: Allocating memory.");
		float [][][]gfa = new float[r][c][s];
		float num=0;
		float denom=0;
		
		
		/****************************************************
		 * Step 4. Run the core algorithm. Note that this program 
		 * 		   has NO knowledge of the MIPAV data structure and 
		 * 		   uses NO MIPAV specific components. This dramatic 
		 * 		   separation is a bit inefficient, but it dramatically 
		 * 		   lower the barriers to code re-use in other applications.  		  
		 ****************************************************/
		
		//Compute the GFA
		for (int i=0; i<r;i++){
			for (int j=0; j<c; j++){
				for (int k=0;k<s;k++){
					float shSum=0;
					for (int l=0; l<t;l++){
						shSum+=Math.pow(shCoeffs[i][j][k][l],2);
					}
					
					gfa[i][j][k]= (float) Math.sqrt(1-(Math.pow(shCoeffs[i][j][k][0],2)/shSum));
					num=0;
					denom=0;
					shSum = 0;
				}
			}
		}

		/****************************************************
		 * Step 5. Retrieve the image data and put it into a new
		 * 			data structure. Be sure to update the file information
		 * 			so that the resulting image has the correct
		 * 		 	field of view, resolution, etc.  
		 ****************************************************/
		System.out.println(getClass().getCanonicalName()+"\t"+"DWITensorContrasts: Setting up exports.");
		int []ext=shReal.getModelImageCopy().getExtents();
		ImageData out;
		
		//GFA OUT
		out = (new ImageDataFloat(gfa));
		out.setHeader(shReal.getHeader());

		out.setName(shReal.getName()+"_GFA");
		gfaVolume.setValue(out);
		gfa=null;

		
	}

}
