package edu.vanderbilt.masi.plugins.labelfusion;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.vanderbilt.masi.algorithms.labelfusion.MultiClassSTAPLE;

public class MultiClassSTAPLEPlugin extends ProcessingAlgorithm {
	
	private static final String cvsversion = "$Revision: 1.5 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Runs the STAPLE Label Fusion Algorithm for multiple labels.";
	private static final String longDescription = "";
	
	// Input Parameters
	public ParamVolumeCollection obsVals;
	public ParamFile raterMap;
	public ParamFile initialTheta;
	public ParamInteger maxIter;
	public ParamFloat consensusThresh;
	public ParamFloat epsilon;
	public ParamFloat nudge;
	public ParamBoolean determineConsensus;
	
	// Output Parameters
	public ParamVolumeCollection labels;
	public ParamVolume labelProbabilities;
	public ParamVolumeCollection theta;

	@Override
	protected void createInputParameters(ParamCollection arg0) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(true);
		info.setStatus(DevelopmentStatus.BETA);
		
		inputParams.setPackage("MASI");
		inputParams.setCategory("Label Fusion");
		inputParams.setName("MC_STAPLE");
		inputParams.setLabel("Multi-Class STAPLE");
		
		inputParams.add(obsVals = new ParamVolumeCollection("Observed Rater Volumes"));
		obsVals.setMandatory(true);
		
		inputParams.add(raterMap = new ParamFile("Rater Map File"));
		raterMap.setMandatory(true);
		
		inputParams.add(initialTheta = new ParamFile("Initial Thetas File"));
		initialTheta.setMandatory(true);
		
		inputParams.add(maxIter = new ParamInteger("Maximum number of iterations"));
		maxIter.setValue(100);
		maxIter.setMandatory(false);
		
		inputParams.add(consensusThresh = new ParamFloat("Voxel Consensus Threshold"));
		consensusThresh.setValue(0.99f);
		consensusThresh.setMandatory(false);
		
		inputParams.add(epsilon = new ParamFloat("Convergence Threshold"));
		epsilon.setValue(1e-4f);
		epsilon.setMandatory(false);
		
		inputParams.add(nudge = new ParamFloat("Nudge Value for Avoiding Label Inversion"));
		nudge.setValue(10f);
		nudge.setMandatory(false);
		
		inputParams.add(determineConsensus = new ParamBoolean("Determine and skip consensus voxels"));
		determineConsensus.setMandatory(false);
		determineConsensus.setValue(false);
		
	}

	@Override
	protected void createOutputParameters(ParamCollection arg0) {
		outputParams.add(labels = new ParamVolumeCollection("Segmentations"));
		outputParams.add(labelProbabilities = new ParamVolume("Label Probabilities"));
		outputParams.add(theta = new ParamVolumeCollection("Performance Parameter Estimates"));

	}

	@Override
	protected void execute(CalculationMonitor arg0)
			throws AlgorithmRuntimeException {
		MultiClassSTAPLE MCS = new MultiClassSTAPLE(obsVals,raterMap,initialTheta,maxIter,consensusThresh,epsilon,nudge,determineConsensus.getValue());
		for(ImageData im: MCS.getLabelSet()){
			labels.add(im);
		}
		labelProbabilities.setValue(MCS.getProbabilityVolume());
		theta.setValue(MCS.getThetaVolume());

	}

}
