package edu.vanderbilt.masi.plugins.labelfusion;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.labelfusion.HierarchyMapper;

public class PluginHierarchyMapper extends ProcessingAlgorithm {

	private static final String cvsversion = "$Revision: 1.7 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Maps an input label volume to a particular level of the hierarchy";
	private static final String longDescription = "";
	
	//Input Parameters
	ParamVolumeCollection inputLabels;
	ParamFileCollection hierarchy;
	ParamInteger level;
	ParamFile levels;
	ParamFile raterMap;
	
	
	//Output Parameters
	ParamVolumeCollection outputLabels;
	
	@Override
	protected void createInputParameters(ParamCollection arg0) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(true);
		info.setStatus(DevelopmentStatus.ALPHA);
		
		inputParams.setPackage("MASI");
		inputParams.setCategory("Hierarchy_Tools");
		inputParams.setName("Hierarchy_Mapper");
		inputParams.setLabel("Hierarchy Mapper");
		
		inputParams.add(inputLabels = new ParamVolumeCollection("Input Atlas Labels"));
		inputLabels.setMandatory(true);
		
		inputParams.add(hierarchy = new ParamFileCollection("Hierarchy Text File(s)"));
		hierarchy.setMandatory(true);

		inputParams.add(level = new ParamInteger("Hierarchy Level (if only using one hierarchy)"));
		level.setMandatory(false);
		level.setValue(-1);
		
		inputParams.add(levels = new ParamFile("Map of levels to map hierarchies to (only necessary with multiple hierachies)"));
		levels.setMandatory(false);
		
		inputParams.add(raterMap = new ParamFile("Map of raters to hierarchies (only necessary with multiple hierarchies, one indexed)"));
		raterMap.setMandatory(false);
		
		
	}

	@Override
	protected void createOutputParameters(ParamCollection arg0) {
		
		outputParams.add(outputLabels = new ParamVolumeCollection("Re-mapped Labels"));

	}

	@Override
	protected void execute(CalculationMonitor arg0)
			throws AlgorithmRuntimeException {
		
		JistPreferences prefs = JistPreferences.getPreferences();
		
		JistLogger.logOutput(JistLogger.WARNING, "Level: " +  level.getValue());
		JistLogger.logOutput(JistLogger.WARNING, "Levels: " + levels.getValue());
		JistLogger.logOutput(JistLogger.WARNING, "Rater Map: "+raterMap.getValue());
		JistLogger.logOutput(JistLogger.WARNING, "Debug Level: "+prefs.getDebugLevel());
		
		int levelInt = level.getInt();
		
		boolean usingOne = levelInt > 0;
		boolean usingMultiple = levels.getValue() != null;
		
		if(usingOne && usingMultiple){
			JistLogger.logError(JistLogger.WARNING,"Error both one and multiple hierarchies set.\nPunting.");
			return;
		}else if(!usingOne && !usingMultiple){
			JistLogger.logError(JistLogger.WARNING,"Error neither one nor multiple are hierarchies set.\nPunting.");
			return;
		}
		
		HierarchyMapper HM;
		
		int[] raterInds = new int[inputLabels.getParamVolumeList().size()];
		Arrays.fill(raterInds,0);
		
		if(usingOne)
			HM = new HierarchyMapper(hierarchy.getValue().get(0),level.getInt());
		else{
			HM = new HierarchyMapper(hierarchy.getValue(),levels.getValue());
			if(raterMap.getValue()!=null)
				raterInds = readFile(raterMap.getValue());
			else
				JistLogger.logOutput(JistLogger.WARNING, "No rater map set. Assuming all are class 1.");
		}
		
		
		if(!HM.isInitialized()){
			JistLogger.logError(JistLogger.SEVERE, "Hierarchy Mapper was not initizlized. Exitting");
			return;
		}
		
		ParamVolume pv;
		
		for(int i=0;i < raterInds.length; i++){
			pv = inputLabels.getParamVolumeList().get(i);
			JistLogger.logOutput(JistLogger.WARNING,"Mapping volume "+i);
			outputLabels.add(HM.map(pv.getImageData(),raterInds[i]));
		}
		
	}
	
	public int[] readFile(File f){
		BufferedReader br;
		List<Integer> t = null;
		try {
			br = new BufferedReader(new FileReader(f));
			String line;
			
			t = new ArrayList<Integer>();
			
			while((line=br.readLine())!=null){
				t.add(Integer.parseInt(line)-1);
			}
			
			br.close();
			
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		int[] o = new int[t.size()];
		for(int i=0;i<o.length;i++)
			o[i] = t.get(i);
		
		return o;
	}

}
