package edu.vanderbilt.masi.plugins.regression;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.regression.DataCollection;
import edu.vanderbilt.masi.algorithms.regression.Forest;
import edu.vanderbilt.masi.algorithms.regression.Regression;
import edu.vanderbilt.masi.algorithms.regression.XmlIO;

public class PluginRegressionTester extends ProcessingAlgorithm {

	/**
	 * Input Parameters
	 */
	private ParamString forestFileName;
	private ParamVolume testingVolume;
	private ParamInteger boxCount;

	/**
	 * Output Parameters
	 */
	private ParamFile resultFile;

	/**
	 * Declare Plugin Information Variables
	 */
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "")
			.replace("$", "").replace(" ", "");
	private static final String shortDescription = "Regression Forest Testing.";
	private static final String longDescription = "";

	@Override
	protected void createInputParameters(ParamCollection inputParams) {

		// Set plugin information
		inputParams.setPackage("MASI");
		inputParams.setCategory("Regression");
		inputParams.setLabel("Regression Forest Testing");
		inputParams.setName("Regression Forest Tester");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Bo Li", "bo.li.2@vanderbilt.edu", ""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		// Set input parameters to control the system
		// TODO: bind the input parameters
		inputParams.add(forestFileName = new ParamString("forestFileName", ""));
		inputParams.add(testingVolume = new  ParamVolume("Volume"));
		inputParams.add(boxCount = new ParamInteger("boxCount", 10));
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(resultFile = new ParamFile("Boxes",
				new FileExtensionFilter(new String[] { "txt" })));
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		AlgorithmWrapper wrapper = new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}

	protected class AlgorithmWrapper extends AbstractCalculation {

		public void execute(ProcessingAlgorithm parent) {
			this.setLabel("PARSING INPUTS");

			// Indicate that the Plugin has started.
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");

			// main process
			int[][] boxes = null;
			try {
				 boxes = runTesting();
			} catch (InterruptedException e) {
				System.err.println(getClass().getCanonicalName()
						+ e.getMessage());
				return;
			}

			// save as file
			String f = getOutputDirectory().getAbsoluteFile().toString();
			File file = new File(f + File.separator + resultFile.getName() + ".txt");

			try {
				PrintWriter out = new PrintWriter(file);
				for (int i = 0; i < boxes.length; ++i) {
					out.println("Box" + i + ": ");
					for (int j = 0; j < 6; ++j) {
						out.println(boxes[i][j] + " ");
					}
					out.println();
				}
				out.close();
			} catch (IOException e) {
				System.err.println(getClass().getCanonicalName()
						+ e.getMessage());
				return;
			}

			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			resultFile.setValue(file);
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
		
		private int[][] runTesting() throws InterruptedException {
			XmlIO xmlIO = new XmlIO();
			Forest forest = xmlIO.LoadDocument(forestFileName.getValue());
			
			ImageData image = testingVolume.getImageData();
			int[][][] matrix = new int[testingVolume.getRows()][testingVolume
					.getCols()][testingVolume.getSlices()];
			for (int i = 0; i < testingVolume.getRows(); ++i) {
				for (int j = 0; j < testingVolume.getCols(); ++j) {
					for (int k = 0; k < testingVolume.getSlices(); ++k) {
						matrix[i][j][k] = image.getInt(i, j, k);
					}
				}
			}
			
			DataCollection dc = new DataCollection(0);
			dc.LoadTestData(matrix);
			
			int[][] boxes = new int[boxCount.getInt()][6];

			Regression regression = new Regression();
			double[] box = regression.RunTesting(dc, forest, boxCount.getInt());
			
			for (int i = 0; i < boxCount.getInt(); ++i) {
				for (int j = 0; j < 6; ++j) {
					boxes[i][j] = (int)box[i * 6 + j];
				}
			}
							
			return boxes;
		}
	}
}
