package edu.vanderbilt.masi.plugins.regression;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.regression.DataCollection;
import edu.vanderbilt.masi.algorithms.regression.Forest;
import edu.vanderbilt.masi.algorithms.regression.Regression;
import edu.vanderbilt.masi.algorithms.regression.XmlIO;

public class PluginRegressionTrainer extends ProcessingAlgorithm {
	/**
	 * Input Parameters
	 */
	private ParamInteger numberOfTree;
	private ParamInteger numberOfBox;
	private ParamVolumeCollection vol;
	private ParamString boxesFile;
	
	/**
	 * Output Parameters
	 */
	private ParamFile forestFile; 
	
	/**
	 * Declare Plugin Information Variables
	 */
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "")
			.replace("$", "").replace(" ", "");
	private static final String shortDescription = "Regression Forest Training.";
	private static final String longDescription = "";
	
	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		// Set plugin information
		inputParams.setPackage("MASI");
		inputParams.setCategory("Regression");
		inputParams.setLabel("Regression Forest Training");
		inputParams.setName("Regression Forest Trainer");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Bo Li", "bo.li.2@vanderbilt.edu", ""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);
		
		// Set input parameters to control the system
		// TODO: bind the input parameters
		inputParams.add(numberOfTree = new ParamInteger("numberOfTree", 10));
		inputParams.add(numberOfBox = new ParamInteger("numberOfBox", 10));
		inputParams.add(vol = new ParamVolumeCollection("Volumes"));
		inputParams.add(boxesFile = new ParamString("boxesFileName", ""));
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(forestFile = new ParamFile("Boxes",
				new FileExtensionFilter(new String[] { "txt" })));
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		AlgorithmWrapper wrapper = new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}
	
	protected class AlgorithmWrapper extends AbstractCalculation {

		public void execute(ProcessingAlgorithm parent) {
			this.setLabel("PARSING INPUTS");

			// Indicate that the Plugin has started.
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");

			// main process
			File file = null;
			try {
				file = runTraining();
			} catch (FileNotFoundException e) {
				System.err.println(getClass().getCanonicalName()
						+ e.getMessage());
				return;
			}

			// save as file
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			forestFile.setValue(file);
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
		
		private File runTraining() throws FileNotFoundException{
			List<int[][][]> matrixs = new ArrayList<int[][][]>();
			List<int[][]> boxesList = new ArrayList<int[][]>();
			
			for (ParamVolume item : vol.getParamVolumeList()) {
				matrixs.add(parseMatrix(item));
			}
			
			boxesList = parseBoxesList(boxesFile.getValue());
			
			DataCollection dc = new DataCollection(numberOfBox.getInt());
			for (int i = 0; i < matrixs.size(); ++i) {
				dc.AddDataItem(matrixs.get(i), boxesList.get(i));
			}
			
			Regression regression = new Regression();
			Forest forest = regression.RunTraining(dc, numberOfTree.getInt());
			
			String f = getOutputDirectory().getAbsoluteFile().toString();
			XmlIO xmlIO = new XmlIO();			
			File file = xmlIO.SaveDocuemnt(forest, f + File.separator + forestFile.getName() + ".xml");
			
			return file;
		}
		
		private int[][][] parseMatrix(ParamVolume paramVolume) {
			ImageData image = paramVolume.getImageData();
			int[][][] matrix = new int[paramVolume.getRows()][paramVolume
					.getCols()][paramVolume.getSlices()];
			for (int i = 0; i < paramVolume.getRows(); ++i) {
				for (int j = 0; j < paramVolume.getCols(); ++j) {
					for (int k = 0; k < paramVolume.getSlices(); ++k) {
						matrix[i][j][k] = image.getInt(i, j, k);
					}
				}
			}
			return matrix;
		}
		
		private List<int[][]> parseBoxesList(String filename) throws FileNotFoundException {
			List<int[][]> boxesList = new ArrayList<int[][]>();
			
			@SuppressWarnings("resource")
			Scanner in = new Scanner(new File(filename));
			while (in.hasNext()) {
				in.nextLine();
				int count = in.nextInt();
				
				int[][] boxes = new int[count][6];
				for (int i = 0; i < count; ++i) {
					for (int j = 0; j < 6; ++j) {
						boxes[i][j] = in.nextInt();
					}
				}
				
				boxesList.add(boxes);
			}
			
			return boxesList;
		}
	}

}
