package edu.vanderbilt.masi.plugins.solar;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;
import java.io.InputStreamReader;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ExecutionContext;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.solar.Polygenic;
import edu.vanderbilt.masi.algorithms.solar.Solar;


public class PluginRunSolarPolygenic extends ProcessingAlgorithm {
	ParamFileCollection pedigree;
	//ParamFileCollection solarPath;
	ParamFileCollection phenotypes;
	ParamFileCollection trait;
	ParamFileCollection covariates;
	ParamVolume imageData;
	
	
	//Non required options
	ParamBoolean screen;
	ParamFloat prob;
	ParamFileCollection fix;
	ParamBoolean all;
	ParamFileCollection testcovar;
	ParamBoolean testrhoe;
	ParamBoolean testrhog;
	ParamBoolean testrhoc;
	ParamBoolean testrhop;
	ParamBoolean rhopse;
	ParamBoolean sporadic;
	ParamBoolean keephouse;
	ParamBoolean house;

	//Write the output solar command
	ParamFile lastMod;
	ParamFile nocovarMod;
	ParamFile nocovarOut;
	ParamFile null0Mod;
	ParamFile polygenicLogsOut;
	ParamFile polygenicOut;
	ParamFile polygenicResid;
	ParamFile polygenicResidStats;
	ParamFile polyMod;
	ParamFile polyOut;
	ParamFile sporMod;
	ParamFile sporOut;
	
	
	
	ParamFile stderr;
	ParamFile stdout;
	
	
	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.5 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Run Solar Eclipse";
	private static final String longDescription = "JIST wrapper to run Solar Eclipse Polygenic model found here: http://www.nitrc.org/projects/se_linux/";

	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createInputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		
		//The base command (i.e., some command in ~/SOLARPATH/bin/<mycommand>
		//inputParams.add(solarPath = new ParamFileCollection("Absolute_path_to_Solar",new FileExtensionFilter(new String[]{"txt"})));
		inputParams.add(pedigree = new ParamFileCollection("All of the pedigree files to run (.ped)", new FileExtensionFilter(new String[]{"ped","txt"})));
		pedigree.setMandatory(true);
		inputParams.add(phenotypes = new ParamFileCollection("All of the phenotype files to run (.csv)",new FileExtensionFilter(new String[]{"csv","txt"})));
		phenotypes.setMandatory(true);
		inputParams.add(trait = new ParamFileCollection("All of the traits",new FileExtensionFilter(new String[]{"txt"})));
		trait.setMandatory(true);
		inputParams.add(covariates = new ParamFileCollection("All of the covariates",new FileExtensionFilter(new String[]{"txt"})));
		covariates.setMandatory(true);
		inputParams.add(imageData = new ParamVolume("File of the images to include in the model"));
		imageData.setMandatory(false);

		//Options
		inputParams.add(screen = new ParamBoolean("Do you want to use the -screen option?"));
		screen.setMandatory(false);
		inputParams.add(prob = new ParamFloat("Probability level for keeping covariates significant",(float) 0.0,(float) 1.0,(float) 0.1));
		prob.setMandatory(false);
		inputParams.add(fix = new ParamFileCollection("Covariates to fix regardless of probability level"));
		fix.setMandatory(false);
		inputParams.add(all = new ParamBoolean("Fix all covariates?"));
		all.setMandatory(false);
		inputParams.add(testcovar = new ParamFileCollection("Covaraite that you want to test significance of"));
		testcovar.setMandatory(false);
		inputParams.add(testrhoe = new ParamBoolean("Test significance of rhoe"));
		testrhoe.setMandatory(false);
		inputParams.add(testrhog = new ParamBoolean("Test significance of rhog"));
		testrhog.setMandatory(false);
		inputParams.add(testrhoc = new ParamBoolean("Test significance of rhoc"));
		testrhoc.setMandatory(false);
		inputParams.add(testrhop = new ParamBoolean("Test significance of rhop"));
		testrhop.setMandatory(false);
		inputParams.add(rhopse = new ParamBoolean("Get standard error of rhop"));
		rhopse.setMandatory(false);
		inputParams.add(sporadic = new ParamBoolean("Evaluate only sporadic models"));
		sporadic.setMandatory(false);
		inputParams.add(keephouse = new ParamBoolean("keep the household effect"));
		keephouse.setMandatory(false);
		inputParams.add(house = new ParamBoolean("House - Examine the house effect"));
		house.setMandatory(false);
		
		inputParams.setPackage("MASI");
		inputParams.setCategory("External.SolarEclipse");
		inputParams.setLabel("Solar Eclipse Polygenic");
		inputParams.setName("Solar_Eclipse_Polygenic");

		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("MASI","Steve Damon","Electrical Engineering"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(true);
		info.setStatus(DevelopmentStatus.BETA);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createOutputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(lastMod = new ParamFile("last.mod file"));
		outputParams.add( nocovarMod= new ParamFile("nocovar.mod file"));
		outputParams.add( nocovarOut= new ParamFile("nocovar.out file"));
		outputParams.add( null0Mod= new ParamFile("null0.mod file"));
		outputParams.add( polygenicLogsOut= new ParamFile("polygenicLogs.out file"));
		outputParams.add( polygenicOut= new ParamFile("polygenic.out file"));
		outputParams.add( polygenicResid= new ParamFile("polygenic.residuals file"));
		outputParams.add( polygenicResidStats= new ParamFile("polygenic.residual.stats file"));
		outputParams.add( polyMod= new ParamFile("poly.mod file"));
		outputParams.add( polyOut= new ParamFile("poly.out file"));
		outputParams.add( sporMod= new ParamFile("spor.mod file"));
		outputParams.add( sporOut= new ParamFile("spor.out file"));
		outputParams.add(stdout = new ParamFile("EXT STD OUT"));
		outputParams.add(stderr = new ParamFile("EXT STD ERR"));
	}


	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#execute(edu.jhu.ece.iacl.pipeline.CalculationMonitor)
	 */
	@Override
	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		Solar solar = new Solar();
		boolean isInstalled = false;
		
		//Check if solar is installed and throw and exception if it isn't
		isInstalled = solar.getStatus();
		
		if (!isInstalled){
			JistLogger.logError(JistLogger.SEVERE, "ERROR: Solar does not appear to be installed. Please ensure that:\n\t(1)Solar is in your path\n\t(2)Solar will start without core dumping.");
			throw new RuntimeException("ERROR: Solar does not appear to be installed.");
		}
		
		for (int i=0;i<pedigree.size();i++){
			File outputDir = this.getOutputDirectory();
			String command = Polygenic.solarStringBuilder(pedigree, phenotypes, trait, covariates, imageData, screen, prob, fix, all, testcovar, testrhoe, testrhog, testrhoc, testrhop, rhopse, sporadic, keephouse,house, outputDir,i);
			
			System.out.println("Running the following polygenic model:");
			System.out.println(command);
					
			Process p;
			int exitStatus =0;
			try {
				p = Runtime.getRuntime().exec(command);
				p.waitFor();
				exitStatus = p.exitValue();
				if (exitStatus!=0){
					throw new RuntimeException(this.getAlgorithmName()+" Failed with exit status "+String.valueOf(exitStatus));
				}
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			
			//Read stdout and write to file solar.out
			File thisOut = new File(this.getOutputDirectory()+File.separator+"solar.out");
			File thisErr = new File(this.getOutputDirectory()+File.separator+"solar.err");
			String line;
			JistLogger.logOutput(JistLogger.INFO,"*************************Begin External CMD STDOUT*************************");

			try {
				p = Runtime.getRuntime().exec(command);
				BufferedWriter writer = new BufferedWriter(new FileWriter(thisOut));
				BufferedWriter writerErr = new BufferedWriter(new FileWriter(thisErr));
				
				BufferedReader in = new BufferedReader(new InputStreamReader(p.getInputStream()));
				BufferedReader err = new BufferedReader(new InputStreamReader(p.getErrorStream()));
			       while ((line = in.readLine()) != null) {
			         writer.write(line);
			         JistLogger.logOutput(JistLogger.INFO,line+"\n");
			       }
			       while ((line = err.readLine())!=null){
			    	   writerErr.write(line);
			       }
			       writerErr.close();
			       err.close();
			       in.close();
			       writer.close();
		       p.waitFor();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			JistLogger.logOutput(JistLogger.INFO,"**************************End External CMD STDOUT**************************");

			
			lastMod.setValue(new File(this.getOutputDirectory()+File.separator+"last.mod"));;
			nocovarMod.setValue(new File(this.getOutputDirectory()+File.separator+"nocovar.mod"));
			nocovarOut.setValue(new File(this.getOutputDirectory()+File.separator+"nocovar.out"));
			null0Mod.setValue(new File(this.getOutputDirectory()+File.separator+"null0.mod"));
			polygenicLogsOut.setValue(new File(this.getOutputDirectory()+File.separator+"polygenic.logs.out"));
			polygenicOut.setValue(new File(this.getOutputDirectory()+File.separator+"polygenic.out"));
			polygenicResid.setValue(new File(this.getOutputDirectory()+File.separator+"polygenic.residuals"));
			polygenicResidStats.setValue(new File(this.getOutputDirectory()+File.separator+"polygenic.residuals.stats"));
			polyMod.setValue(new File(this.getOutputDirectory()+File.separator+"poly.mod"));
			polyOut.setValue(new File(this.getOutputDirectory()+File.separator+"poly.out"));
			sporMod.setValue(new File(this.getOutputDirectory()+File.separator+"spor.mod"));
			sporOut.setValue(new File(this.getOutputDirectory()+File.separator+"spor.out"));
			stdout.setValue(thisOut);
			stderr.setValue(thisErr);
			
		}
		
	}
 
}

