package edu.vanderbilt.masi.plugins.utilities;

import java.io.File;
import java.io.FileNotFoundException;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
import java.io.*;

import edu.vanderbilt.masi.algorithms.utilities.LabelAnalysis;

public class PluginAccuracyAnalysis extends ProcessingAlgorithm {

	public String alglabel = "Segmentation Accuracy Analysis";
	public String algname = "Segmentation_Accuracy_Analysis";
	
	// input parameters
	public ParamVolume truthvol;
	public ParamVolume estvol;
	
	// output parameters
	public ParamFile summaryfile;
	
	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Runs segmentation accuracy analysis given 'truth' and 'estimated' label volumes.";
	private static final String longDescription = "";
	
	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createInputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Asman","andrew.j.asman@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Asman"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.RC);
		
		inputParams.setPackage("MASI");
		inputParams.setCategory("Utilities");
		inputParams.setLabel("Segmentation Accuracy Analysis");
		inputParams.setName("Segmentation_Accuracy_Analysis");
		
		// set the truth volume
		inputParams.add(truthvol=new ParamVolume("Truth Volume"));
		truthvol.setLoadAndSaveOnValidate(false);
		
		// set the estimated volume
		inputParams.add(estvol=new ParamVolume("Estimated Volume"));
		estvol.setLoadAndSaveOnValidate(false);
	}
	
	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createOutputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.setName("Segmentation_Accuracy_Analysis");
		
		// handle the accuracy output
		outputParams.add(summaryfile = new ParamFile("Accuracy Summary"));
	}
	
	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		try {
			ExecuteWrapper wrapper=new ExecuteWrapper();
			monitor.observe(wrapper);
			wrapper.execute(this);

		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	protected class ExecuteWrapper extends AbstractCalculation {
		public void execute(ProcessingAlgorithm alg) throws FileNotFoundException{
			
			// load the truth volume
			float[][] dicevals = LabelAnalysis.dice(truthvol, estvol);
			
			// Set the output directory
			File outdir = new File(
					alg.getOutputDirectory() +
					File.separator +
					FileUtil.forceSafeFilename(alg.getAlgorithmName()) +
					File.separator);
			outdir.mkdirs();
			
			// Set the output string
			String out = "";
			int num_unique_labels = dicevals.length;
			for (int l = 0; l < num_unique_labels; l++)
				out = String.format("%sDice (label %02d): %f\n", out, (int)dicevals[l][0], dicevals[l][1]);
			
			// calculate the mean Dice
			float meandice = 0;
			for (int l = 1; l < num_unique_labels; l++)
				meandice += dicevals[l][1];
			meandice /= (num_unique_labels-1);
			out = String.format("%sMean Dice (Non-Background): %f\n", out, meandice);
			
			// Create the file
			try {
				// write the output to the file
				File f = new File(outdir, String.format("Segmentation_Accuracy.txt")); 
				FileWriter fstream = new FileWriter(f);
				BufferedWriter b = new BufferedWriter(fstream);
				b.write(out);
				b.close();
				fstream.close();
				
				// write the summary file to disk
				summaryfile.setValue(f);
				summaryfile.writeAndFreeNow(alg);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}

}
