package apps;

import data.*;
import misc.*;
import tools.*;

import java.util.*;
import java.util.logging.*;
import java.util.regex.*;
import java.io.*;
import java.text.*;

/**
 * Converts point sets to scheme files, given additional imaging parameters.
 * 
 * @author Philip Cook
 * @version $Id: PointSetToScheme.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public class PointSetToScheme {

    /**
     * logging object
     */
    private static final Logger logger = Logger.getLogger("apps.PointSetToScheme");


    public static void main(String[] args) {

	CL_Initializer.CL_init(args);

	// need diffusion time and modQ from command line

	double modQ = CL_Initializer.modQ;

	double tau = CL_Initializer.diffusionTime;

	int M = CL_Initializer.M;
	int N = CL_Initializer.N;

	boolean flipX = false;
	boolean flipY = false;
	boolean flipZ = false;

	// insist on UK style numbers, no comma decimal points
	Locale.setDefault(Locale.UK);

	// global b value
	double b = 0.0;

	// Some scanners define b-value as |g|^2 * \beta
	// where \beta is what they CLAIM the b-value is.
	// If you use normalized gradient directions, you need to increase b accordingly
	// to make the same ADC calculation.
	boolean useGradMod = false;

	DecimalFormat df = new DecimalFormat("0.000000");

	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-fixedbvalue")) {
		M = Integer.parseInt(args[i+1]);
		N = Integer.parseInt(args[i+2]);
		b = Double.parseDouble(args[i+3]);
		tau = Double.parseDouble(args[i+4]);

		modQ = Math.sqrt(b / tau);

		CL_Initializer.markAsParsed(i,5);
	    }
	    else if (args[i].equals("-usegradmod")) {
		useGradMod = true;
		logger.info("Gradient direction magnitude will be incorporated into b-values");
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-flipx")) {
                flipX = true;
                CL_Initializer.markAsParsed(i);
            }
            else if (args[i].equals("-flipy")) {
                flipY = true;
                CL_Initializer.markAsParsed(i);

            }
            else if (args[i].equals("-flipz")) {
                flipZ = true;
                CL_Initializer.markAsParsed(i);
            }

	}

	CL_Initializer.checkParsing(args);
	
	try {

	    double[][] points = readPoints(CL_Initializer.inputFile, flipX, flipY, flipZ);

	    double[] mod = getPointModulus(points);

	    points = normalizePoints(points);

	    StringBuffer buffer = new StringBuffer();

	    buffer.append(tau + "\n");
	    buffer.append((N + M) + "\n");
	    
	    for (int i = 0; i < M; i++) {
		buffer.append("0.0\n0.0\n0.0\n");
	    }

	    boolean warnAboutGradDirMod = false;

	    for (int i = 0; i < N; i++) {
		for (int j = 0; j < 3; j++) {
		    if (useGradMod) {
			buffer.append(df.format(points[i][j] * modQ * mod[i]) + "\n");
		    }
		    else {

			if ( mod[i] != 0.0 && !(Math.abs(1.0 - mod[i]) < 1E-5) ) { 
			    warnAboutGradDirMod = true;
			}
			
			buffer.append(df.format(points[i][j] * modQ) + "\n");
		    }
		}
	    }

	    if (OutputManager.outputFile == null) {
		System.out.print(buffer.toString());
	    }
	    else {
		
		FileOutput out = new FileOutput(OutputManager.outputFile);
		
		out.writeString(buffer.toString());
		
		out.close();
	    }


	    if (warnAboutGradDirMod) {
		logger.warning("Some measurements have non-unit gradient directions. Directions have been " + 
			       "normalized to unit length");
	    }

	    
	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}


	   
    }



    /**
     * Reads points in the format <BR> <code>
     * numPoints <BR>
     * x y z <BR>
     * x y z <BR>
     * ... <BR> </code>
     * or in the format <BR> <code>
     * numPoints <BR>
     * x <BR>
     * y <BR> 
     * z <BR>
     * x <BR>
     * ... <BR> </code>
     * from a text file. If the points are not unit vectors, they are normalized.
     * 
     * @param filename the name of the file to read from. If <code>null</code>, the method reads
     * from the standard input. 
     */
    public static double[][] readPoints(String filename) throws IOException {
	return readPoints(filename, false, false, false);
    }


    /**
     * Reads points in the format <BR> <code>
     * numPoints <BR>
     * x y z <BR>
     * x y z <BR>
     * ... <BR> </code>
     * or in the format <BR> <code>
     * numPoints <BR>
     * x <BR>
     * y <BR> 
     * z <BR>
     * x <BR>
     * ... <BR> </code>
     * or
     * <code>
     * x y z <BR>
     * x y z <BR>
     * ... <BR> </code> 
     * from a text file. Does not normalize points to unity, call <code>normalizePoints</code> to do that.
     * 
     * @param filename the name of the file to read from. If <code>null</code>, the method reads
     * from the standard input. 
     */
    public static double[][] readPoints(String filename, boolean flipX,
					boolean flipY, boolean flipZ) throws IOException {


	// handle input a little differently because it's text
	Vector<String> lines = new Vector<String>();

	Scanner fileScanner = null;

	if (filename == null) {
	    fileScanner = new Scanner(System.in);
	}
	else {
	    fileScanner = new Scanner(new File(filename));
	}

	// Read in the file line by line.
	fileScanner.useDelimiter("\r\n|\n");

	// Store all the lines in a vector.
	while(fileScanner.hasNext()) {
	    
	    // ignore empty lines - avoids problems with trailing newlines
	    
	    String next = fileScanner.next();
	    
	    if (next.length() > 0) {
		lines.add(next);
	    }
	    
	}

	fileScanner.close();
	    
	
	// now figure out format of input file
	// if there is a header line, the first entry must be the number of points
	boolean firstLineIsHdr = false;

	boolean oneNumberPerLine = false;
	
	int numberOfPoints = 0;
	
	try {
	    String[] firstLineElements = lines.elementAt(0).trim().split("\\s+");

	    numberOfPoints = Integer.parseInt(firstLineElements[0]);
	    
	    // does this number correspond to the number of points?
	    if ( numberOfPoints == ((lines.size() - 1) / 3) || numberOfPoints == (lines.size() - 1) ) {
		firstLineIsHdr = true;
	    }
	    else {
		// if the first line does not start with the number of points, it must be a point
		
		if (firstLineElements.length != 3) {
		    throw new LoggedException("Unknown point set format");
		}
		
		numberOfPoints = lines.size();
	    }

	}
	catch (NumberFormatException e) {
	    // if number of points is not first, then there must be three numbers per line
	    // and number of lines == number of points
	    numberOfPoints = lines.size();
	}
	    
	
	if (firstLineIsHdr) {
	    try {
		Double.parseDouble(lines.elementAt(1).trim());
		oneNumberPerLine = true;
	    }
	    catch (NumberFormatException e) {
		oneNumberPerLine = false;
	    }
	}

	double[][] points = new double[numberOfPoints][3];

	if (oneNumberPerLine) {	
	    for (int i = 0; i < numberOfPoints; i++) {
		for (int j = 0; j < 3; j++) {
		    double component = Double.parseDouble(lines.elementAt(i * 3 + j + 1));
		    points[i][j] = component;
		}
	    }
	}
	else {

	    int offset = firstLineIsHdr ? 1 : 0;

	    for (int i = 0; i < numberOfPoints; i++) {
		
		String[] comps = lines.elementAt(i+offset).trim().split("\\s+");
		
		if (comps.length > 3) {
		    throw new LoggedException("Unknown point set format");
		}
		
		for (int c = 0; c < 3; c++) {
		    double comp = Double.parseDouble(comps[c]);
		    points[i][c] = comp;
		}
	    }
	}
	

	return points;

    }
	

    /**
     * Gets the modulus of the vector from (0, 0, 0) to the point.
     *
     */
    public static double[] getPointModulus(double[][] points) {

	int numberOfPoints = points.length;

	double[] mod = new double[numberOfPoints];

	// normalize points
	for (int i = 0; i < numberOfPoints; i++) {
	    mod[i] = Math.sqrt(points[i][0] * points[i][0] + points[i][1] * points[i][1] + 
			    points[i][2] * points[i][2]);
	}

	return mod;
    }


    /**
     * Normalizes points to unit length.
     *
     */
    public static double[][] normalizePoints(double[][] points) {

	int numberOfPoints = points.length;

	double[][] normPoints = new double[numberOfPoints][3];

	// normalize points
	for (int i = 0; i < numberOfPoints; i++) {
	    double mod = Math.sqrt(points[i][0] * points[i][0] + points[i][1] * points[i][1] + 
				   points[i][2] * points[i][2]);

	    // don't worry about small rounding errors
	    if (Math.abs(1.0 - mod) > 1E-6 && mod > 0.0) {
		points[i][0] = points[i][0] / mod;
		points[i][1] = points[i][1] / mod;
		points[i][2] = points[i][2] / mod;
	    }

	    // get rid of -0.0
	    for (int j = 0; j < 3; j++) {
		if (points[i][j] == 0.0) { 
		    points[i][j] = Math.abs(points[i][j]);
		}
	    }

	}

	return points;
    }
    

}
