package apps;

import imaging.*;

import misc.LoggedException;

import numerics.*;
import tools.*;
import data.*;
import tractography.*;

import java.util.logging.*;
import java.util.Random;
import java.util.zip.*;


import java.io.*;


/**
 * Processes streamline output from StreamlineTractography or from a previous invocation of this
 * program. The program can output streamlines or connection probability images. It can also be
 * used to change the format of streamlines stored on disk.  
 *
 * @author Philip Cook
 * @version $Id: ProcessStreamlines.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @see apps.StreamlineTractography
 */
public class ProcessStreamlines {

    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.apps.ProcessStreamlines");

    private static int MAX_PDS = 1000;


    /**
     * Output manager
     */
    private static OutputManager om;


    public static void main(String[] args) {
	
	if (args.length == 0) {
	    System.exit(0);
	}


	// default is to output raw binary
	boolean outputTractVoxels = false;
	
	boolean outputTractsOOGL = false;
	
	boolean binaryOOGL = false;
	
	boolean outputCP = false; // if true, output connection probability maps
	
	// if true, implies outputCP
	boolean outputCBS = false;

 	// if true, implies outputCP
	// "anatomical connectivity map", global streamline count
	boolean outputACM = false;
	
	// if true, implies outputCP
	boolean normalizeCP = false;

	// dimensions of seed space
	int xDataDim = 0;
	int yDataDim = 0;
	int zDataDim = 0;

	double xVoxelDim = 0.0;
	double yVoxelDim = 0.0;
	double zVoxelDim = 0.0;

	String seedFile = null;
	String targetFile = null;
	String waypointFile = null;
	String exclusionFile = null;
	String endFile = null;

        // target options
        boolean directionalCP = false; // by default, don't produce separate left / right maps

	Vector3D forwards = null;

        boolean countFirstEntry = true; // by default, break on entry to target region


	// discard streamlines on entry to exclusion ROI
	// default behaviour
	boolean discardOnExclusionEntry = true; 

	// truncate at second waypoint entry point
	boolean truncateLoops = false;

	// discard if fibers loop
	boolean discardLoops = false;

        FreeFormROI allROIs = null;

	double seedPointX = -1.0;
	double seedPointY = -1.0;
	double seedPointZ = -1.0;

	boolean voxelCoordinates = false; // true if seeds are specified as voxels


        // regionIndex counts from zero, but for I/O purposes we
        // index from 1. If it's -1 then we process all regions in the seed file in sequence
        int regionIndex = -1;

	// likewise the seed
	int seedIndex = -1;

        int minTractPoints = 0;	

        int maxTractPoints = 0;

        double minTractLength = 0.0;	

        double maxTractLength = 0;	

	String outputRoot = null;

	double resampleStepSize = 0.0;

	CL_Initializer.inputModel = "raw";

	// default number of PDs is 1
	CL_Initializer.numPDsIO = 1;

	CL_Initializer.CL_init(args);

	// gzip output if true
	boolean gzip = OutputManager.gzipOut;


	// number of iterations for tracking
	int iterations = 0;
	
	// do not print to stderr if true
	boolean silent = false;


	for (int i = 0; i < args.length; i++) {

	    // image args
            if (args[i].equals("-resamplestepsize")) { 
		resampleStepSize = Double.parseDouble(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-noresample")) { 
		resampleStepSize = -1.0;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-seedpointmm")) {
		seedPointX = Double.parseDouble(args[i + 1]);
		seedPointY = Double.parseDouble(args[i + 2]);
		seedPointZ = Double.parseDouble(args[i + 3]);
		CL_Initializer.markAsParsed(i, 4);
	    }
 	    else if (args[i].equals("-seedpointvox")) {
		seedPointX = Double.parseDouble(args[i + 1]);
		seedPointY = Double.parseDouble(args[i + 2]);
		seedPointZ = Double.parseDouble(args[i + 3]);
		voxelCoordinates = true;
		CL_Initializer.markAsParsed(i, 4);
	    } 
	    else if (args[i].equals("-seedfile")) {
		seedFile = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
 	    else if (args[i].equals("-regionindex")) {
                regionIndex = Integer.parseInt(args[i+1]) - 1;
                CL_Initializer.markAsParsed(i, 2);
	    } 
 	    else if (args[i].equals("-seedindex")) {
                int tmp = Integer.parseInt(args[i+1]);

                if (tmp <= 0) {
                    throw new LoggedException("Invalid seed index specified. Minimum index is 1");
                }
                
                seedIndex = tmp - 1;
                CL_Initializer.markAsParsed(i, 2);
	    } 
            else if (args[i].equals("-waypointfile")) {
                waypointFile = args[i + 1];
                CL_Initializer.markAsParsed(i,2);
	    }
            else if (args[i].equals("-exclusionfile")) {
                exclusionFile = args[i + 1];
                CL_Initializer.markAsParsed(i,2);
	    }
            else if (args[i].equals("-targetfile")) {
                targetFile = args[i + 1];
                CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-endpointfile")) {
		endFile = args[i + 1];
		CL_Initializer.markAsParsed(i, 2);
	    } 
            else if (args[i].equals("-mintractpoints")) {
		minTractPoints = Integer.parseInt(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
            else if (args[i].equals("-mintractlength")) {
		minTractLength = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
            else if (args[i].equals("-maxtractpoints")) {
		maxTractPoints = Integer.parseInt(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
            else if (args[i].equals("-maxtractlength")) {
		maxTractLength = Double.parseDouble(args[i+1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-truncateinexclusion")) {
		discardOnExclusionEntry = false;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-truncateloops")) {
		truncateLoops = true;
                CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-discardloops")) {
		discardLoops = true;
                CL_Initializer.markAsParsed(i);
	    }
            else if (args[i].equals("-allowmultitargets")) {
                countFirstEntry = false;
                CL_Initializer.markAsParsed(i);
	    }
             else if (args[i].equals("-directional")) {
                 directionalCP = true;
		 if (args.length < i+4) {
		     throw new LoggedException("-directionalcp requires a vector defining the forward direction");
		 }
		 forwards = new Vector3D(Double.parseDouble(args[i+1]), Double.parseDouble(args[i+2]),
					 Double.parseDouble(args[i+3]));

                 CL_Initializer.markAsParsed(i, 4);
 	    }
	    else if (args[i].equals("-outputtracts")) {
		if (i < args.length - 1) {

                    if (args[i+1].equals("oogl")) {
                        outputTractsOOGL = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("ooglbinary")) {
                        outputTractsOOGL = true;
			binaryOOGL = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("voxels")) {
                        outputTractVoxels = true;
                        CL_Initializer.markAsParsed(i, 2);
                    }
                    else if (args[i+1].equals("raw")) { 
                        CL_Initializer.markAsParsed(i, 2);
                    }

                }
		else { 
		    CL_Initializer.markAsParsed(i, 1);
		}
		// raw is default

		   
	    }
	    else if  (args[i].equals("-outputcp")) {
		outputCP = true;
		normalizeCP = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if  (args[i].equals("-outputcbs")) {
		outputCBS = true;
		outputCP = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if  (args[i].equals("-outputsc")) {
		outputCP = true;
		normalizeCP = false;
		CL_Initializer.markAsParsed(i);
	    }
	    else if  (args[i].equals("-outputacm")) {
		outputCP = true;
		outputACM = true;
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-iterations")) {
		iterations = Integer.parseInt(args[i + 1]);
		CL_Initializer.markAsParsed(i, 2);
	    }
	    else if (args[i].equals("-outputroot")) {
		outputRoot = args[i + 1];
		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-silent")) {
		silent = true;
	    	CL_Initializer.markAsParsed(i);
	    }
	}	    

	CL_Initializer.checkParsing(args);

	xDataDim = CL_Initializer.dataDims[0];
	yDataDim = CL_Initializer.dataDims[1];
	zDataDim = CL_Initializer.dataDims[2];

	xVoxelDim = CL_Initializer.voxelDims[0];
	yVoxelDim = CL_Initializer.voxelDims[1];
	zVoxelDim = CL_Initializer.voxelDims[2];


	if (truncateLoops && discardLoops) {
	    logger.warning("Discard and truncate loops both selected. Program will truncate loops " + 
			   "(run again witth discard option to remove them if desired)");
	    
	    discardLoops = false;
	}


	if (iterations == 0) {
	    if (outputCP) {
		iterations = StreamlineTractography.DEFAULT_PICO_ITERATIONS;
	    }
	    else {
		iterations = 1;
	    }
	}
       
	if (outputCBS && seedIndex > 0) {
	    throw new LoggedException("Cannot specify -seedindex with -outputcbs. Concatenate all seeds " + 
				      "from the ROI into a single file and use -regionIndex");
	}

	// NOTE: this may be wrong! Normally we can guess numPDs from the input streamlines
	// CBS requires a bit more care and makes use of this variable
	int numPDs = CL_Initializer.numPDsIO;

	// If dims are specified as args, then these will be used to check files for consistency

	// If dims are all zero at this point, then they will be populated from the first
	// file (seed, waypoint, etc) that is specified, and subsequent files must agree in data
	// and voxel dims
	int[] inputDataDims = new int[] {xDataDim, yDataDim, zDataDim};
	double[] inputVoxelDims = new double[] {xVoxelDim, yVoxelDim, zVoxelDim};


	// header from input, either a seed vol or another vol
	// if a seed vol is specified, then it is used to initialize the output headers
	// otherwise, use targets, if they exist.
	ImageHeader ihInit = null;
       

	if (seedFile != null) {

	    // if seed file is given, take data dims and voxel dims from seed file

	    try {
		ihInit = ImageHeader.readHeader(seedFile);
	    }
	    catch (IOException e) {
		throw new LoggedException(e);
	    }

	    // get / check data and voxel dims
	    if (!ImageHeader.checkDims(seedFile, inputDataDims, inputVoxelDims)) {
		throw new LoggedException("Inconsistent data or voxel dimensions specified");
	    }

	}
    
        short[][][] waypointVol = null;

        if (waypointFile != null) {

	    if (ihInit == null) {
		try {
		    ihInit = ImageHeader.readHeader(waypointFile);
		}
		catch (IOException e) {
		    throw new LoggedException(e);
		}
	    }

            waypointVol = StreamlineTractography.readShortVolume(waypointFile);

	    if (!ImageHeader.checkDims(waypointFile, inputDataDims, inputVoxelDims)) {
		throw new LoggedException("Inconsistent data or voxel dimensions specified");
	    }

        }

        short[][][] exclusionVol = null;

        if (exclusionFile != null) {

	    if (ihInit == null) {
		try {
		    ihInit = ImageHeader.readHeader(exclusionFile);
		}
		catch (IOException e) {
		    throw new LoggedException(e);
		}
	    }

            exclusionVol = StreamlineTractography.readShortVolume(exclusionFile);

	    if (!ImageHeader.checkDims(exclusionFile, inputDataDims, inputVoxelDims)) {
		throw new LoggedException("Inconsistent data or voxel dimensions specified");
	    }

        }

        short[][][] targetVol = null;
        
        if (targetFile != null) {

	    if (ihInit == null) {
		try {
		    ihInit = ImageHeader.readHeader(targetFile);
		}
		catch (IOException e) {
		    throw new LoggedException(e);
		}
	    }


            targetVol = StreamlineTractography.readShortVolume(targetFile);

	    if (!ImageHeader.checkDims(targetFile, inputDataDims, inputVoxelDims)) {
		throw new LoggedException("Inconsistent data or voxel dimensions specified");
	    }

        }


        short[][][] endVol = null;

        if (endFile != null) {

	    if (ihInit == null) {
		try {
		    ihInit = ImageHeader.readHeader(endFile);
		}
		catch (IOException e) {
		    throw new LoggedException(e);
		}
	    }

            endVol = StreamlineTractography.readShortVolume(endFile);

	    if (!ImageHeader.checkDims(endFile, inputDataDims, inputVoxelDims)) {
		throw new LoggedException("Inconsistent data or voxel dimensions specified");
	    }

        }

	xDataDim = inputDataDims[0];
	yDataDim = inputDataDims[1];
	zDataDim = inputDataDims[2];

	xVoxelDim = Math.abs(inputVoxelDims[0]);
	yVoxelDim = Math.abs(inputVoxelDims[1]);
	zVoxelDim = Math.abs(inputVoxelDims[2]);


	// get seeds
	if (seedFile == null) {

	    if (seedPointX < 0.0 || seedPointY < 0.0 || seedPointZ < 0.0) {
		if (outputCP && !outputACM) {
		    throw new LoggedException("Seed points must be specified for connection " + 
					      "probability images");
		}
	    }
	    else {
		short[][][] seedVol = new short[xDataDim][yDataDim][zDataDim];
		
		if (voxelCoordinates) {
		    seedVol[(int)seedPointX][(int)seedPointY][(int)seedPointZ] = 1;
		} else {
		    seedVol[(int)(seedPointX / xVoxelDim) ][(int)(seedPointY / yVoxelDim)]
			[(int)(seedPointZ / zVoxelDim)] = 1;
		}
		
		allROIs = new FreeFormROI(seedVol, xVoxelDim, yVoxelDim, zVoxelDim);
	    }
	    
	}
	else {
	    short[][][] seedVol = StreamlineTractography.readShortVolume(seedFile);
	    allROIs = new FreeFormROI(seedVol, xVoxelDim, yVoxelDim, zVoxelDim);
	}


	if (allROIs == null && !outputACM) {
	    if (outputRoot != null) {
		throw new LoggedException("Without ROI information, output is to a single stream " + 
					  "(default stdout). Specify a file with -outputfile");
	    }
	}
	else {
	    // if seed point or seed file given, we structure output by ROI
	    if (outputRoot == null) {
		outputRoot = "";
	    }
	}

	if (outputRoot == null) {
	    om = new OutputManager();
	}

		
	TractSource tractSource = new TractSource(CL_Initializer.inputFile, CL_Initializer.inputModel, 
						  xVoxelDim, yVoxelDim, zVoxelDim);
	

	if (outputCBS && targetVol == null) {
	    throw new LoggedException("Cannot do connectivity segmentation without targets");
	}
	
	if ((outputTractVoxels || outputCP) && (xDataDim == 0.0 || yDataDim == 0.0 || zDataDim == 0.0 || 
			 xVoxelDim == 0.0 || yVoxelDim == 0.0 || zVoxelDim == 0.0)) {
	    throw new 
		LoggedException("Connection probability and voxel output requires " +
				" data and voxel dimensions to be specified.");
	}


	StreamlineROI_Filter filter = 
	    new StreamlineROI_Filter(xDataDim, yDataDim, zDataDim, xVoxelDim, yVoxelDim, zVoxelDim);

	if (waypointFile != null) {
	    filter.setWaypoints(waypointVol);
	    filter.setTruncateLoops(truncateLoops);
	    filter.setDiscardLoops(discardLoops);
        }

	if (exclusionFile != null) {
	    filter.setExclusionROIs(exclusionVol);
	    filter.setDiscardOnExclusionEntry(discardOnExclusionEntry);
	    
	}        

	if (endFile != null) {
	    filter.setEndZones(endVol);
	}        


	// if resampleStepSize == 0.0 use default resampling scheme
	// if < 0 then do not resample
	// else use specified step size
	if (resampleStepSize > 0.0) {
	    filter.setResampleStepSize(resampleStepSize);
	}
	else if (resampleStepSize < 0.0 || CL_Initializer.inputModel.equals("voxels") || zVoxelDim == 0.0) {
	    // no need to resample if we have voxel input, it's already been done

	    // if no waypoints or targets, then no need to resample 
	    filter.setResampleTracts(false);
	}
	
	if (minTractPoints > 0) {
	    filter.setMinTractPoints(minTractPoints);
	}
	if (minTractLength > 0.0) {
	    filter.setMinTractLength(minTractLength);
	}
	if (maxTractPoints > 0) {
	    filter.setMaxTractPoints(maxTractPoints);

	    // makes no sense to resample tracts if we are restricting the maximum number of points
	    filter.setResampleTracts(false);
	}
	if (maxTractLength > 0.0) {
	    filter.setMaxTractLength(maxTractLength);
	}

	
	// number ROIs from 1 upwards to maintain correspondence to 
	// values in seed file.
	int outputRegionID = regionIndex + 1; 
	
	
	// connection probs
	AnalyzeHeader ahFloat = null;
	
	// streamline counts
	AnalyzeHeader ahInt = null;
	
	// cbs
	AnalyzeHeader ahShort = null;
	
	ahFloat = new AnalyzeHeader();
	ahFloat.bitpix = 32;

	ahFloat.width = (short)xDataDim;
	ahFloat.height = (short)yDataDim;
	ahFloat.depth = (short)zDataDim;
	
	// output negative pixel dimensions if they were input
	ahFloat.pixelWidth = (float)inputVoxelDims[0];
	ahFloat.pixelHeight = (float)inputVoxelDims[1];
	ahFloat.pixelDepth = (float)inputVoxelDims[2];

	// take centre and pixelDims from ihInit if possible
	// this preserves sign of pixelDims in output
	// for internal processing, we always use positive voxel dimensions
	if (ihInit != null) {

	    double[] origin = ihInit.getOrigin();

	    ahFloat.centre = new short[] {(short)origin[0], (short)origin[1], (short)origin[2]};

	    ahFloat.pixelWidth = (float)ihInit.xVoxelDim();
	    ahFloat.pixelHeight = (float)ihInit.yVoxelDim();
	    ahFloat.pixelDepth = (float)ihInit.zVoxelDim();
	}
	
	ahFloat.datatype = AnalyzeHeader.DT_FLOAT;

	ahFloat.description = "Camino procstreamlines";
	
	ahInt = new AnalyzeHeader(ahFloat);
	ahInt.datatype = AnalyzeHeader.DT_SIGNED_INT;
	
	ahShort = new AnalyzeHeader(ahFloat);
	ahShort.datatype = AnalyzeHeader.DT_SIGNED_SHORT;
	ahShort.bitpix = 16;
	
	
	try {

	    int numRegions = 1;

	    if (allROIs != null) {
		numRegions = allROIs.numberOfRegions();
	    }
	    
	    // do things on a tract by tract basis, which is cumbersome
	    // can change to read the entire collection for each seed point if this
	    // seems reasonable
	    
	    // output numbering for CBS images is
	    // outputRoot%d_%d_%d, regionIndex+1, pd, direction
	    // if directional, output 0 and 1 for direction, where 0 is in the direction
	    // of the forwards vector

	    // output numbering for targetCP images is
	    // outputRoot%d_%d_%d_%d, regionIndex+1, sp+1, pd, direction
	    // if directional, output 0 and 1 for direction, where 0 is in the direction
	    // of the forwards vector

	    // output numbering for raw CP images is
	    // outputRoot%d_%d_%d, regionIndex+1, sp+1, pd
	    
	    if (outputACM) {
		
		int tractCounter = 0;

		if (!silent) {
	    	    System.err.println();
		}
    
		ConnectionProbabilityImage acm = 
		    new ConnectionProbabilityImage(xDataDim, yDataDim, zDataDim, 
						   xVoxelDim, yVoxelDim, zVoxelDim);
		
		while (tractSource.more()) {
		    
		    if (!silent) {		    
		        System.err.print("\rProcessing streamline " + (tractCounter+1));
		    }
	
		    TractCollection nextTC = new TractCollection(2, 100.0);
		    
		    nextTC.addTract(tractSource.nextTract());
		    
		    nextTC = filter.processTracts(nextTC);
		    
		    acm.processTracts(nextTC);
			
		    tractCounter++;
		}

		// now output
		if (normalizeCP) {
		    double[][][] cp = acm.getConnectionProbabilities();
		    
		    AnalyzeHeader.writeImage(cp, ahFloat, outputRoot + "acm_cp", gzip);
		}
		else {
		    int[][][] sc = acm.getStreamlineCounts();
		    
		    AnalyzeHeader.writeImage(sc, ahInt, outputRoot + "acm_sc", gzip);
		}
		
		if (!silent) {
	    	    System.err.println();
		}

		if (allROIs != null) {
		    
		    int seeds = 0;

		    for (int region = 0; region < numRegions; region++) {
			
			FreeFormROI roi = (FreeFormROI)allROIs.getRegion(region);

			Voxel[] seedVoxels = roi.getVoxels();

			seeds += seedVoxels.length;
		    }

		    int expectedTracts = seeds * iterations;

		    // it is probably a cause for concern if we have less than 1 tract per seed per iteration
		    // of course if we have multi-fiber voxels, we might have more.
 		    if (expectedTracts > tractCounter) {
 			logger.warning("Minimum expected number of tracts was " + seeds + " [seeds] * " + iterations + 
 				       " [iterations] = " + expectedTracts + ".\nActual number of tracts was " +
 				       tractCounter);
 		    }

		}
		

	    }
	    else if (!outputCP && outputRoot == null) {

		DataOutputStream dout = om.getOutputStream();
		
		// if no outputRoot is given, output to stdout or single file
		
		// just read tracts, filter and and output
		// no need to worry about tracts per voxel, etc
		int tractCounter = 0;

		if (outputTractsOOGL) {
		    dout.write(new String("LIST\n").getBytes());	    
		}

		if (!silent) {
	    	    System.err.println();
		}
    
		while (tractSource.more()) {
		
		    if (!silent) {		    
		        System.err.print("\rProcessing streamline " + (tractCounter+1));
		    }
	
		    TractCollection nextTC = new TractCollection(2, 100.0);
		    
		    nextTC.addTract(tractSource.nextTract());
		    
		    nextTC = filter.processTracts(nextTC);
		    
		    for (int i = 0; i < nextTC.numberOfTracts(); i++) {
			
			Tract t = nextTC.getTract(i);
			    
			if (outputTractsOOGL) {
			    
			    if (binaryOOGL) {
				t.writeOOGL_BinaryVECT(dout);	
			    }
			    else {
				dout.write(t.toOOGLVECT().getBytes());
			    }
				
			}
			else if (outputTractVoxels) {
				
			    VoxelList voxels = t.toVoxelList(xVoxelDim, yVoxelDim, zVoxelDim);
			    voxels.writeVoxelList(dout);
			    
			}
			else { // raw
			    t.writeRaw(dout);
			}
		    }
			
			
		    tractCounter++;
		}
		
		dout.close();
		
	    }
	    else {

		int seedCounter = 0;

		// read first Tract outside of loop, that way we can tell if a chunk of streamlines
		// come from the same seed and different PD or a different seed
		
		// don't filter here because we need this tract regardless of its waypoint
		// or exclusion status
		
		TractCollection nextTC = new TractCollection(2, 100.0);
		    
		nextTC.addTract(tractSource.nextTract());
		
		
		regions : for (int region = 0; region < numRegions; region++) {

		    FreeFormROI roi = (FreeFormROI)allROIs.getRegion(region);

		    Point3D[] seeds = roi.getSeedPoints();
		    Voxel[] seedVoxels = roi.getVoxels();
		
		    // number ROIs from 1 upwards to maintain correspondence to 
		    // values in seed file.
		    outputRegionID = region + 1; 
		
		    if (seedIndex == -1) {

			if (regionIndex > -1) {
			    if (region != regionIndex) {
				continue regions;
			    }
			}
		   
			if (!silent) { 
			    System.err.println("\nProcessing ROI " + outputRegionID + " of " + numRegions);
		        }
		    }
		    else {
			if (seeds.length + seedCounter - 1 < seedIndex) {
			    seedCounter += seeds.length;
			    continue regions;
			}
			else if (seedCounter > seedIndex) {
			    break regions;
			}
			else {
			    if (!silent) {	
			        System.err.println("\nProcessing ROI " + outputRegionID + " of " + numRegions);
			    }
	  	 	}
			
		    }
		    
		
		    if (outputCBS) {

			// ignore seedindex because it does not apply to outputCBS
			// an exception would have been thrown earlier if it was specified
	
			ConnectivitySegmentedImage[] cbsImages = new ConnectivitySegmentedImage[MAX_PDS];
			
			for (int sp = 0; sp < seeds.length; sp++) {
			 
			    if (!silent) {	   
			        System.err.print("\rProcessing seed " + (sp + 1) + " of " + seeds.length); 
			    }

			    // index of current PD
			    int pd = 0;
			    
			    Voxel spVoxel = seedVoxels[sp];
			    
			    VoxelList nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										       zVoxelDim);
			    
			    Voxel ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());
			    
			    
			    // a bunch of (iterations) tracts sharing the same SEED VOXEL are 
			    // interpreted as belonging to a different PD in the same seed
			    // this means we can only have one CP map per voxel.
			    
			    while (spVoxel.equals(ntSeedVoxel) && nextTC != null) {

				// initialize output for this PD 
				if (cbsImages[pd] == null) {
				    cbsImages[pd] = new ConnectivitySegmentedImage(roi, targetVol, 
										  xVoxelDim, yVoxelDim, zVoxelDim);
				    if (directionalCP) {
					cbsImages[pd].setDirectional(forwards);
				    }
				    
				    cbsImages[pd].setCountFirstEntry(countFirstEntry);
				}
				
				cbsImages[pd].processTracts(sp, filter.processTracts(nextTC));
				
				for (int i = 1; i < iterations; i++) {
				    TractCollection tc = filter.processTract(tractSource.nextTract());
				    cbsImages[pd].processTracts(sp, tc);
				}
				
				if (tractSource.more()) {
				    pd++;
				    
				    nextTC = new TractCollection(2, 100.0);
				
				    nextTC.addTract(tractSource.nextTract());
				
				    nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										     zVoxelDim);
				
				    ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());
				}
				else {
				    nextTC = null;
				    nextTractVoxels = null;
				    ntSeedVoxel = null;

				    if (sp < seeds.length - 1) {
					throw new 
					    LoggedException("No more tracts in input after processing seed " + 
							    (sp+1) + " of " + seeds.length);
				    }
				}
				
			    }
			    
			}
			
			
			// now output CBS
			for (int p = 0; p < MAX_PDS; p++) {
			    if (cbsImages[p] != null) {
				int vols = directionalCP ? 2 : 1;
				
				short[][][][] labelledSeeds = cbsImages[p].getSegmentedSeeds();
			    
				double[][][][] targetCPImage = null;
				
				int[][][][] targetSCImage =  null;
				
				if (normalizeCP) {
				    targetCPImage = cbsImages[p].getMaxTargetCP();
				}
				else {
				    targetSCImage = cbsImages[p].getMaxTargetSC();
				}
				
				
				for (int v = 0; v < vols; v++) {
				    
				    
				    // note we output index v not v+1
				    // this is because v is NOT a number, it is a boolean
				    // 0 if seedpd.dot(forwards) > 0
				    
				    AnalyzeHeader.writeImage(labelledSeeds[v], ahShort, 
							     outputRoot + "labels_" + (region+1) + "_" + 
							     (p+1) + "_" + v, gzip);
				    
				    if (normalizeCP) {
					AnalyzeHeader.writeImage(targetCPImage[v], ahFloat, 
								 outputRoot + "labelcp_" + (region+1) + "_" + 
								 (p+1) + "_" + v, gzip);
				    }
				    else {
					AnalyzeHeader.writeImage(targetSCImage[v], ahInt, 
								 outputRoot + "labelsc_" + (region+1) + "_" + 
								 (p+1) + "_" + v, gzip);
				    }
				    
				}
			    }

			}
			
		    }
		    else if (outputCP && targetVol != null) {

			for (int sp = 0; sp < seeds.length; sp++) {

			    if (seedIndex > -1) {
				if (seedCounter != seedIndex) {
				    seedCounter++;
				    continue;
				}
			    }
			    
			    seedCounter++;
			 
			    if (!silent) {	
			    	System.err.print("\rProcessing seed " + (sp + 1) + " of " + seeds.length); 
			    }		
			   
			    // index of current PD
			    int pd = 0;
		
			    Voxel spVoxel = seedVoxels[sp];
		    
			    VoxelList nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										       zVoxelDim);
		    
			    Voxel ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());

		    
			    while (spVoxel.equals(ntSeedVoxel) && nextTC != null) {

				TargetCP_Image cpImage = 
				    new TargetCP_Image(targetVol, xVoxelDim, yVoxelDim, zVoxelDim);

				if (directionalCP) {
				    cpImage.setDirectional(forwards);
				}
				cpImage.setCountFirstEntry(countFirstEntry);
			
				cpImage.processTracts(filter.processTracts(nextTC));

				for (int i = 1; i < iterations; i++) {
				    TractCollection tc = filter.processTract(tractSource.nextTract());
				    cpImage.processTracts(tc);
				}

				int vols = directionalCP ? 2 : 1;
			
				double[][][][] targetCPImage = null;
			
				int[][][][] targetSCImage =  null;
			
				if (normalizeCP) {
				    targetCPImage = cpImage.getConnectionProbabilities();
				}
				else {
				    targetSCImage = cpImage.getStreamlineCounts();
				}
			
				
				for (int v = 0; v < vols; v++) {
		
				    // note we output index v not v+1
				    // this is because v is NOT a number, it is a boolean
				    // 0 if seedpd.dot(forwards) > 0

				    if (normalizeCP) {
					AnalyzeHeader.writeImage(targetCPImage[v], ahFloat, 
								 outputRoot + (region+1) + "_" + (sp+1) + 
								 "_" + (pd+1) + "_" + v, gzip);
				    }
				    else {
					AnalyzeHeader.writeImage(targetSCImage[v], ahInt, 
								 outputRoot + (region+1) + "_" + 
								 (sp+1) + "_" + (pd+1) + "_" + v, gzip);
				    }
			    
				}
			
			
				pd++;
			
				if (tractSource.more()) {
				    nextTC = new TractCollection(2, 100.0);
			
				    nextTC.addTract(tractSource.nextTract());
			    
				    nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										     zVoxelDim);
			    
				    ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());
				}
				else {
				    nextTC = null;
				    nextTractVoxels = null;
				    ntSeedVoxel = null;

				    if (sp < seeds.length - 1) {
					throw new 
					    LoggedException("No more tracts in input after processing seed " + 
							    (sp+1) + " of " + seeds.length);
				    }

				}
			
			    }

		    
			}
		    }
		    else if (outputCP) { // no targets
			
			for (int sp = 0; sp < seeds.length; sp++) {
			    
			    if (seedIndex > -1) {
				if (seedCounter != seedIndex) {
				    seedCounter++;
				    continue;
				}
			    }
			    
			    seedCounter++;
				
			    if (!silent) {
			        System.err.print("\rProcessing seed " + (sp + 1) + " of " + seeds.length); 
			    }

			    // index of current PD
			    int pd = 0;
		
			    Voxel spVoxel = seedVoxels[sp];
		    
			    VoxelList nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										       zVoxelDim);
		    
			    Voxel ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());

		    
			    while (spVoxel.equals(ntSeedVoxel) && nextTC != null) {

				ConnectionProbabilityImage cpImage = 
				    new ConnectionProbabilityImage(xDataDim, yDataDim, zDataDim, 
								   xVoxelDim, yVoxelDim, zVoxelDim);
			
				cpImage.processTracts(filter.processTracts(nextTC));

				for (int i = 1; i < iterations; i++) {
				    TractCollection tc = filter.processTract(tractSource.nextTract());
				    cpImage.processTracts(tc);
				}

				// now output
				if (normalizeCP) {
				    double[][][] cp = cpImage.getConnectionProbabilities();

				    AnalyzeHeader.writeImage(cp, ahFloat, outputRoot + (region+1)
							     + "_" + (sp+1) + "_" + (pd+1), gzip);
				}
				else {
				    int[][][] sc = cpImage.getStreamlineCounts();
			    
				    AnalyzeHeader.writeImage(sc, ahInt, outputRoot + (region+1)
							     + "_" + (sp+1) + "_" + (pd+1), gzip);
				}
			
				
				if (tractSource.more()) {
				    nextTC = new TractCollection(2, 100.0);
			    
				    nextTC.addTract(tractSource.nextTract());
			    
				    nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										     zVoxelDim);
			    
				    ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());
				}
				else {
				    nextTC = null;
				    nextTractVoxels = null;
				    ntSeedVoxel = null;

				    if (sp < seeds.length - 1) {
					throw new 
					    LoggedException("No more tracts in input after processing seed " + 
							    (sp+1) + " of " + seeds.length);
				    }

				}
				
				pd++;

			    }
			} // for sp

		    } 
		    else { // output tracts, have an outputRoot
		    
			// if outputroot is specified it implies that the output is to be structured
			// by ROI, as in the old track

			FileOutputStream fout = null;
			DataOutputStream dout = null;
			
			if (outputTractsOOGL) {
			    
			    if (gzip) {
				fout = new FileOutputStream
				    (outputRoot + outputRegionID + ".oogl.gz");
				
				dout = new 
				    DataOutputStream(new GZIPOutputStream(fout, 1024*1024*16));
				
			    }
			    else {
				
				fout = new FileOutputStream(outputRoot + outputRegionID + ".oogl");
				
				dout = new 
				    DataOutputStream(new BufferedOutputStream(fout, 1024*1024*16));	
			    }
			    
			    dout.write(new String("LIST\n").getBytes());	    
			    
			} 
			else if (outputTractVoxels) {
			
			    if (gzip) {
				fout = new FileOutputStream(outputRoot + outputRegionID + ".Bshort.gz");
				dout = new DataOutputStream
				    (new GZIPOutputStream(fout, 1024*1024*16));
			    }
			    else {
				fout = new FileOutputStream(outputRoot + outputRegionID + ".Bshort");
				
				dout = new DataOutputStream
				    (new BufferedOutputStream(fout, 1024*1024*16));
			    }
			}
			else { // raw
			    
			    if (gzip) {
				fout = new FileOutputStream(outputRoot + outputRegionID + ".Bfloat.gz");
				dout = new DataOutputStream
				    (new GZIPOutputStream(fout, 1024*1024*16));
			    }
			    else {
				fout = new FileOutputStream(outputRoot + outputRegionID + ".Bfloat");
				dout = new DataOutputStream
				    (new BufferedOutputStream(fout, 1024*1024*16));
			    }
			}
			
		 	if (!silent) {		
			    System.err.println();
			}

			for (int sp = 0; sp < seeds.length; sp++) {
			    
			    if (seedIndex > -1) {
				if (seedCounter != seedIndex) {
				    seedCounter++;
				    continue;
				}
			    }

			    seedCounter++;
			    
			    if (!silent) {				    
			    	System.err.print("\rProcessing seed " + (sp + 1) + " of " + seeds.length); 
		    	    }

			    Voxel spVoxel = seedVoxels[sp];
			    
			    VoxelList nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, zVoxelDim);
		    
			    Voxel ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());

			    while (spVoxel.equals(ntSeedVoxel) && nextTC != null) {

				TractCollection filtered = filter.processTracts(nextTC);

				for (int tr = 0; tr < filtered.numberOfTracts(); tr++) {
				    
				    if (outputTractsOOGL) {
					dout.write(filtered.getTract(tr).toOOGLVECT().getBytes());
					
				    }
				    else if (outputTractVoxels) {
					
					filtered.getTract(tr).toVoxelList(xVoxelDim, yVoxelDim, 
									  zVoxelDim).writeVoxelList(dout);
					
				    }
				    else { // raw
					filtered.getTract(tr).writeRaw(dout);
				    }
				}
				
				
				if (tractSource.more()) {
				    nextTC = new TractCollection(2, 100.0);
				    
				    nextTC.addTract(tractSource.nextTract());
			    
				    nextTractVoxels = nextTC.getTract(0).toVoxelList(xVoxelDim, yVoxelDim, 
										     zVoxelDim);
				    
				    ntSeedVoxel = nextTractVoxels.getVoxel(nextTractVoxels.seedPointIndex());
				}
				else {
				    nextTC = null;
				    nextTractVoxels = null;
				    ntSeedVoxel = null;

				    if (sp < seeds.length - 1) {
					throw new 
					    LoggedException("No more tracts in input after processing seed " + 
							    (sp+1) + " of " + seeds.length);
				    }

				}
				
			    }
			}
		    	
			dout.close();
			
		    } // else (outputting tracts with outputroot specified)
		
		} // end for regions
		
	    } // end else (ie if outputCP || tracts by region)

	    if (tractSource.more()) {
		logger.warning("Tract processing finished but there are more Tracts in input. Check -seedfile and -iterations options.");
	    }
	  
	    if (!silent) {	
	    	System.err.println("\n");
	    }
	}
	catch(IOException e) {
	    throw new LoggedException(e);
	}
    
    }



}
