package apps;

import java.io.*;

import java.util.*;
import java.util.logging.Logger;

import data.*;
import imaging.*;
import misc.*;
import numerics.*;
import tools.*;



/**
 *
 * Produces a scalar-modulated RGB image, colored by the tensor eigenvector or ODF peak direction.
 * The application element reads command line options and outputs the RGB image.
 * <p>
 * This class also serves as an image object for PD_OrientationViewer.
 * 
 * @author Philip Cook
 * @version $Id: RGB_ScalarImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 *  
 */
public class RGB_ScalarImage {

    // 16 Mb output buffer
    public static final int BUFFERSIZE = 16777216;

    // array of vectors. Allow direct access for speed
    protected Vector3D[][][][] vectors = null;

    protected double[][][] scalarVol = null;

    protected double[][][] normScalarVol = null;

    // between 0.0 and 1.0
    private double[][][] red = null;
    private double[][][] green = null;
    private double[][][] blue = null;


    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.apps.RGB_ScalarImage");

    
    protected int xDataDim = 0;
    protected int yDataDim = 0;
    protected int zDataDim = 0;

    protected double xVoxelDim = 1.0;
    protected double yVoxelDim = 1.0;
    protected double zVoxelDim = 1.0;
     

    // range of scalar values for normScalarVol calculation. 
    // Anything > max set to 1, < min set to 0.
    protected double minScalarValue = 0.0;
    protected double maxScalarValue = 0.0;
    
    // gamma correction factor for the background scalar image
    private double gsGamma = 1.0;

    // gamma correction factor applied independently to R, G, B channels
    private double rgbGamma = 1.0;

   
    
    public RGB_ScalarImage(Vector3D[][][][] vecs, double[] voxelDims, double[][][] scalars, 
			   double minScalar, double maxScalar) {
	
	vectors = vecs;
	scalarVol = scalars;
	
	xDataDim = vecs.length;
	yDataDim = vecs[0].length;
	zDataDim = vecs[0][0].length;

	xVoxelDim = voxelDims[0];
	yVoxelDim = voxelDims[1];
	zVoxelDim = voxelDims[2];

	setNormalizedScalars(minScalar, maxScalar);		

	calculateRGB();
	
    }



//     /**
//      * The Analyze RGB format is not widely used, and software varies on the expected
//      * ordering. The ordering here is rgb for each voxel, one byte for each channel.
//      * The img file therefore contains red1, green1, blue1, red2... .
//      *
//      * @see imaging.AnalyzeHeader
//      */
//     public void writeAnalyze(String fileRoot) throws IOException {
	
// 	AnalyzeHeader ah = new AnalyzeHeader();

// 	ah.datatype = AnalyzeHeader.DT_RGB;
// 	ah.bitpix = 24;

// 	ah.width = (short)xDataDim;
// 	ah.height = (short)yDataDim;
// 	ah.depth = (short)zDataDim;

// 	ah.pixelWidth = 1.0f;
// 	ah.pixelHeight = 1.0f;
// 	ah.pixelDepth = 1.0f;

// 	int[][][] r = new int[xDataDim][yDataDim][zDataDim];
// 	int[][][] g = new int[xDataDim][yDataDim][zDataDim];	
// 	int[][][] b = new int[xDataDim][yDataDim][zDataDim];

// 	for (int k = 0; k < zDataDim; k++) { 
// 	    for (int j = 0; j < yDataDim; j++) {
// 		for (int i = 0; i < xDataDim; i++) {
// 		    int rgb = rgbIndex(i,j,k);

// 		    r[i][j][k] = rgb >> 16;
		    
// 		    g[i][j][k] = (rgb >> 8) & 0xff;

// 		    b[i][j][k] = rgb & 0xff;
		    
// 		}
// 	    }
// 	}

// 	AnalyzeHeader.writeImage(r, g, b, ah, fileRoot, false);
	
//     }


    /**
     * Writes this image, determines output type from extension (either .mha, .mhd or .vtk).
     */
    public void writeImage(String file) throws IOException {
	if (file.endsWith(".mha") || file.endsWith(".mhd")) {
	    writeMeta(file);
	}
	else if (file.endsWith(".vtk")) {
	    writeVTK(file);
	}
	else {
	    logger.warning("No recognized extension to file " + file + " - image not written");
	}
    }


    /**
     * Writes the RGB image in Meta I/O format. Data is binary rgb byte triplets for each voxel.
     *
     *
     */
    public void writeMeta(String fileRoot) throws IOException {

	String fileName = null;

	if (fileRoot.endsWith(".mha")) {
	    fileName = fileRoot;
	}
	else {
	    fileName = fileRoot + ".mha"; 
	}

	DataOutputStream dout = 
	    new DataOutputStream(new BufferedOutputStream(new FileOutputStream(fileName), BUFFERSIZE));
	
	
	MetaImageHeader mh = new MetaImageHeader();

	mh.dataType = MetaImageHeader.DataType.UCHAR;
	mh.channels = 3;

	mh.dimSize[0] = xDataDim;
	mh.dimSize[1] = yDataDim;
	mh.dimSize[2] = zDataDim;

	mh.spacing[0] = xVoxelDim;
	mh.spacing[1] = yVoxelDim;
	mh.spacing[2] = zVoxelDim;
	
	mh.dataFile = "LOCAL";

	mh.writeHeader(dout);
	
	// now write data
	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    int rgb = rgbIndex(i,j,k);
		    
		    dout.writeByte( (byte)(rgb >> 16) );
		    
		    dout.writeByte( (byte)( (rgb >> 8) & 0xff ) );

		    dout.writeByte( (byte)(rgb & 0xff) );
		    
		}
	    }
	}
	
	dout.close();
	
    }


    /**
     * Writes a VTK 2.0 (ie, not the new XML) file. The RGB triplets are stored as scalar values.
     *
     */
    public void writeVTK(String fileRoot) throws IOException {

	String fileName = null;

	if (fileRoot.endsWith(".vtk")) {
	    fileName = fileRoot;
	}
	else {
	    fileName = fileRoot + ".vtk"; 
	}

	DataOutputStream dout = 
	    new DataOutputStream(new BufferedOutputStream(new FileOutputStream(fileName), BUFFERSIZE));


	
	dout.write(new String("# vtk DataFile Version 2.0\n").getBytes("US-ASCII"));
	dout.write(new String("Camino RGB-scalar image\n").getBytes("US-ASCII"));
	dout.write(new String("BINARY\n").getBytes("US-ASCII"));
	dout.write(new String("DATASET STRUCTURED_POINTS\n").getBytes("US-ASCII"));

	String dataDimString = "DIMENSIONS " + xDataDim + " " + yDataDim + " " + zDataDim + "\n"; 

	dout.write(new String(dataDimString).getBytes("US-ASCII"));

	String voxelDimString = "SPACING " + xVoxelDim + " " + yVoxelDim + " " + zVoxelDim + "\n"; 

	dout.write(voxelDimString.getBytes("US-ASCII"));

	dout.write(new String("ORIGIN 0 0 0\n").getBytes("US-ASCII"));
	dout.write(new String("POINT_DATA " + xDataDim * yDataDim * zDataDim + "\n").getBytes("US-ASCII"));

	dout.write(new String("COLOR_SCALARS scalars 3\n").getBytes("US-ASCII"));

    	// now write data
	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    int rgb = rgbIndex(i,j,k);
		    
		    dout.writeByte( (byte)(rgb >> 16) );
		    
		    dout.writeByte( (byte)( (rgb >> 8) & 0xff ) );

		    dout.writeByte( (byte)(rgb & 0xff) );
		    
		}
	    }
	}
	
	dout.close();


    }


    /**
     * @return the 24-bit rgb triplet for the specified volumetric index.
     *
     */
    protected final int rgbIndex(int i, int j, int k) {


	// gamma first, then scale
	double r = Math.pow(red[i][j][k], rgbGamma);
	double g = Math.pow(green[i][j][k], rgbGamma);
	double b = Math.pow(blue[i][j][k], rgbGamma);
	
	double gsScalar = Math.pow(normScalarVol[i][j][k], gsGamma);
	
	int ri = 0;
	int gi = 0;
	int bi = 0;
	
	ri = (int)(255.0 * gsScalar * r);
	gi = (int)(255.0 * gsScalar * g);
	bi = (int)(255.0 * gsScalar * b);
    
	return bi + 256 * (gi + 256 * ri);

    }



    /**
     * Calculates scalar range over which grey levels are mapped. Anything less than the minimum
     * is set to black, anything greater than the maximum is set to white. 
     *
     * @param chop the fraction of voxels to exclude. The smallest and largest (chop * 100) % of
     * the scalar values will be mapped to black and white respectively.
     */
     void calculateScalarRange(double chop) {
	int counter = 0;

	double[] scalarFlat = new double[xDataDim * yDataDim * zDataDim];

	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
                   
		    scalarFlat[counter++] = scalarVol[i][j][k];
		    
		}
	    }
	}

	// calculate scalar range from data
	Arrays.sort(scalarFlat);
        
	int minIndex = (int)(chop * scalarFlat.length);

	int maxIndex = scalarFlat.length - minIndex - 1;
        
	minScalarValue = scalarFlat[minIndex];
	maxScalarValue = scalarFlat[maxIndex];

	if (maxScalarValue < Double.MAX_VALUE) {
	    // we're OK
	}
	else {
	    // NaN or Infinity
	    while (maxIndex >= 0 && !(maxScalarValue <= Double.MAX_VALUE)) {
		maxIndex--;
		maxScalarValue = scalarFlat[maxIndex];
	    }
	}


	if (minScalarValue > -1.0 * Double.MAX_VALUE) {
	    // we're OK
	}
	else {
	    // -Infinity
	    while (minIndex < (scalarFlat.length  - 1) && !(minScalarValue >= -1.0 * Double.MAX_VALUE)) {
		minIndex++;
		minScalarValue = scalarFlat[minIndex];
	    }
	}
	

    }


    /**
     * Sets scalar range from the range present in the data, after excluding outliers.
     * Anything less than the minimum
     * is set to black, anything greater than the maximum is set to white. 
     *
     * @param chop the fraction of voxels to exclude. The smallest and largest (chop * 100) % of
     * the scalar values will be mapped to black and white respectively.
     */
    public void setNormalizedScalars(double chop) {

	if ( !( (chop >= 0.0) && (chop < 1.0) ) ) {
	    throw new LoggedException("Fraction of scalar values to exclude was : " + chop);
	}

	calculateScalarRange(chop);
	setNormalizedScalars();
    }

    
    /**
     * Sets scalar range between the specified minimum (maps to black) and maximum 
     * (maps to white).
     * Values outside the range are clamped. If the two parameters are equal, the scalar range
     * is computed from the data.
     */
    public void setNormalizedScalars(double minScalar, double maxScalar) {

	// if the values are the same, we calculate from the data 
	if (minScalar == maxScalar) {
	    calculateScalarRange(0.005);
	}
	else {
	    minScalarValue = minScalar;
	    maxScalarValue = maxScalar;
	}

	setNormalizedScalars();
    }


    /**
     * Sets normalized scalars according to the current scalar range. 
     */
    private void setNormalizedScalars() {

	normScalarVol = new double[xDataDim][yDataDim][zDataDim];

	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    
		    if (scalarVol[i][j][k] < minScalarValue) {
			normScalarVol[i][j][k] = 0.0;
		    }
		    else if (scalarVol[i][j][k] > maxScalarValue) {
			normScalarVol[i][j][k] = 1.0;
		    }
		    else {
 			if ((maxScalarValue - minScalarValue) != 0) {
			    normScalarVol[i][j][k] = 
				(scalarVol[i][j][k] - minScalarValue) / (maxScalarValue - minScalarValue);   
 			}
 			else {
 			    normScalarVol[i][j][k] = 1.0;
 			}
 		    }
		    
		    
		}
	    }
	}
    }


    public static void main(String[] args) {

	CL_Initializer.inputDataType = "double";
        CL_Initializer.maxTensorComponents = 1;

	CL_Initializer.inputModel = "dteig";

	CL_Initializer.numPDsIO = -1;

        CL_Initializer.CL_init(args);
        
	// default output format is meta
	String outputFormat = "meta";

	// root.{mha, vtk} is written
	String outputRoot = "";

	String picoPDF = "bingham";

        int xDataDim = 0;
        int yDataDim = 0;
        int zDataDim = 0;

        double xVoxelDim = 0.0;
        double yVoxelDim = 0.0;
        double zVoxelDim = 0.0;

        double minScalar = 0.0;
        double maxScalar = 0.0;

        String scalarFile = null;


	double gsGamma = 1.0;
	double rgbGamma = 1.0;

	int eigIndex = 0;

        for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-outputroot")) {
                outputRoot = args[i+1];
                CL_Initializer.markAsParsed(i, 2);
		
	    }
	    if (args[i].equals("-outputformat")) {
                outputFormat = args[i+1];
                CL_Initializer.markAsParsed(i, 2);
	    }
            if (args[i].equals("-scalarrange")) { 
                minScalar = Double.parseDouble(args[i+1]);
                maxScalar = Double.parseDouble(args[i+2]);
                CL_Initializer.markAsParsed(i, 3);
            }
            if (args[i].equals("-scalarfile")) {
                scalarFile = args[i+1];
                CL_Initializer.markAsParsed(i, 2);
            }
            if (args[i].equals("-gsgamma")) {
		gsGamma = Double.parseDouble(args[i+1]);
                CL_Initializer.markAsParsed(i,2);
            }
            if (args[i].equals("-rgbgamma")) {
		rgbGamma = Double.parseDouble(args[i+1]);
                CL_Initializer.markAsParsed(i,2);
            }
	    if (args[i].equals("-e1")) {
                eigIndex = 0;
                CL_Initializer.markAsParsed(i);
            }
            if (args[i].equals("-e2")) {
                eigIndex = 1;
                CL_Initializer.markAsParsed(i);
            }
            if (args[i].equals("-e3")) {
                eigIndex = 2;
                CL_Initializer.markAsParsed(i);
            }
            if (args[i].equals("-pdf")) {
                picoPDF = args[i+1];
                CL_Initializer.markAsParsed(i, 2);
            }

        }


        CL_Initializer.checkParsing(args);


	if (CL_Initializer.numPDsIO < 0) {
	    if (CL_Initializer.inputModel.equals("pds")) {
		// default is 3 for sfpeaks
		CL_Initializer.numPDsIO = 3;
	    }
	    else {
		// default is 1 for everything else
		CL_Initializer.numPDsIO = 1;
	    }
	}

	xDataDim = CL_Initializer.dataDims[0];
	yDataDim = CL_Initializer.dataDims[1];
	zDataDim = CL_Initializer.dataDims[2];
	
	xVoxelDim = CL_Initializer.voxelDims[0];
	yVoxelDim = CL_Initializer.voxelDims[1];
	zVoxelDim = CL_Initializer.voxelDims[2];


	double[][][] scalarVol = null;


	if (scalarFile != null) {

	    ImageHeader ih = null;
	    
	    DataSource scalars = null;

	    if (ImageHeader.imageExists(scalarFile)) {

		try {
		    ih = ImageHeader.readHeader(scalarFile);
		}
		catch (IOException e) {
		    throw new LoggedException(e);

		}
		// get data dims, voxel dims, from header
		int[] dataDims = new int[] {xDataDim, yDataDim, zDataDim};
		double[] voxelDims = new double[] {xVoxelDim, yVoxelDim, zVoxelDim};
		
		ImageHeader.checkDims(scalarFile, dataDims, voxelDims);

		// if these weren't set by CL, they will be set by checkDims
	        xDataDim = dataDims[0];
	        yDataDim = dataDims[1];
	        zDataDim = dataDims[2];

	        xVoxelDim = Math.abs(voxelDims[0]);
	        yVoxelDim = Math.abs(voxelDims[1]);
	        zVoxelDim = Math.abs(voxelDims[2]);


		scalars = ih.getImageDataSource();
	    }
	    else {
		scalars = new VoxelOrderDataSource(scalarFile, 1, CL_Initializer.inputDataType);
	    }

	    scalarVol = new double[xDataDim][yDataDim][zDataDim];

	    
	    for (int k = 0; k < zDataDim; k++) { 
		for (int j = 0; j < yDataDim; j++) {
		    for (int i = 0; i < xDataDim; i++) {
			scalarVol[i][j][k] = scalars.nextVoxel()[0];
		    }
		}
	    }
	}

	RGB_ScalarImage image = null;

	if (CL_Initializer.inputModel.equals("dteig")) {
            VoxelOrderDataSource data = 
		new VoxelOrderDataSource(CL_Initializer.inputFile, 12 * CL_Initializer.maxTensorComponents,
					 CL_Initializer.inputDataType);
	    
	    image = imageFromTensorEigenSys(data, new int[] {xDataDim, yDataDim, zDataDim}, new double[] 
					    {xVoxelDim, yVoxelDim, zVoxelDim}, scalarVol,
					    minScalar, maxScalar, eigIndex);
	}
	else if (CL_Initializer.inputModel.equals("pds")) {

            VoxelOrderDataSource data = new VoxelOrderDataSource(CL_Initializer.inputFile, 
								 6 + 8 * CL_Initializer.numPDsIO, 
								 CL_Initializer.inputDataType);

	    image = imageFromSphFuncPDs(data, new int[] {xDataDim, yDataDim, zDataDim}, new double[] 
		{xVoxelDim, yVoxelDim, zVoxelDim}, scalarVol, minScalar, maxScalar);
	}
	else if (CL_Initializer.inputModel.equals("pico")) {

	    int paramsPerPD = 1;

	    if (picoPDF.equals("bingham")) {
		paramsPerPD = 2;
	    }
	    if (picoPDF.equals("acg")) {
		paramsPerPD = 3;
	    }


            VoxelOrderDataSource data = 
		new VoxelOrderDataSource(CL_Initializer.inputFile, 
					 1 + (10 + paramsPerPD) * CL_Initializer.numPDsIO, 
					 CL_Initializer.inputDataType);


	    image = RGB_ScalarImage.imageFromPICoPDFs(data, CL_Initializer.numPDsIO, paramsPerPD,
						      new int[] {xDataDim, yDataDim, zDataDim}, 
						      new double[] {xVoxelDim, yVoxelDim, zVoxelDim}, 
						      scalarVol, minScalar, maxScalar);

        }
	else if (CL_Initializer.inputModel.equals("ballstick")) {
            VoxelOrderDataSource data = 
		new VoxelOrderDataSource(CL_Initializer.inputFile, 7, CL_Initializer.inputDataType);


	    image = RGB_ScalarImage.imageFromBallStick(data, new int[] {xDataDim, yDataDim, zDataDim}, 
						       new double[] {xVoxelDim, yVoxelDim, zVoxelDim}, 
						       scalarVol, minScalar, maxScalar);

	}

        else {
            throw new misc.LoggedException("Unrecognized input model " + CL_Initializer.inputModel);
        }
	
	// apply gamma correction
	image.setScalarGamma(gsGamma);
	image.setRGB_Gamma(rgbGamma);
	
	try {

	    if (outputFormat.equals("meta")) {
		image.writeMeta(outputRoot);
	    }
	    else if (outputFormat.equals("vtk")) {
		image.writeVTK(outputRoot);
	    }
	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}
	
    }



    /**
     * Sets RGB values for display. The RGB value for each voxel is taken from the vectors.
     *
     */
    protected void calculateRGB() {
	
	if (red == null) {
	    red = new double[xDataDim][yDataDim][zDataDim];
	    green = new double[xDataDim][yDataDim][zDataDim];
	    blue = new double[xDataDim][yDataDim][zDataDim];
	}
	
	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    double r = Math.abs(vectors[i][j][k][0].x);
		    double g = Math.abs(vectors[i][j][k][0].y);
		    double b = Math.abs(vectors[i][j][k][0].z);

		    if (vectors[i][j][k].length > 1) {
			Vector3D average = 
			    new Vector3D(Math.abs(vectors[i][j][k][0].x) + Math.abs(vectors[i][j][k][1].x),
					 Math.abs(vectors[i][j][k][0].y) + Math.abs(vectors[i][j][k][1].y),
					 Math.abs(vectors[i][j][k][0].z) + 
					 Math.abs(vectors[i][j][k][1].z)).normalized();
			
			r = average.x;
			g = average.y;
			b = average.z;

		    }

		    red[i][j][k] = r;
		    green[i][j][k] = g;
		    blue[i][j][k] = b;
		    
		}
	    }
	}
	
    }


    /**
     * Read image from ODF peak input. The image has a maximum of two directions per voxel.
     *
     * @param scalars if null, then the Hessian trace of the first peak is used.
     */
    public static RGB_ScalarImage imageFromSphFuncPDs(VoxelOrderDataSource data, int[] dataDims, 
						      double[] voxelDims, double[][][] scalars, 
						      double minScalar, double maxScalar) {

	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];
	
	double[][][] hessianTr = new double[xDataDim][yDataDim][zDataDim];
	
	Vector3D[][][][] vecs = new Vector3D[xDataDim][yDataDim][zDataDim][];

	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    double[] voxel = data.nextVoxel();
                    
		    Vector3D v1 = null;
		    Vector3D v2 = null;

		    v1 = new Vector3D(voxel[6], voxel[7], voxel[8]);

                    hessianTr[i][j][k] = Math.abs(voxel[10] + voxel[13]);

		    if (hessianTr[i][j][k] > 0.0) {
			// it's OK
		    }
		    else {
			// negative Hessian, or NaN
			hessianTr[i][j][k] = 0.0;
		    }

                   
                    if (voxel[2] >= 2.0) {
			
			// get second peak
                        v2 = new Vector3D(voxel[14], voxel[15], voxel[16]);

                    }
		    
		    if (v2 == null) {
			vecs[i][j][k] = new Vector3D[] {v1};
		    }
		    else {
			vecs[i][j][k] = new Vector3D[] {v1, v2};
		    }

                }
	    }
	}


	if (scalars == null) {
	    scalars = hessianTr;
	}
	
	return new RGB_ScalarImage(vecs, voxelDims, scalars, minScalar, maxScalar);

    }


    /**
     * Read an image from PICoPDFs. 
     * The image has a maximum of two directions per voxel.
     *
     * @param maxPDs maximum number of PDs in the voxel. Only two will be displayed.
     * @param paramsPerPD scalar PICo parameters per PD.
     * @param scalars if null, then the sum of the concentration parameters is used. 
     */
    public static RGB_ScalarImage imageFromPICoPDFs(VoxelOrderDataSource data, int maxPDs, 
						    int paramsPerPD, int[] dataDims, 
						   double[] voxelDims, double[][][] scalars, double minScalar,
						   double maxScalar) {
	return imageFromPICoPDFs(data, maxPDs, paramsPerPD, dataDims, voxelDims, scalars, minScalar, maxScalar, 0);
    }

    /**
     * Read an image from PICoPDFs. 
     * The image has a maximum of two directions per voxel.
     *
     * @param maxPDs maximum number of PDs in the voxel. Only two will be displayed.
     * @param paramsPerPD scalar PICo parameters per PD.
     * @param scalars if null, then the sum of the concentration parameters is used. 
     * @param eigIndex which of the scatter matrix eigenvectors to use (0-2).
     */
    public static RGB_ScalarImage imageFromPICoPDFs(VoxelOrderDataSource data, int maxPDs, 
						    int paramsPerPD, int[] dataDims, 
						   double[] voxelDims, double[][][] scalars, double minScalar,
						   double maxScalar, int eigIndex) {
	
	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];

	Vector3D[][][][] vecs = new Vector3D[xDataDim][yDataDim][zDataDim][];
	
	double[][][] sumK = new double[xDataDim][yDataDim][zDataDim];
	
	
	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    
		    double[] voxel = data.nextVoxel();

		    int numPDs = (int)voxel[0];
		    
                    Vector3D v1 = new Vector3D(voxel[2 + 3*eigIndex], voxel[3 + 3*eigIndex], voxel[4 + 3*eigIndex]);
                    Vector3D v2 = null;

		    for (int p = 0; p < paramsPerPD; p++) {
			sumK[i][j][k] += Math.abs(voxel[11+p]);
		    }

		    if (numPDs >= 2) {		
			int start = 11 + paramsPerPD;
			
			v2 = new Vector3D(voxel[start + 1 + 3*eigIndex], voxel[start + 2 + 3*eigIndex], 
					  voxel[start + 3 + 3*eigIndex]);

			double tmp = 0.0;

			for (int p = 0; p < paramsPerPD; p++) {
			    tmp += Math.abs(voxel[start + 10 + p]);
			}
			
			sumK[i][j][k] = (sumK[i][j][k] + tmp) / 2.0;
                    }

		    if (v2 == null) {
			vecs[i][j][k] = new Vector3D[] {v1};
		    }
		    else {
			vecs[i][j][k] = new Vector3D[] {v1, v2};
		    }
		    
		}
	    }
	}
	
	if (scalars == null) {
	    scalars = sumK;
	}
	
	return new RGB_ScalarImage(vecs, voxelDims, scalars, minScalar, maxScalar);
	
    }


    /**
     * Read image from Ball and stick input. The image has a maximum of one direction per voxel.
     *
     * @param scalars if null, then the mixing fraction f is used.
     */
    public static RGB_ScalarImage imageFromBallStick(VoxelOrderDataSource data, int[] dataDims, 
						      double[] voxelDims, double[][][] scalars, 
						      double minScalar, double maxScalar) {

	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];
	
	double[][][] f = new double[xDataDim][yDataDim][zDataDim];
	
	Vector3D[][][][] vecs = new Vector3D[xDataDim][yDataDim][zDataDim][];

	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    double[] voxel = data.nextVoxel();
                    
		    Vector3D v1 = null;

		   vecs[i][j][k] = new Vector3D[] {new Vector3D(voxel[4], voxel[5], voxel[6])};

                    f[i][j][k] = Math.abs(voxel[3]);

		    if (f[i][j][k] >= 0.0) {
			// it's OK
		    }
		    else {
			f[i][j][k] = 0.0;
		    }

                }
	    }
	}


	if (scalars == null) {
	    scalars = f;
	}
	
	return new RGB_ScalarImage(vecs, voxelDims, scalars, minScalar, maxScalar);

    }


    /**
     * Read an image from tensor eigen system input. 
     * The image has a maximum of two directions per voxel.
     *
     * @param scalars if null, then the FA of the tensor (or mean FA of the first two tensors)
     * is used.
     */
    public static RGB_ScalarImage imageFromTensorEigenSys(VoxelOrderDataSource data, int[] dataDims, 
							  double[] voxelDims, 
							  double[][][] scalars, double minScalar,
							  double maxScalar) {
	return imageFromTensorEigenSys(data, dataDims, voxelDims, scalars, minScalar, maxScalar, 0);
    }



    /**
     * Read an image from tensor eigen system input. 
     * The image has a maximum of two directions per voxel.
     *
     * @param scalars if null, then the FA of the tensor (or mean FA of the first two tensors)
     * is used.
     * @param eigIndex 
     */
    public static RGB_ScalarImage imageFromTensorEigenSys(VoxelOrderDataSource data, int[] dataDims, 
							  double[] voxelDims, 
							  double[][][] scalars, double minScalar,
							  double maxScalar, int eigIndex) {
	
	int xDataDim = dataDims[0];
	int yDataDim = dataDims[1];
	int zDataDim = dataDims[2];

	Vector3D[][][][] vecs = new Vector3D[xDataDim][yDataDim][zDataDim][];
	
	double[][][] fa = new double[xDataDim][yDataDim][zDataDim];
	
	for (int k = 0; k < zDataDim; k++) { 
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    
		    double[] voxel = data.nextVoxel();
		    
                    Vector3D v1 = null;

		    switch (eigIndex) {

		    case 0 : v1 = new Vector3D(voxel[1], voxel[2], voxel[3]); break;
		    case 1 : v1 = new Vector3D(voxel[5], voxel[6], voxel[7]); break;
		    case 2 : v1 = new Vector3D(voxel[9], voxel[10], voxel[11]); break;
		    default : throw new LoggedException("Invalid eigenvector index " + eigIndex);
		    }
			
			

                    Vector3D v2 = null;

		    double lmean = (voxel[0] + voxel[4] + voxel[8]) / 3.0;

		    if (lmean > 0.0) {

			fa[i][j][k] = 
			    Math.sqrt(1.5 * ((voxel[0] - lmean) * (voxel[0] - lmean) + 
					     (voxel[4] - lmean) * (voxel[4] - lmean) + 
					     (voxel[8] - lmean) * (voxel[8] - lmean)) / 
				      (voxel[0] * voxel[0] + voxel[4] * voxel[4] + voxel[8] * voxel[8]));
		    }
		    
                    // positive scalars only
		    if ( fa[i][j][k] >= 0.0 && fa[i][j][k] < 1.0) {
			// Sensible value
		    }
		    else {
			if (fa[i][j][k] < 0.0) {
			    fa[i][j][k] = 0.0;
			}
			else if (fa[i][j][k] > 1.0) {
			    fa[i][j][k] = 1.0;
			}
		    }

                    if (voxel.length > 12) {
    
                        lmean = (voxel[12] + voxel[16] + voxel[20]) / 3.0;
			
                        if (lmean > 0.0) {
			    switch (eigIndex) {
				
			    case 0 : v2 = new Vector3D(voxel[13], voxel[14], voxel[15]); break;
			    case 1 : v2 = new Vector3D(voxel[17], voxel[18], voxel[19]); break;
			    case 2 : v2 = new Vector3D(voxel[21], voxel[22], voxel[23]); break;
			    default : throw new LoggedException("Invalid eigenvector index " + eigIndex);
			    }
                        
			    double tmp = 
				Math.sqrt(1.5 * ((voxel[12] - lmean) * (voxel[12] - lmean) + 
						 (voxel[16] - lmean) * (voxel[16] - lmean) + 
						 (voxel[20] - lmean) * (voxel[20] - lmean)) / 
					  (voxel[12] * voxel[12] + voxel[16] * voxel[16] + 
					   voxel[20] * voxel[20]));
                            
                            if (tmp < 0.0) {
                                tmp = 0.0;
                            }
                            else if (tmp > 1.0) {
                                tmp = 1.0;
                            }
			    
			    fa[i][j][k] = (fa[i][j][k] + tmp) / 2.0;

			}
			    
                    }

		    if (v2 == null) {
			vecs[i][j][k] = new Vector3D[] {v1};
		    }
		    else {
			vecs[i][j][k] = new Vector3D[] {v1, v2};
		    }
		    
		}
	    }
	}
	
	if (scalars == null) {
	    scalars = fa;
	}
	
	return new RGB_ScalarImage(vecs, voxelDims, scalars, minScalar, maxScalar);
	
    }



    /**
     * Sets the gamma constant <code>g</code> for the rgb component. For each color channel c, the 
     * image value is 255 * c^g, where c is between 0 and 1.
     *
     * If <code>g == 0.0</code>, the image is greyscale.
     *
     */
    public void setRGB_Gamma(double g) {
	
	if (g >= 0.0) {
	    rgbGamma = g;
	}
	else {
	    throw new IllegalArgumentException("Cannot use gamma value " + g);
	}
    }

    
    public double rgbGamma() {
	return rgbGamma;
    }


    /**
     * Sets the gamma constant <code>g</code> for the scalar component. For each scalar value s, the 
     * image value is 255 * s^g, where s is between 0 and 1.
     *
     *
     */
    public void setScalarGamma(double g) {
	
	if (g >= 0.0) {
	    gsGamma = g;
	}
	else {
	    throw new IllegalArgumentException("Cannot use gamma value " + g);
	}
    }

    public double scalarGamma() {
	return gsGamma;
    }



    public double[] getVoxelDims() {
	return new double[] {xVoxelDim, yVoxelDim, zVoxelDim};
    }

    
}
