package imaging;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;
import java.text.*;

import misc.*;
import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import tools.*;

/**
 *
 * This scheme is for scheme files that define the "B-matrix" - gradient directions
 * and b-values only. The main difference between this and SchemeV0 is that there is
 * no restriction on the order of measurements - zero measurements may appear anywhere.
 * <p>
 * The diffusion time is unknown but is assumed to be 29 ms, and |q| is scaled to produce
 * the desired b-value. There is no requirement for b to be constant across measurements.
 * <p>
 * 
 * @author Philip Cook
 * @version $$
 *  
 */
public class B_MatrixScheme implements Scheme {

    
    /**
     *
     * logging object.
     *
     */
    private static final Logger logger = Logger.getLogger("camino.imaging.B_MatrixScheme");

    
    /**
     * Gyromagnetic ratio of protons in water.
     */
    public static final double GAMMA = 2.6751987E8;


    /** Gradient directions, normalized to unity. */
    protected double[][] gDir;

    /** b-values. One per measurement, including zeros. */
    protected double[] bVal;

    /**
     * The wavenumber for each measurement. The modulus |q| is calculated such that |q|^2 * tau = b.
     * The gradient directions and b-values are given in the scheme file.
     */
    protected double[][] q;


    /**
     * The diffusion time, by default it is set to 29 ms. It is not defined in the scheme; 
     * it is used only to scale |q| such that |q|^2 * tau = b. This is the effective diffusion time,
     * not DELTA. 
     *
     */
    protected double tau = 0.029;


    /**
     * true for measurements at b=0, false otherwise
     */
    protected boolean[] zero;


    /**
     * Total number of measurements, including any ignored measurements.
     */
    protected int numMeas;


    /**
     * The number of zero measurements.
     */
    protected int numZeroMeas;


  

    /**
     * The mean non-zero radial wavenumber. 
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE parameters are here for compliance with the Scheme interface. 
     * 
     *
     */
    protected double meanNonZeroModQ = -1.0;


    /**
     * The number of measurements to ignore in each voxel.
     */
    protected int ignoreMeasurements;


    /**
     * separation of diffusion gradient blocks, default 40 ms.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|.
     * The PGSE parameters are here for compliance with the Scheme interface. 
     */
    private double DELTA = 0.04;
    
    /**
     * Duration of diffusion gradient blocks, default 33 ms.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE parameters are here for compliance with the Scheme interface. 
     */
    private double delta = 0.033;

    
    /**
     * Strength of gradients, default 0.022 T / m.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * Other PGSE parameters are here for compliance with the Scheme interface. 
     */
    private double modG = 0.022;

    
    /**
     * Time of first gradient block.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE parameters are here for compliance with the Scheme interface. 
     */
    private double halfP90 = 0.0;


    /**
     * Constructor that reads the imaging parameters from a text file.
     * 
     * @param filename The name of the file.
     */
    public B_MatrixScheme(String filename) {
        this(filename, 1.0, 1.0);
    }

    public int numDWMeasurements() {return numMeasurements()-numZeroMeasurements();};
    public boolean isIgnored(int j) {return false;};
    /**
     * Constructor that reads the imaging parameters from a text file.
     * 
     * @param filename The name of the file.
     * @param qScale a scaling factor to apply to each wavevector.
     * @param tauScale a scaling factor to apply to each wavevector.
     */
    public B_MatrixScheme(String filename, double qScale, double tauScale) {


        Vector<String> lines = new Vector<String>();

        try {

            Scanner fileScanner = new Scanner(new File(filename));

            // Read in the file line by line.
            fileScanner.useDelimiter("\r\n|\n");
        
            // Get first line.
            String l1 = fileScanner.next();
	    
	    String version = "VERSION: 2";

	    if (!version.equals(l1)) {
		throw new LoggedException("Attempted to create Version 2 scheme but scheme file " +
					  "is not version 2");
	    }

            // Store all the lines in a vector.
            while(fileScanner.hasNext()) {
		// ignore empty lines
		
		String next = fileScanner.next();

		if (next.length() > 0) {
		    lines.add(next);
		}
            }
            fileScanner.close();

        } catch(Exception e) {
            throw new LoggedException(e);
        }

	if (lines.size() == 0) {
            // infinite loop in inverters in this case
            throw new 
                LoggedException("Number of measurements in scheme file " + filename + " is 0");
        }

	// local variables used here, class variables assigned in init method
	int numMeas = lines.size();

	// these get normalized here
	double[][] gDir = new double[numMeas][3];

	double[] bVal = new double[numMeas];
	
        for (int i = 0; i < numMeas; i++) {
		
	    Scanner lineScanner = new Scanner(lines.elementAt(i));

	    gDir[i][0] = lineScanner.nextDouble();
	    gDir[i][1] = lineScanner.nextDouble();
	    gDir[i][2] = lineScanner.nextDouble();

	    double modGradDir = Math.sqrt(gDir[i][0] * gDir[i][0] + gDir[i][1] * gDir[i][1] + 
					  gDir[i][2] * gDir[i][2]);
	    
	    if (modGradDir > 0.0 && Math.abs(1.0 - modGradDir) > 1E-5) {
		gDir[i][0] /= modGradDir;
		gDir[i][1] /= modGradDir;
		gDir[i][2] /= modGradDir;
	    }

	    bVal[i] = lineScanner.nextDouble();
	    
	}

	init(gDir, bVal, qScale, tauScale);
   
    }


    /**
     * Constructor from gradient directions and b-values.
     * 
     * @param gDir the normalized gradient directions.
     * @param bVal b-values, one per gradient direction.
     * @param qScale a scaling factor to apply to each wavevector.
     * @param tauScale a scaling factor to apply to each wavevector.
     */
    public B_MatrixScheme(double[][] gDir, double[] bVal, double qScale, double tauScale) {
	init(gDir, bVal, qScale, tauScale);
    }


    /**
     * Constructor from gradient directions and b-values.
     * 
     * @param gDir the normalized gradient directions.
     * @param bVal b-values, one per gradient direction.
     * @param qScale a scaling factor to apply to each wavevector.
     * @param tauScale a scaling factor to apply to each wavevector.
     */
    private void init(double[][] gDir, double[] bVal, double qScale, double tauScale) {

	this.gDir = gDir;
	this.bVal = bVal;
	
	numMeas = gDir.length;
	numZeroMeas = 0;
	ignoreMeasurements = 0;
	
	zero = new boolean[numMeas];
	
	tau = tau * tauScale;
	
	q = new double[numMeas][3];

        for (int i = 0; i < numMeas; i++) {	
	    
	    // b = |q|^2 * tau
	    double modQ = qScale * Math.sqrt(bVal[i] / tau);
	    
	    if (modQ == 0.0) {
		numZeroMeas++;
		zero[i] = true;
	    }
	    
	    q[i][0] = gDir[i][0] * modQ;
	    q[i][1] = gDir[i][1] * modQ;
	    q[i][2] = gDir[i][2] * modQ;

	}
	
    
	// if command line arguments for delta, DELTA and G have been specified, overide
	// with commandline arguments
	// this overrides the non-zero b values in the scheme file but keeps the directions
        if(SimulationParams.sim_delta_set || SimulationParams.sim_DELTA_set || SimulationParams.sim_G_set){
            
            logger.warning("commandline values for delta ("+SimulationParams.sim_delta+
			   "s), DELTA ("+SimulationParams.sim_DELTA+"s) and G ("+SimulationParams.sim_G+
			   "Tm^-1) will override those specified in scheme file");
            
            double new_delta=SimulationParams.sim_delta;
            double new_DELTA=SimulationParams.sim_DELTA;
            double new_G= SimulationParams.sim_G;
            
            double newModq= GAMMA*new_DELTA*new_G*qScale;
            
            // set the parameters correctly
            delta=new_delta;
            modG= new_G;
            
            tau = (new_DELTA - new_delta / 3.0) * tauScale;
            
	    double newB = newModq * newModq * tau;

            for(int i=0; i<numMeas; i++){

		if (!zero[i]) {
		    bVal[i] = newB;
		}

		for(int j=0; j<3; j++){
		    if (!zero[i]) {
			q[i][j] = gDir[i][j] * newModq;
		    }
		}
	    }
            
        }


    }

    /**
     * Returns the number of measurements in each voxel.
     * 
     * @return The number of measurements in each voxel.
     */
    public int numMeasurements() {
        return numMeas - ignoreMeasurements;
    }


    /**
     * Returns the number of q=0 measurements in each voxel.
     * 
     * @return The number of q=0 measurements in each voxel.
     */
    public int numZeroMeasurements() {
        return numZeroMeas;
    }
    
    /**
     * Returns the number of ignored measurements in each voxel.
     * 
     * @return The number of ignored measurements in each voxel.
     */
    public int numIgnoredMeasurements() { return 0;}

    /**
     * Returns one wavenumber from the list.
     * 
     * @param i
     *            The index of the wavenumber to return.
     * 
     * @return The i-th wavenumber.
     */
    public double[] getQ(int i) {
        double[] qi = new double[q[0].length];
        for (int j = 0; j < qi.length; j++) {
            qi[j] = q[i][j];
        }

        return qi;
    }

 

    /**
     * Returns the mean non-zero radial wavenumber. Following SchemeV1, this includes 
     * any ignored measurements. SchemeV0 excludes ignored measurements when calculating
     * the mean.
     * 
     * @return The mean non-zero |q|.
     */
    public double getMeanNonZeroModQ() {

        if (meanNonZeroModQ != -1) {
            return meanNonZeroModQ;
        }

        meanNonZeroModQ = 0.0;

        for (int i = 0; i < numMeas; i++) {
            
	    if (!zero[i]) {
		
		double[] q = getQ(i);
		double modQ = 0.0;
		
		for (int j = 0; j < q.length; j++) {
		    modQ += q[j] * q[j];
		}
		
		modQ = Math.sqrt(modQ);
		meanNonZeroModQ += modQ;
	    }
        }

        meanNonZeroModQ /= (double) (numMeas - numZeroMeas);

        return meanNonZeroModQ;
    }


    /**
     * Returns the diffusion time for one measurement. Note, they are
     * all the same in this scheme class.
     * 
     * @param i The index of the measurement.
     *
     * @return The diffusion time.
     */
    public double getDiffusionTime(int i) {
        return tau;
    }


    /**
     *
     * Computes |q| on the fly for this measurement.
     *
     */
    public double getModQ(int i) {
        return Math.sqrt(q[i][0] * q[i][0] + q[i][1] * q[i][1] + q[i][2] * q[i][2]);
    }


    /**
     * Returns the diffusion times for all measurements. Note, they are
     * all the same in this scheme class.
     * 
     * @return The array diffusion times.
     */
    public double[] getDiffusionTimes() {
        double[] taus = new double[numMeas];
        for(int i=0; i<taus.length; i++) {
            taus[i] = tau;
        }
        return taus;
    }


    /**
     * Returns the diffusion times for all measurements with nonzero q. 
     * 
     * 
     * @return <code>taus</code>, array of diffusion times, where <code>taus[i]</code> corresponds
     * to element <code>i</code> in the array returned from <code>normalizeData</code>.
     */
    public double[] getNonZeroQ_DiffusionTimes() {
	double[] diffTimes = new double[numMeas - numZeroMeas];
	
        for(int i=0; i < diffTimes.length; i++) {
	    diffTimes[i] = tau;
	}
	
	return diffTimes;
    }


    /**
     * Sets the diffusion time for one measurement.  Note, they are
     * all the same for this scheme class, so this changes all of
     * them.
     * 
     * @param diffusionTime The new diffusion time.
     * 
     * @param i The index of the measurement.
     */
    public void setDiffusionTime(double diffusionTime, int i) {
        tau = diffusionTime;
    }


    /**
     * Returns the list of normalized non-zero qs. Each q is normalized by its modulus. This 
     * is the same behaviour as SchemeV1. SchemeV0 normalizes each q by the mean 
     * non-zero |q|. 
     *
     * @return <code>qHat</code>, array of vectors, where <code>qHat[i]</code> corresponds
     * to element <code>i</code> in the array returned from <code>normalizeData</code>.
     *
     */
    public double[][] getNormNonZeroQs() {

        // Get the list of wavenumbers.
        double[][] gNoZeros = new double[numMeas - numZeroMeas][3];
      	
	int nextG = 0;

	for(int i=0; i<numMeas; i++){
	    if (!zero[i]) {
		gNoZeros[nextG][0] = gDir[i][0];
		gNoZeros[nextG][1] = gDir[i][1];
		gNoZeros[nextG][2] = gDir[i][2];

		nextG++;
	    }
	}

	return gNoZeros;
    }


    /**
     * Returns the list of non-zero qs.
     */
    public double[][] getNonZeroQs() {

	// Get the list of wavenumbers.
        double[][] qNoZeros = new double[numMeas - numZeroMeas][3];
      
	
	int nextQ = 0;

	for(int i=0; i<numMeas; i++){
	    if (!zero[i]) {
		qNoZeros[nextQ][0] = q[i][0];
		qNoZeros[nextQ][1] = q[i][1];
		qNoZeros[nextQ][2] = q[i][2];
		
		nextQ++;
	    }
	}

	return qNoZeros;
    }


    /**
     * Doesn't actually do anything for this class.
     * 
     * @param i
     *            The number of measurements to ignore.
     */
    public void setIgnore(int i) {
	logger.warning("Ignoring measurements is only supported by Scheme V0");
	ignoreMeasurements = i;
    }


    /**
     * Returns the number of measurements in each voxel to ignore. These are
     * assumed to be the last ones in the voxel.
     * 
     * @return The number of measurements to ignore.
     */
    public int getIgnore() {
        return ignoreMeasurements;
    }


    /**
     * Normalizes a voxel set of measurements acquired with this
     * sequence by the geometric mean of the q=0 measurements.  If the
     * mean is zero, the normalization does nothing.
     * 
     * @param data Unnormalized set of measurements.
     * 
     * @return Normalized measurements, for all data where q != 0.
     */
    public double[] normalizeData(double[] data) {

        double normFac = geoMeanZeroMeas(data);
        if(normFac == 0) {
            logger.warning("Mean b=0 measurement is zero.  Cannot normalize.");
            normFac = 1.0;
        }

        double[] norm = new double[numMeas - numZeroMeas];
        int nextInd = 0;
        for(int i=0; i<numMeas; i++) {
            if(!zero[i]) {
                norm[nextInd] = data[i]/normFac;
                nextInd += 1;
            }
        }
        
        return norm;
    }


    /**
     * Computes the geometric mean of the q=0 measurements.
     * 
     * @param data Unnormalized set of measurements.
     * 
     * @return The geometric mean.
     */
    public double geoMeanZeroMeas(double[] data) {

	if(numZeroMeas == 0) {
            return 0.0;
	}
	
	double geo = 1.0;
	for(int i = 0; i < numMeas; i++) {
	    if(zero[i]) {
		geo = geo*data[i];
	    }
	}

        return Math.pow(geo, 1.0/(double)numZeroMeas);
    }


  
   
    /**
     * Gets the duration of gradient blocks for the i-th measurement.
     * Note: they are all the same in this scheme class.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     *
     * @param i The index of the measurement.
     *
     * @return delta
     */
    public double getDelta(int i) {
        return delta;
    }

    
    /**
     * Sets the duration of gradient blocks for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     * 
     * @param delta gradient block duration
     *
     * @param i The index of the measurement.
     */
    public void setDelta(double delta, int i) {
        this.delta = delta;
    }

    
    /**
     * Gets the separation of gradient block onsets for the i-th
     * measurement.  Note: they are all the same in this scheme class.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     *
     * @param i The index of the measurement.
     *
     * @return DELTA
     */
    public double getDELTA(int i) {
        return DELTA;
    }

    
    /**
     * Sets the separation of gradient block onsets for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE parameters are here for compliance with the Scheme interface. 
     *
     * @param delta gradient block separation
     *
     * @param i The index of the measurement.
     */
    public void setDELTA(double DELTA, int i) {
        this.DELTA = DELTA;
    }

    
    /** 
     * Gets the strength of the gradient for the i-th measurement.
     * Note: they are all the same in this scheme class.
     *
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE parameters are here for compliance with the Scheme interface. 
     * 
     * @param i The index of the measurement.
     *
     * @return gradient strength
     */
    public double getModG(int i) {
        return modG;
    }


    /**
     * Sets new value for the gradient strength for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     * 
     * @param modG gradient strength
     *
     * @param i The index of the measurement.
     */
    public void setModG(double modG, int i) {
        this.modG = modG;
    }

    
    /** 
     * Gets time of first gradient block onset for the i-th measurement.
     * Note: they are all the same in this scheme class.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     * 
     * @param i The index of the measurement.
     *
     * @return time of start of block
     */
    public double getHalfP90(int i) {
        return halfP90;
    }

    
    /** 
     * Sets time of first gradient block onset for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * <p>
     * NOTE: All PGSE parameters are undefined for this scheme unless passed on the 
     * command line. The default values will likely not be correct and will not produce
     * the correct |q|. 
     * The PGSE accessor methods are here for compliance with the Scheme interface. 
     * 
     * @param t1 block start time in ms
     *
     * @param i The index of the measurement.
     */
    public void setHalfP90(double halfP90, int i) {
        this.halfP90 = halfP90;
    }
    
    /**
     * returns the version number (2)
     * 
     * @return 2
     */
    public final int getVersion(){
    	return 2;
    }



    /**
     * Gets a scheme containing a subset of the measurements.
     *
     */
    public Scheme getSubsetScheme(int[] indices) {
	
	int size = indices.length;

	double[][] gSub = new double[size][3];

	double[] bValSub = new double[size];

	for (int i = 0; i < size; i++) {
	    gSub[i][0] = gDir[indices[i]][0];
	    gSub[i][1] = gDir[indices[i]][1];
	    gSub[i][2] = gDir[indices[i]][2];

	    bValSub[i] = bVal[indices[i]];
	}

	B_MatrixScheme subset = new B_MatrixScheme(gSub, bValSub, 1.0, 1.0);

	return subset;
    }
  
 
    /**
     * Negates the X component of the gradient directions
     */
    public void flipX() {
	for(int i=0;i< q.length;i++) {
	    gDir[i][0] = -gDir[i][0];
	    q[i][0]= -q[i][0];
	}
    }

    /**
     * Negates the Y component of the gradient directions
     */
    public void flipY() {
	for(int i=0;i< q.length;i++) {
	    gDir[i][0] = -gDir[i][1];
	    q[i][0]= -q[i][1];
	}
    }

    /**
     * Negates the Z component of the gradient directions
     */
    public void flipZ() {
	for(int i=0;i< q.length;i++) {
	    gDir[i][0] = -gDir[i][2];
	    q[i][0]= -q[i][2];
	}
    }

    /**
     * @param order the new order of the gradients
     *
     * Reorders the gradient directions
     */
    public void gradOrder(int [] order){
	//expects a 3 element array consisting of 0 (x dir), 1 (y dir) and 2 (z dir)

	double tempX=0;
	double tempY=0;
	double tempZ=0;
	for(int i=0;i< numMeas;i++) {
	    tempX = q[i][order[0]];
	    tempY = q[i][order[1]];
	    tempZ = q[i][order[2]];

	    q[i][0]=tempX;
	    q[i][1]=tempY;
	    q[i][2]=tempZ;

	    tempX = gDir[i][order[0]];
	    tempY = gDir[i][order[1]];
	    tempZ = gDir[i][order[2]];

	    gDir[i][0]=tempX;
	    gDir[i][1]=tempY;
	    gDir[i][2]=tempZ;
	}
    }

    /**
     * Returns the String representation of this object. Any ignored measurements
     * are included.
     *
     */
    public String toString() {

	DecimalFormat df = new DecimalFormat("   0.000000;  -0.000000");
	
	StringBuffer sb = new StringBuffer();

	sb.append("VERSION: 2");
	sb.append("\n");

	for (int i = 0; i < numMeas; i++) {
	    
	    sb.append(df.format(gDir[i][0]));
	    sb.append(df.format(gDir[i][1]));
	    sb.append(df.format(gDir[i][2]));
	    sb.append(df.format(bVal[i]));
	    sb.append("\n");
	}
	
	return sb.toString();

    }

    
}
