package imaging;

import java.util.*;
import java.util.logging.Logger;
import java.text.*;

import misc.*;
import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import tools.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * Defines an MR imaging sequence.  Note: this scheme class is
 * deprecated and should be replaced by SchemeV1 wherever possible.
 * 
 * <dd>
 * 
 * <dt>Description:
 * 
 * Contains all the parameters describing a diffusion imaging
 * sequence.  These scheme class assumes constant diffusion time,
 * delta and DELTA and is deprecated and should be replaced by
 * SchemeV1 wherever possible.
 * 
 * <dd>
 * 
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: SchemeV0.java,v 1.4 2009/03/27 01:26:41 bennett Exp $
 *  
 */
public class SchemeV0 implements Scheme {

    
    /**
     * logging object
     */
    private static final Logger logger = Logger.getLogger("camino.imaging.SchemeV0");

    
    /**
     * Gyromagnetic ratio of protons in water.
     */
    public static final double GAMMA = 2.6751987E8;


    private String schemeString = null;


    /**
     * The wavenumber for each measurement.
     */
    protected double[][] q;


    /**
     * The diffusion time.
     */
    protected double tau;


    /**
     * The number of zero measurements.
     */
    protected int M;


    /**
     * The mean non-zero radial wavenumber.
     */
    protected double meanNonZeroModQ = -1;


    /**
     * The number of measurements to ignore in each voxel.
     */
    protected int ignoreMeasurements;


    /**
     * duration of diffusion gradient blocks
     */
    private double delta= 0.034;
    
    
    /**
     * separation of diffusion gradient blocks
     */
    private double DELTA= 0.04;
    
    
    /**
     * strength of gradients
     */
    private double modG= 0.022;

    
    /**
     * time of first gradient block
     */
    private double halfP90 = 0.0;

    
    /**
     * Constructor from a list of wavenumbers and a diffusion time.
     * 
     * @param wavenumbers
     *            Nx3 matrix of wavenumbers.
     * 
     * @param diffusionTime
     *            The diffusion time.
     */
    public SchemeV0(double[][] wavenumbers, double diffusionTime) {
        init(wavenumbers, diffusionTime);
    }


    /**
     * Constructor given the number of zero and non-zero measurements, a fixed
     * |q| and a value for the diffusion time.
     * 
     * @param M
     *            The number of zero measurements.
     * 
     * @param N
     *            The number of non-zero measurements.
     * 
     * @param modQ
     *            |q|.
     * 
     * @param diffusionTime
     *            The diffusion time.
     */
    public SchemeV0(int M, int N, double modQ, double diffusionTime) {
        tau = diffusionTime;
        this.M = M;
        q = new double[N + M][3];

        for (int i = 0; i < M; i++) {
            for (int j = 0; j < 3; j++) {
                q[i][j] = 0.0;
            }
        }

        // Read in the directions.
        double[][] gradientDirections = SphericalPoints.getElecPointSet(N);

        // Copy them into the wavenumber array.
        for (int i = 0; i < N; i++) {
            for (int j = 0; j < 3; j++) {
                q[i + M][j] = gradientDirections[i][j] * modQ;
            }
        }
    }


    /**
     * Constructor that reads the imaging parameters from a text file.
     * 
     * @param filename The name of the file.
     */
    public SchemeV0(String filename) {
        this(filename, 1.0, 1.0);
    }


    /**
     * Constructor that reads the imaging parameters from a text file.
     * 
     * @param filename The name of the file.
     * @param qScale a scaling factor to apply to each wavevector.
     * @param tauScale a scaling factor to apply to each wavevector.
     */
    public SchemeV0(String filename, double qScale, double tauScale) {
        FileInput f = new FileInput(filename);
        double diffusionTime = f.readDouble() * tauScale;

        int numMeas = (int)f.readDouble();

        if (numMeas == 0) {
            // infinite loop in this case
            throw new 
                misc.LoggedException("Number of measurements in scheme file " + filename + " is 0");
        }


        double[][] wavenumbers = new double[numMeas][3];

        for (int i = 0; i < numMeas; i++) {
            for (int j = 0; j < 3; j++) {
                wavenumbers[i][j] = f.readDouble() * qScale;
            }
        }

        f.close();

        
        // if command line arguments for delta, DELTA and G have been specified, overide
        // with commandline arguments
        if(SimulationParams.sim_delta_set || SimulationParams.sim_DELTA_set || SimulationParams.sim_G_set){
            
            logger.warning("commandline values for delta ("+SimulationParams.sim_delta+
                    		"s), DELTA ("+SimulationParams.sim_DELTA+"s) and G ("+SimulationParams.sim_G+
                    		"Tm^-1) will override those specified in scheme file");
            
            double new_delta=SimulationParams.sim_delta;
            double new_DELTA=SimulationParams.sim_DELTA;
            double new_G= SimulationParams.sim_G;
            
            double newModq= GAMMA*new_DELTA*new_G*qScale;
            
            // set the parameters correctly
            delta=new_delta;
            DELTA= new_DELTA;
            modG= new_G;
            
            diffusionTime= new_DELTA*tauScale;
            
            for(int i=0; i<numMeas; i++){
                double modq=0.0;
                for(int j=0; j<3; j++){
                    modq+=wavenumbers[i][j]*wavenumbers[i][j];
                }
                modq=Math.sqrt(modq);
                
                if(modq!=0.0){
                    for(int j=0; j<3; j++){
                        wavenumbers[i][j]*=(newModq/modq);
                    }
                }
            }
        }
        
        
        init(wavenumbers, diffusionTime);
    }


    /**
     * Creates and initializes data structures from a list of wavenumbers and a
     * diffusion time. Called by the constructors.
     * 
     * @param wavenumbers
     *            Nx3 matrix of wavenumbers.
     * 
     * @param diffusionTime
     *            The diffusion time.
     */
    protected void init(double[][] wavenumbers, double diffusionTime) {
        q = wavenumbers;
        tau = diffusionTime;
        ignoreMeasurements = 0;

        // Count the zero measurements
        M = 0;
        while (M < q.length && q[M][0] == 0.0 && q[M][1] == 0.0 && q[M][2] == 0.0) {
            M++;
        }
    }


    /**
     * Returns the number of measurements in each voxel.
     * 
     * @return The number of measurements in each voxel.
     */
    public int numMeasurements() {
        return q.length - ignoreMeasurements;
    }


    /**
     * Returns the number of q=0 measurements in each voxel.
     * 
     * @return The number of q=0 measurements in each voxel.
     */
    public int numZeroMeasurements() {
        return M;
    }


    /**
     * Returns one wavenumber from the list.
     * 
     * @param i
     *            The index of the wavenumber to return.
     * 
     * @return The i-th wavenumber.
     */
    public double[] getQ(int i) {
        double[] qi = new double[q[0].length];
        for (int j = 0; j < qi.length; j++) {
            qi[j] = q[i][j];
        }

        return qi;
    }

 

    /**
     * Returns the mean non-zero radial wavenumber.
     * 
     * @return The mean non-zero |q|.
     */
    public double getMeanNonZeroModQ() {

        if (meanNonZeroModQ != -1) {
            return meanNonZeroModQ;
        }

        meanNonZeroModQ = 0.0;
        int N = numMeasurements();
        for (int i = M; i < N; i++) {
            double[] q = getQ(i);
            double modQ = 0.0;
            for (int j = 0; j < q.length; j++) {
                modQ += q[j] * q[j];
            }
            modQ = Math.sqrt(modQ);
            meanNonZeroModQ += modQ;
        }

        meanNonZeroModQ /= (double) (N - M);

        return meanNonZeroModQ;
    }


    /**
     * Returns the diffusion time for one measurement.  Note, they are
     * all the same in this scheme class.
     * 
     * @param i The index of the measurement.
     *
     * @return The diffusion time.
     */
    public double getDiffusionTime(int i) {
        return tau;
    }

    public double getModQ(int i) {
        return Math.sqrt(q[i][0] * q[i][0] + q[i][1] * q[i][1] + q[i][2] * q[i][2]);
    }


    /**
     * Returns the diffusion times for all measurement.  Note, they are
     * all the same in this scheme class.
     * 
     * @return The array diffusion times.
     */
    public double[] getDiffusionTimes() {
        double[] taus = new double[numMeasurements()];
        for(int i=0; i<taus.length; i++) {
            taus[i] = tau;
        }
        return taus;
    }


    /**
     * Returns the diffusion times for all measurements with nonzero q. 
     * 
     * 
     * @return <code>taus</code>, array of diffusion times, where <code>taus[i]</code> corresponds
     * to element <code>i</code> in the array returned from <code>normalizeData</code>.
     */
    public double[] getNonZeroQ_DiffusionTimes() {
        
	double[] taus = new double[numMeasurements() - M];
        
	for(int i=0; i < taus.length; i++) {
            taus[i] = tau;
        }
        return taus;
    }


    /**
     * Sets the diffusion time for one measurement.  Note, they are
     * all the same for this scheme class, so this changes all of
     * them.
     * 
     * @param diffusionTime The new diffusion time.
     * 
     * @param i The index of the measurement.
     */
    public void setDiffusionTime(double diffusionTime, int i) {
        tau = diffusionTime;
    }


    /**
     * Returns the list of normalized non-zero qs. Zero qs at the start of the
     * list are removed and the remaining qs are normalized by the mean non-zero
     * |q|.
     * 
     * @return {{q1x, q1y, q1z}, {q2x, q2y, q2z}, ..., {q(N-M)x, q(N-M)y,
     *         q(N-M)z}}
     */
    public double[][] getNormNonZeroQs() {

        int M = numZeroMeasurements();

        // Get the list of wavenumbers.
        double[][] qNoZeros = new double[numMeasurements() - M][3];
        double modSqSum = 0;
        for (int i = 0; i < qNoZeros.length; i++) {
            double[] q = getQ(i + M);
            qNoZeros[i][0] = q[0];
            qNoZeros[i][1] = q[1];
            qNoZeros[i][2] = q[2];

            modSqSum = modSqSum + q[0] * q[0] + q[1] * q[1] + q[2] * q[2];
        }

        // Normalize the wavenumbers by the average modulus.
        double meanMod = Math.sqrt(modSqSum / (double) qNoZeros.length);
        for (int i = 0; i < qNoZeros.length; i++) {
            qNoZeros[i][0] /= meanMod;
            qNoZeros[i][1] /= meanMod;
            qNoZeros[i][2] /= meanMod;
        }

        return qNoZeros;
    }


    /**
     * Returns the list of non-zero qs. Zero qs at the start of the
     * list are removed and the list contains the remaining qs.
     * 
     * @return {{q1x, q1y, q1z}, {q2x, q2y, q2z}, ..., {q(N-M)x, q(N-M)y,
     *         q(N-M)z}}
     */
    public double[][] getNonZeroQs() {

        int M = numZeroMeasurements();

        // Get the list of wavenumbers.
        double[][] qNoZeros = new double[numMeasurements() - M][3];
        double modSqSum = 0;
        for (int i = 0; i < qNoZeros.length; i++) {
            double[] q = getQ(i + M);
            qNoZeros[i][0] = q[0];
            qNoZeros[i][1] = q[1];
            qNoZeros[i][2] = q[2];
        }

        return qNoZeros;
    }


    /**
     * Sets the number of measurements in each voxel to ignore. These are
     * assumed to be the last ones in the voxel.
     * 
     * @param i
     *            The number of measurements to ignore.
     */
    public void setIgnore(int i) {
        ignoreMeasurements = i;
    }


    /**
     * Returns the number of measurements in each voxel to ignore. These are
     * assumed to be the last ones in the voxel.
     * 
     * @return The number of measurements to ignore.
     */
    public int getIgnore() {
        return ignoreMeasurements;
    }


    /**
     * Normalizes a voxel set of measurements acquired with this
     * sequence by the geometric mean of the q=0 measurements.  If the
     * mean is zero, the normalization does nothing.
     * 
     * @param data Unnormalized set of measurements.
     * 
     * @return Normalized measurements, for all data where q != 0.
     */
    public double[] normalizeData(double[] data) {

        // Find the geometric mean of the q=0 measurements.
        double gm = geoMeanZeroMeas(data);
        gm = (gm == 0)?1.0:gm;

        // Now do the normalization.
        double[] norm = new double[data.length - M];
        for (int i = 0; i < data.length - M; i++) {
            norm[i] = data[i + M] / gm;
        }

        return norm;
    }


    /**
     * Computes the geometric mean of the q=0 measurements.
     * 
     * @param data Unnormalized set of measurements.
     * 
     * @return The geometric mean.
     */
    public double geoMeanZeroMeas(double[] data) {

        if (M == 0) {
            return 1.0;
        }

        double gm = 1.0;
        for (int i = 0; i < M; i++) {
            gm *= data[i];
        }
        gm = Math.pow(gm, 1.0 / (double) M);

        return gm;
    }


    /**
     * Returns the SchemeV0 for an indexed sequence.
     * 
     * @param bmxIndex
     *            Index of the sequence to use.
     * 
     * @see ScannerSequences
     */
    public static SchemeV0 getSchemeV0(int bmxIndex) {
        return getSchemeV0(bmxIndex, 1.0, 1.0);
        
    }



    /**
     * Returns the SchemeV0 for an indexed sequence.
     * 
     * @param bmxIndex
     *            Index of the sequence to use.
     * 
     * @see ScannerSequences
     */
    public static SchemeV0 getSchemeV0(int bmxIndex, double qScale, 
                                                         double tauScale) {
        
        if (bmxIndex == 0) {
            // Jones 60 directions, 3 q=0.            
            double deltaT = 0.032;
            double DELTAT = 0.04 * tauScale;
            double modGT = 0.022;
            double modQ = GAMMA * deltaT * modGT * qScale;

            double[][] qHat = ScannerSequences.bMatToQ_Hats(ScannerSequences.ion3Z60Q);

            double[][] wavenumbers = new double[qHat.length][qHat[0].length];
            for (int i = 0; i < qHat.length; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ;
                }
            }

            SchemeV0 s = new SchemeV0(wavenumbers, DELTAT);
            s.setDelta(deltaT, 0);
            s.setModG(modGT, 0);

            return s;
        }

        else if (bmxIndex == 1) {
            // Jones 54 directions |q|=Q, 6 q=0.


            double deltaT = 0.034;
            double DELTAT = 0.04 * tauScale;
            double modGT = 0.022;
            double modQ = GAMMA * deltaT * modGT * qScale;

            double[][] qHat = ScannerSequences.bMatToQ_Hats(ScannerSequences.ion6Z54Q);

            double[][] wavenumbers = new double[qHat.length][qHat[0].length];
            for (int i = 0; i < qHat.length; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ;
                }
            }

            SchemeV0 s = new SchemeV0(wavenumbers, DELTAT);
            s.setDelta(deltaT, 0);
            s.setModG(modGT, 0);

            return s;
        }

        else if (bmxIndex == 6 || bmxIndex == 7) {
            // Isotropic 54 |q|=Q, 6 q=0, 6 |q|=Q/2.

            double deltaT = 0.034;
            double DELTAT = 0.04 * tauScale;
            double modGT = 0.022;
            double modQ = GAMMA * deltaT * modGT * qScale;

            double[][] qHat = ScannerSequences
                    .bMatToQ_Hats(ScannerSequences.ion6Z54Q_Iso);

            double[][] wavenumbers = new double[qHat.length][qHat[0].length];
            for (int i = 0; i < qHat.length - 6; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ;
                }
            }
            for (int i = qHat.length - 6; i < qHat.length; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ / 2.0;
                }
            }

            SchemeV0 s = new SchemeV0(wavenumbers, DELTAT);

            // Index 7 means ignore the low |q| measurements.
            if (bmxIndex == 7) {
                s.setIgnore(6);
            }

            s.setDelta(deltaT, 0);
            s.setModG(modGT, 0);            

            return s;

        }

        else if (bmxIndex == 8) {
            // Isotropic 46 |q|=Q, 2 q=0.

            double deltaT = 0.034;
            double DELTAT = 0.04 * tauScale;
            double modGT = 0.022;
            double modQ = GAMMA * deltaT * modGT * qScale;

            double[][] qHat = ScannerSequences.bMatToQ_Hats(ScannerSequences.sghms2Z46Q);

            double[][] wavenumbers = new double[qHat.length][qHat[0].length];
            for (int i = 0; i < qHat.length; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ;
                }
            }

            SchemeV0 s = new SchemeV0(wavenumbers, DELTAT);
            s.setDelta(deltaT, 0);
            s.setModG(modGT, 0);

            return s;

        }

        // The Manchester data, with TE encoded in the index.
        // The index is 9XXX, where XXX is TE in milliseconds.
        else if (bmxIndex > 9000 && bmxIndex < 10000) {
            // N = 61, M = 8.

            double TE = (double) (bmxIndex - 9000) / 1000;

            // Model parameters computed as in
            // Research/Documents/DiffusionMRI/VoxClassTE/TE_AndB.nb.
            double K1 = 0.01247487;
            double A1 = -0.00361185;
            double modGT = 0.0419707;

            double deltaT = TE / 2.0 - K1;
            double DELTAT = (TE / 2.0 - A1 - K1 / 3.0) * tauScale;


            double modQ = GAMMA * deltaT * modGT * qScale;
            double[][] qHat = ScannerSequences.mancM8N61;

            double[][] wavenumbers = new double[qHat.length][qHat[0].length];
            for (int i = 0; i < qHat.length; i++) {
                for (int j = 0; j < qHat[0].length; j++) {
                    wavenumbers[i][j] = qHat[i][j] * modQ;
                }
            }

            SchemeV0 s = new SchemeV0(wavenumbers, DELTAT);
            s.setDelta(deltaT, 0);
            s.setModG(modGT, 0);

            return s;

        }

        throw new RuntimeException("Did not recognize b-matrix index.(bmxIndex="
                + bmxIndex + "). Did you forget the scheme file?");
    }


    /**
     * Gets the duration of gradient blocks for the i-th measurement.
     * Note: they are all the same in this scheme class.
     *
     * @param i The index of the measurement.
     *
     * @return delta
     */
    public double getDelta(int i) {
        return delta;
    }

    
    /**
     * Sets the duration of gradient blocks for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * 
     * @param delta gradient block duration
     *
     * @param i The index of the measurement.
     */
    public void setDelta(double delta, int i) {
        this.delta = delta;
    }

    
    /**
     * Gets the separation of gradient block onsets for the i-th
     * measurement.  Note: they are all the same in this scheme class.
     *
     * @param i The index of the measurement.
     *
     * @return DELTA
     */
    public double getDELTA(int i) {
        return DELTA;
    }

    
    /**
     * Sets the separation of gradient block onsets for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * 
     * @param delta gradient block separation
     *
     * @param i The index of the measurement.
     */
    public void setDELTA(double DELTA, int i) {
        this.DELTA = DELTA;
    }

    
    /** 
     * Gets the strength of the gradient for the i-th measurement.
     * Note: they are all the same in this scheme class.
     * 
     * @param i The index of the measurement.
     *
     * @return gradient strength
     */
    public double getModG(int i) {
        return modG;
    }


    /**
     * Sets new value for the gradient strength for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * 
     * @param modG gradient strength
     *
     * @param i The index of the measurement.
     */
    public void setModG(double modG, int i) {
        this.modG = modG;
    }

    
    /** 
     * Gets time of first gradient block onset for the i-th measurement.
     * Note: they are all the same in this scheme class.
     * 
     * @param i The index of the measurement.
     *
     * @return time of start of block
     */
    public double getHalfP90(int i) {
        return halfP90;
    }

    
    /** 
     * Sets time of first gradient block onset for the i-th measurement.
     * Note: they are all the same for this scheme class and this
     * method changes them all.
     * 
     * @param t1 block start time in ms
     *
     * @param i The index of the measurement.
     */
    public void setHalfP90(double halfP90, int i) {
        this.halfP90 = halfP90;
    }
    
    /**
     * returns the version number (zero)
     * 
     * @return 0
     */
    public final int getVersion(){
    	return 0;
    }


    /**
     * Gets a scheme containing a subset of the measurements.
     *
     */
    public Scheme getSubsetScheme(int[] indices) {
	
	int size = indices.length;

	double[][] qSub = new double[size][3];

	for (int i = 0; i < size; i++) {
	    qSub[i][0] = q[indices[i]][0];
	    qSub[i][1] = q[indices[i]][1];
	    qSub[i][2] = q[indices[i]][2];
	}

	SchemeV0 subset = new SchemeV0(qSub, tau);

	return subset;
    }
   
    /**
     * Negates the X component of the gradient directions
     */
    public void flipX() {
	for(int i=0;i< q.length;i++) {
	    q[i][0]=-1*q[i][0];
	}
    }

    /**
     * Negates the Y component of the gradient directions
     */
    public void flipY() {
	for(int i=0;i< q.length;i++) {
	    q[i][1]=-1*q[i][1];
	}
    }

    /**
     * Negates the Z component of the gradient directions
     */
    public void flipZ() {
	for(int i=0;i< q.length;i++) {
	    q[i][2]=-1*q[i][2];
	}
    }

    /**
     * @param order the new order of the gradients
     *
     * Reorders the gradient directions
     */
    public void gradOrder(int [] order){
	//expects a 3 element array consisting of 0 (x dir), 1 (y dir) and 2 (z dir)

	double tempX=0;
	double tempY=0;
	double tempZ=0;
	for(int i=0;i< q.length;i++) {
	    tempX = q[i][order[0]];
	    tempY = q[i][order[1]];
	    tempZ = q[i][order[2]];

	    q[i][0]=tempX;
	    q[i][1]=tempY;
	    q[i][2]=tempZ;
	}
    }

    /**
     * Returns the String representation of this object. Any ignored measurements
     * are included.
     *
     */
    public String toString() {
	DecimalFormat df = new DecimalFormat("0.000000");
	
	StringBuffer sb = new StringBuffer();

	sb.append(df.format(tau));
	sb.append("\n");
	sb.append(q.length);
	sb.append("\n");

	for (int i = 0; i < q.length; i++) {
	    sb.append(df.format(q[i][0]));
	    sb.append("\n");
	    sb.append(df.format(q[i][1]));
	    sb.append("\n");
	    sb.append(df.format(q[i][2]));
	    sb.append("\n");
	}
	
	return sb.toString();
    }

    public int numDWMeasurements() {return numMeasurements()-numZeroMeasurements();};
    public boolean isIgnored(int j) {return false;};
    /**
     * Returns the number of ignored measurements in each voxel.
     * 
     * @return The number of ignored measurements in each voxel.
     */
    public int numIgnoredMeasurements() { return ignoreMeasurements;}
    
}
