package inverters;

import imaging.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Fits the diffusion tensor.
 * 
 * <dt>Description:
 * 
 * <dd>Uses the algebraic expression for the diffusion tensor in terms of the
 * wavenumbers and the measurements.
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: AlgebraicDT_Inversion.java,v 1.2 2005/08/18 11:07:44 ucacmgh
 *          Exp $
 *  
 */
public class AlgebraicDT_Inversion extends DT_Inversion {

    /**
     * To do the algebraic inversion we need hat{q} hat{q}^T for each
     * wavenumber. This array stores them.
     */
    protected double[][] qHatSq;

    /**
     * This is a normalization constant that only needs to be computed once.
     */
    protected double sumB;

    /**
     * The constructor requires the details of the sequence used to generate the
     * data that will be processed.
     * 
     * @param imParams
     *            The imaging sequence parameters.
     */
    public AlgebraicDT_Inversion(Scheme imParams) {
        ip = imParams;

        // Set up the array of q q^T components.
        int M = ip.numZeroMeasurements();
        qHatSq = new double[ip.numMeasurements() - M - ip.numZeroMeasurements()][6];

        // We also compute the normalization used in the algebraic fit.
        sumB = 0.0;
        for (int i = 0; i < qHatSq.length; i++) {
            double[] q = ip.getQ(i + M);
            double modQ_Sq = q[0] * q[0] + q[1] * q[1] + q[2] * q[2];
            qHatSq[i][0] = q[0] * q[0] / modQ_Sq;
            qHatSq[i][1] = q[0] * q[1] / modQ_Sq;
            qHatSq[i][2] = q[0] * q[2] / modQ_Sq;
            qHatSq[i][3] = q[1] * q[1] / modQ_Sq;
            qHatSq[i][4] = q[1] * q[2] / modQ_Sq;
            qHatSq[i][5] = q[2] * q[2] / modQ_Sq;

            sumB += ip.getDiffusionTime(i + M) * modQ_Sq;
        }

    }

    /**
     * Fits the diffusion tensor algebraically.
     * 
     * @param data
     *            The MRI data.
     * 
     * @return {exitcode, ln A^\star(0), Dxx, Dxy, Dxz, Dyy, Dyz, Dzz}
     */
    public double[] invert(double[] data) {

        // Get the normalized data.
        double[] normData = ip.normalizeData(data);

        // Do algebraic fitting.
        double[] K = new double[6];
        for (int i = 0; i < 6; i++) {
            K[i] = 0.0;
        }

        for (int i = 0; i < qHatSq.length; i++) {
            double lA = -Math.log(normData[i]);
            for (int j = 0; j < 6; j++) {
                K[j] += lA * qHatSq[i][j];
            }
        }

        for (int i = 0; i < 6; i++) {
            K[i] *= 15.0 / (2.0 * sumB);
        }

        double trK = K[0] + K[3] + K[5];

        K[0] -= trK / 5.0;
        K[3] -= trK / 5.0;
        K[5] -= trK / 5.0;

        // Construct the array to return. The first element is
        // the exit code. The second element is ln A^\star(0).
        double[] res = new double[8];
        // This version can't go wrong.
        res[0] = 0.0;
        res[1] = Math.log(ip.geoMeanZeroMeas(data));
        for (int i = 0; i < K.length; i++) {
            res[i + 2] = K[i];
        }

        return res;
    }

}
