/* DiffusionSimulation.java created on 25-Nov-2005
 * (simulation)
 * 
 * author: matt (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation;

import java.io.*;
import java.util.Random;
import java.util.logging.Logger;

import numerics.MTRandom;

import misc.LoggedException;

import simulation.dynamics.StepGenerator;
import simulation.dynamics.StepGeneratorFactory;
import simulation.dynamics.Walker;
import simulation.geometry.SubstrateFactory;
import simulation.geometry.SquashyCylinder;
import simulation.geometry.SquashyInflammationSubstrate;
import simulation.geometry.Substrate;
import simulation.measurement.ScanFactory;
import simulation.measurement.SyntheticScan;
import tools.CL_Initializer;

import imaging.Scheme;
import imaging.SchemeV0;
import data.DataSource;
import data.DataSourceException;
import data.OutputManager;

/**
 * top level class in diffusion simulation. This implements the DataSource
 * interface and contains the main monte carlo loop over all walkers and
 * timesteps, calling the step generator and the geometry/substrate objects
 * to amend stpes.
 *
 * 
 * it also uses the synthetic scan to assemble noisy signals and multiple 
 * voxels-worth of readings.
 *
 * @author matt
 *
 */
public class DiffusionSimulation implements DataSource {

	/** file writer for trajectories */
	private final DataOutputStream trajWriter;
    
    /** logging obsject */
    private Logger logger= Logger.getLogger(this.getClass().getName());
    
    /** the dimensionality of the simulation. 
     * this is the central value from which all others are taken */
    public static final int D=3; 

    /** the self-diffusion constant of water at room temperature */
    public static final double DIFF_CONST = CL_Initializer.DIFF_CONST;
    
    /** the membrane transition probability */
    private final double p;

    /** random number generator */
    //private final MTRandom twister = new MTRandom((1321839371<<32)|(129375817));
    private MTRandom twister = new MTRandom(CL_Initializer.seed+189);
    
    /** diffusion simulation parameters */
    private final SimulationParams simParams;
    
    /** synthetc scan parameters */
    private SyntheticScan synthScan;
       
    /** array of walkers */
    private final Walker[] walker;
    
    /** diffusion substrate */
    private Substrate substrate;
    
    /** likely border in which to clone bits of substrate */
    public static double border;
    
    /** object for generating random steps in the walk */
    private StepGenerator stepGenerator;
    
    /** number of voxels to generate */
    private final int numVoxels;
    
    /** index of the current voxel */
    private int voxel=0;
    
    /** flag indicating if we want a separate simulation for each voxel or  not */
    private final boolean separateRuns = SimulationParams.sim_separate_runs;
    
    /** number of timesteps in simulation */
    private final int tmax;
    
    /** time increment associated with a timestep */
    private final double dt;
    
    /** duration of simulation */
    private final double duration;
    
    /** counter for the number of times the substrate initialiser function is called */
    public static int calls=-1;
    
    /**
     * are we selecting only one iteration to output?
     * -1 if not, otherwise equal to iteration number
     */
    private final int onlyRun= SimulationParams.sim_onlyRun;
    
    

    
    /** debug file writers */
    //private BufferedWriter intraWriter=null;
    //private BufferedWriter extraWriter=null;
    
    public DiffusionSimulation(SimulationParams simParams, Scheme imParams){
        
        this.simParams=simParams;

        this.stepGenerator=StepGeneratorFactory.getStepGenerator(simParams);
        
        DiffusionSimulation.border= stepGenerator.getBorder();
        
        this.substrate=SubstrateFactory.getSubstrate(simParams.getGeometryType(), simParams.getGeometryParams(), simParams);
    
        this.numVoxels=CL_Initializer.numVoxels;
    
        this.p=simParams.getP();
               
        this.walker=new Walker[simParams.getN_walkers()];
        
        this.synthScan=ScanFactory.getMeasurementModule(simParams, imParams, substrate, walker);

        this.tmax=simParams.getTmax();
        
        this.dt=simParams.getDt();

        this.duration= SimulationParams.duration;
        
        this.trajWriter= null;
        
        
        logger.info("running simulation: "+walker.length+" walkers, "+simParams.getTmax()+" timesteps, p= "+p);
        logger.info("delta= "+imParams.getDelta(0)+" DELTA= "+imParams.getDiffusionTime(0)+ "G= "+imParams.getModG(0));
    }

    
    public DiffusionSimulation(SimulationParams simParams){
        
        this.simParams=simParams;

        this.stepGenerator=StepGeneratorFactory.getStepGenerator(simParams);
        
        DiffusionSimulation.border= stepGenerator.getBorder();
        
        this.substrate=SubstrateFactory.getSubstrate(simParams.getGeometryType(), simParams.getGeometryParams(), simParams);
    
        this.numVoxels=CL_Initializer.numVoxels;
    
        this.p=simParams.getP();
               
        this.walker=new Walker[simParams.getN_walkers()];
        
        this.synthScan=null;

        this.tmax=simParams.getTmax();
        
        this.dt=simParams.getDt();
        this.duration= simParams.duration;
        
        /** initalise file output -- data output writer wrapped round a
         * 							 buffered stream, wrapped round a file stream!
         */
        try{
        	FileOutputStream fstream = null;
        	try{
        		String fname= SimulationParams.trajFile;
        		fstream= new FileOutputStream(fname);
        	}
        	catch(Exception e){
        		throw new LoggedException(e);
        	}
        	
        	this.trajWriter= new DataOutputStream(new BufferedOutputStream(fstream, simParams.buffsize));
        }
        catch(Exception ioe){
        	throw new LoggedException(ioe);
        }
    
        
        
        logger.info("running simulation: "+walker.length+" walkers, "+simParams.getTmax()+" timesteps, p= "+p);
        logger.info("no scheme used, trajectories created instead. duration= "+duration);
    }
    
    
    /** 
     * sets all walkers in initial conditions
     */
    private void initialiseWalkers(){	

    	
        if(simParams.getInitialConditions()==SimulationParams.SPIKE){
            // initially delta-peaked at centre of substrate
            double midway=substrate.getPeakCoord();
            
            double[] midPoint=new double[D];
            
            
            for(int i=0; i<D; i++){
                midPoint[i]=midway;
            }
            
            for(int i=0; i<walker.length; i++){
                walker[i]=new Walker(midPoint, stepGenerator, substrate, synthScan, trajWriter);
            }
        }
        else if(simParams.getInitialConditions()==SimulationParams.UNIFORM){
            // initially uniformly distributed across substrate
            double[] substrateSize=substrate.getSubstrateSize();
            
            double[] bottomLeft= new double[]{Double.MAX_VALUE, Double.MAX_VALUE, Double.MAX_VALUE};
            double[] topRight= new double[]{0.0, 0.0, 0.0};
            
            for(int i=0; i<walker.length; i++){
                double[] r0= new double[D];
                for(int j=0; j<D; j++){
                    r0[j]= twister.nextDouble()*substrateSize[j];
                    
                    // get top right and bottom left corners of envloping cube
                    if(r0[j]<bottomLeft[j]){
                    	bottomLeft[j]=r0[j];
                    }
                    if(r0[j]>topRight[j]){
                    	topRight[j]=r0[j];
                    }
                }
                
                walker[i]= new Walker(r0, stepGenerator, substrate, synthScan, trajWriter);
            }
        }
        else if(simParams.getInitialConditions()==SimulationParams.INTRACELLULAR){
            // initially uniformly distributed across substrate
            double[] substrateSize=substrate.getSubstrateSize();
            
            boolean extracellular=true;
            
            double[] bottomLeft= new double[]{Double.MAX_VALUE, Double.MAX_VALUE, Double.MAX_VALUE};
            double[] topRight= new double[]{0.0, 0.0, 0.0};
            
        	for(int i=0; i<walker.length; i++){
        	    extracellular=true;
        	    
        		while(extracellular){
            		double[] r0= new double[D];
            		for(int j=0; j<D; j++){
            			r0[j]= twister.nextDouble()*substrateSize[j];
                    
            			// get top right and bottom left corners of enveloping cube
            			if(r0[j]<bottomLeft[j]){
            				bottomLeft[j]=r0[j];
            			}
            			if(r0[j]>topRight[j]){
            				topRight[j]=r0[j];
            			}
            		}
            		walker[i]= new Walker(r0, stepGenerator, substrate, synthScan, trajWriter);
            	
            		extracellular=!substrate.intracellular(walker[i]);
            	}
            }
        }
        else if(simParams.getInitialConditions()==SimulationParams.EXTRACELLULAR){
            // initially uniformly distributed across substrate
            double[] substrateSize=substrate.getSubstrateSize();
            boolean intracellular=true;
            
            double[] bottomLeft= new double[]{Double.MAX_VALUE, Double.MAX_VALUE, Double.MAX_VALUE};
            double[] topRight= new double[]{0.0, 0.0, 0.0};
            
            for(int i=0; i<walker.length; i++){
                intracellular=true;
                
            	while(intracellular){
	                double[] r0= new double[D];
	                for(int j=0; j<D; j++){
	                    r0[j]= twister.nextDouble()*substrateSize[j];
	                    
	                    // get top right and bottom left corners of envloping cube
	                    if(r0[j]<bottomLeft[j]){
	                    	bottomLeft[j]=r0[j];
	                    }
	                    if(r0[j]>topRight[j]){
	                    	topRight[j]=r0[j];
	                    }
	                }
	                
	                walker[i]= new Walker(r0, stepGenerator, substrate, synthScan, trajWriter);
	                intracellular=substrate.intracellular(walker[i]);
	            }
            }
        }
        else if(simParams.getInitialConditions()==SimulationParams.SPECIAL){

        	double[] substrateSize= substrate.getSubstrateSize();
        	
        	for(int i=2; i<walker.length; i++){
                double[] r0= new double[D];
            	for(int j=0; j<D; j++){
            		r0[j]= twister.nextDouble()*substrateSize[j];
                    
            		walker[i]= new Walker(r0, stepGenerator, substrate, synthScan, trajWriter);
            	}
        	}

        }
        else{
            String errMess= new String("unrecognised initial conditions code "+
                    	         simParams.getInitialConditions());
            logger.severe(errMess);
            throw new LoggedException(errMess);
        }
                
    }
    
    
    /** 
     *  runs the simualtion. tmax updates of N_walkers walkers
     *  using the simulation parameters given.
     */
    public void runMainLoop(){
          	
    	boolean[] isIntra= new boolean[walker.length];
        
    	substrate.init();        
    	
    	if((onlyRun==-1)||(calls==onlyRun)){
    	    for(int i=0; i<walker.length; i++){
    	        isIntra[i]=substrate.intracellular(walker[i]);
    	    }

    	    // write initial information to traj file
            if(trajWriter!=null){
                try {
                    trajWriter.writeDouble(dt*(double)tmax);
                    trajWriter.writeDouble((double)simParams.getN_walkers());
                    trajWriter.writeDouble((double)tmax);
                } catch (IOException ioe) {
                    throw new LoggedException(ioe);
                }
			}
        
        
            int when=88759;
            int who=0;        
            
            for(int t=0; t<tmax; t++){
                if((t%100)==0){
                	System.err.print("\r"+100.0*(double)t/(double)(simParams.getTmax())+"%     ");
                }
                
                for(int i=0; i<simParams.getN_walkers(); i++){
    
                	/*if(t==when){
                		if(i==who){
            				System.err.println("bug case reached");
            			}
                	}*/
    
                	
                	
                	if(p==0.0){
                		if(substrate.intracellular(walker[i])!=isIntra[i]){            		
                			logger.severe("walker pos: "+walker[i].r[0]+"  "+walker[i].r[1]+"  "+walker[i].r[2]);
                			throw new LoggedException("t= "+t+" i= "+i+" has crossed. isIntra="+isIntra[i]);
                		}
                	}
                	
                	
                	walker[i].update(t*dt, i);
                }
                
                // get the scan to do its thing...
                if(synthScan!=null){
                	synthScan.update(t);
                }
            }
            
            // write the final walker positions to traj file
            if(trajWriter!=null){
            	for(int i=0; i<simParams.getN_walkers(); i++){
            		try{
            			trajWriter.writeDouble(tmax*dt);
    	        		trajWriter.writeInt(i);
    	        		for(int j=0; j<D; j++){
    	        			trajWriter.writeDouble(walker[i].r[j]);
    	        		}
    	        	}
    	        	catch(IOException ioe){
    	        		throw new LoggedException(ioe);
    	        	}
            	}
            
            
    	        //flush and close the file writer
    	        try{
    	        	trajWriter.flush();
    	        	trajWriter.close();
    	        }
    	        catch(IOException ioe){
    	        	throw new LoggedException(ioe);
    	        }
            }
        }
    }
    
    
    
    /** 
     * initialises a simulation and runs the main loop before
     * constructing a 
     * @see data.DataSource#nextVoxel()
     */
    public double[] nextVoxel() throws DataSourceException {
        
        double[] S=null;
        
        if(!separateRuns){
        	// run the simulation once. subsequent voxels
        	// are different noise realisations of the 
        	// simulation data.
        	
        	if(voxel==0){
        		initialiseWalkers();

        		runMainLoop();
        	}
        }
        else{
        	// run a separate simulation for each voxel
        	
        	stepGenerator= StepGeneratorFactory.getStepGenerator(simParams);
        	initialiseWalkers();
    	    System.err.println("onlyrun= "+onlyRun+" calls= "+calls);
    	    runMainLoop();
        	
        	CL_Initializer.seed=twister.nextInt();
        	twister=new MTRandom(CL_Initializer.seed+189);
        }
        
        /**
         *  TODO: synthetic scan and trajectories currently 
         *  implemented in parallel. in future remove synthscan
         *  in simulation completely. 
         *  currently main output will be {-1.0, -1.0, -1.0} if 
         *  no scheme is passed to diffusion sim. 
         */
        if(synthScan!=null){
            // check compartmental output
            if(SimulationParams.sim_compartmentSignal==SimulationParams.INTRAONLY){
                // intracellular only
                S=synthScan.getCompartmentalSignals(true);
            }
            else if(SimulationParams.sim_compartmentSignal==SimulationParams.EXTRAONLY){
                // extracellular only
                S=synthScan.getCompartmentalSignals(false);
            }
            else if(SimulationParams.sim_compartmentSignal==SimulationParams.ALLCOMPS){
                /* all compartments, so must get data from intra and extracellular as
                 * well as the total, and then concatenate them together into a single
                 * array for output. this SHOULD work, but is a bit hacky.
                 */
                
                // get signals
                double[] Sin= synthScan.getCompartmentalSignals(true);
                double[] Sout= synthScan.getCompartmentalSignals(false);
                double[] Sall= synthScan.getSignals();
                
                // make new array
                S=new double[Sin.length+Sout.length+Sall.length];
                
                // concatenate signals into a single array
                for(int i=0; i<Sin.length; i++){
                    S[i]=Sin[i];
                }
                for(int i=0; i<Sout.length; i++){
                    S[Sin.length+i]=Sout[i];
                }
                for(int i=0; i<Sall.length;i++){
                    S[Sin.length+Sout.length+i]=Sall[i];
                }
            }
            else{
                // non-compartmental.
                S=synthScan.getSignals();
            }
        }
        else{
        	S= new double[] {-1.0, -1.0, -1.0};
        }
    	voxel++;
    
    	return S;
        
    }

    /** 
     * @see data.DataSource#more()
     */
    public boolean more() {
        if(voxel<numVoxels){
            return true;
        }
        else{
            return false;
        }
    }    
  
    /** calulate the mean square displacement of walkers at their current 
     *   locations. This is designed to be used by test code after a 
     *   simulation has run.
     * 
     * @return current mean square dispalacement of walkers
     */
    public double getMeanSquareDisplacement(){
        
        double meanSquareDisp=0.0;
        
        for(int i=0; i<walker.length; i++){
            
            double disp[]=walker[i].getDisplacement();
            double squareDisp=0.0;
            
            for(int j=0; j<D; j++){
                squareDisp+=disp[j]*disp[j];
            }
            
            
            
            meanSquareDisp+=squareDisp;
            
        }
        
        meanSquareDisp/=walker.length;
        
        return meanSquareDisp;
        
        
    }
    
    
    public static final void testDiffusionSimulation(){

        
        SimulationParams simParams;
        DiffusionSimulation diffSim;
        
        
        Object[] geomParams = new Object[] {new Double(1.0), new Double(3.0)};
        double[] stepParams = new double[] {0.01};

        
        SchemeV0 imParams=SchemeV0.getSchemeV0(7);
        
        
        simParams= new SimulationParams(1, 70000, 0.002, SimulationParams.SPIKE, 0, 
                	           geomParams, StepGeneratorFactory.FIXEDLENGTH,
                	           1.5, imParams);
        
        diffSim= new DiffusionSimulation(simParams, imParams);
        
        diffSim.nextVoxel();
        
        simParams= new SimulationParams(1000, 100000, 0.0001, SimulationParams.SPIKE, 
                SubstrateFactory.CELL_ISO, geomParams, StepGeneratorFactory.FIXEDLENGTH, 
                1.5, imParams);
  
        diffSim= new DiffusionSimulation(simParams, imParams);
        
        diffSim.nextVoxel();
        

        // striped lattice needs an extra parameter
        geomParams=new Object[] {new Double(1.0), new Double(20.0), new Double(2.0)};
        simParams= new SimulationParams(100, 100000, 0.0001, SimulationParams.UNIFORM, 
                SubstrateFactory.CELL_STRIPED, geomParams, StepGeneratorFactory.FIXEDLENGTH, 
                10.0, imParams);

        diffSim= new DiffusionSimulation(simParams, imParams);
  
        diffSim.nextVoxel();        
    }

    
    private static void testFullScale(){
        // tests diffusion statistics on a striped substrate
        SimulationParams simParams;
        DiffusionSimulation diffSim;
        
        int tmax=100000;
        
        double p=0;


        Object[] geomParams = new Object[] {new Double(1E-6), new Integer(20), new Double(1.0)};
        double[] stepParams = new double[] {1E-8};

        
        SchemeV0 imParams=SchemeV0.getSchemeV0(7);
        

               
        simParams = new SimulationParams(100, tmax, 
                p, SimulationParams.SPIKE, SubstrateFactory.CELL_STRIPED, geomParams, 
                StepGeneratorFactory.FIXEDLENGTH, 0.5, imParams);
        
        stepParams=StepGeneratorFactory.getStepParamsArray(StepGeneratorFactory.FIXEDLENGTH,
        		simParams, imParams);
        
        simParams.setStepParams(stepParams);

        diffSim= new DiffusionSimulation(simParams, imParams);
        
        double[] S= diffSim.nextVoxel();
    	
    }
    
    
   private static void testReflectionFromCylinders(){
    	
        SimulationParams simParams;
        DiffusionSimulation diffSim;
        
        int tmax=1000;
        
        double p=0.0;

        double[] p_arr= new double[1];

        Object[] geomParams = new Object[] {new Double(1E-6), new Double(2E-5),
    			new Integer(1), new Integer(1), new Integer(1), new Double(2.0)};
        double[] stepParams = new double[] {DiffusionSimulation.DIFF_CONST};

        
        SchemeV0 imParams=SchemeV0.getSchemeV0(7);
        

               
        simParams = new SimulationParams(1, tmax, 
                p, SimulationParams.UNIFORM, SubstrateFactory.CYL_1_INFLAM, geomParams, 
                StepGeneratorFactory.FIXEDLENGTH, 1.0, imParams);
        
        stepParams=StepGeneratorFactory.getStepParamsArray(StepGeneratorFactory.FIXEDLENGTH,
        		simParams, imParams);
        
        simParams.setStepParams(stepParams);

        StepGenerator stepGen= StepGeneratorFactory.getStepGenerator(simParams);
        
        diffSim= new DiffusionSimulation(simParams, imParams);
    	
        diffSim.substrate.init();
        
    	// that's everything. now we'll reach inside and set up the test case.
        SquashyCylinder cylinder= new SquashyCylinder(new double[]{1.0, 1.0, 0.0}, 0.5, 0.0);
        ((SquashyInflammationSubstrate)diffSim.substrate).cylinder[0]=cylinder;
    	
        
        double[] P = cylinder.getPosition();
        double[] V = cylinder.getAxis();
        double r = cylinder.getRadius();
        
        System.err.println("cylinder position =("+P[0]+","+P[1]+","+P[2]+")");
        System.err.println("cylinder axis =("+V[0]+","+V[1]+","+V[2]+")");
        System.err.println("radius = "+r);
        
        double[] r0 = new double[D];
        
        r0[0]=P[0]+0.9*r;
        r0[1]=P[1];
        r0[2]=0.0;
        
        Walker walker = new Walker(r0);
        
        double[] step= new double[D];
        
        step[0]=0.2*r;
        step[1]=0.0;
        step[2]=0.2*r;
        
        double[] normal= new double[D];
        double[] d= new double[1];
        double origLength=0.0;
        
        for(int i=0; i<D; i++){
        	origLength+=step[i]*step[i];
        }
        origLength=Math.sqrt(origLength);
      
        
        System.err.println("walker pos = "+walker.r[0]+","+walker.r[1]+","+walker.r[2]+")");
        System.err.println("step = ("+step[0]+","+step[1]+","+step[2]+")");
        
        boolean[] in = new boolean[1];
        
        double[] newStep= new double[]{0.0, 0.0, 0.0};
        
        
        boolean crosses =diffSim.substrate.crossesMembrane(walker, newStep, step, normal, d, false, origLength, in, p_arr);
    	
        System.err.println("crosses membrane is "+crosses);
        
        double[] toBarrier= new double[D];
        double[] amended= new double[D];
        double[] unamended= new double[D];
        
        diffSim.substrate.testAmendment(walker, step, normal, d, origLength, toBarrier, amended, unamended);
        
        System.err.println("toBarrier =("+toBarrier[0]+","+toBarrier[1]+","+toBarrier[2]+")");
        System.err.println("amended =("+amended[0]+","+amended[1]+","+amended[2]+",");
        System.err.println("unamended =("+unamended[0]+","+unamended[1]+","+unamended[2]+",");
        
        
        
    }
   
   
   	private static void testNonIntersection(){
   	
       SimulationParams simParams;
       DiffusionSimulation diffSim;
       
       int tmax=1000;
       
       double p=0.0;

/*       try{
           DiffusionSimulation.meanSquareWriter = new BufferedWriter(new FileWriter("squareDisps_p="+p+"_stripes_1.dat"));
       }
       catch(IOException ioe){
           throw new LoggedException(ioe);
       }
*/

       Object[] geomParams = new Object[] {new Double(1E-6), new Double(2E-5),
   			new Integer(1), new Integer(1), new Integer(1), new Double(2.0)};
       double[] stepParams = new double[] {DiffusionSimulation.DIFF_CONST};

       
       SchemeV0 imParams=SchemeV0.getSchemeV0(7);
       

              
       simParams = new SimulationParams(1, tmax, 
               p, SimulationParams.UNIFORM, SubstrateFactory.CYL_1_INFLAM, geomParams, 
               StepGeneratorFactory.FIXEDLENGTH, 1.0, imParams);
       
       stepParams=StepGeneratorFactory.getStepParamsArray(StepGeneratorFactory.FIXEDLENGTH,
       		simParams, imParams);
       
       simParams.setStepParams(stepParams);

       StepGenerator stepGen = StepGeneratorFactory.getStepGenerator(simParams);
       
       diffSim= new DiffusionSimulation(simParams, imParams);
   	
       diffSim.substrate.init();
       
   	// that's everything. now we'll reach inside and set up the test case.
       SquashyCylinder cylinder= new SquashyCylinder(new double[]{1.0, 1.0, 0.0}, 2.0, 0.0);
       ((SquashyInflammationSubstrate)diffSim.substrate).cylinder[0]=cylinder;
   	
       
       double[] P = cylinder.getPosition();
       double[] V = cylinder.getAxis();
       double r = cylinder.getRadius();
       
       System.err.println("cylinder position =("+P[0]+","+P[1]+","+P[2]+")");
       System.err.println("cylinder axis =("+V[0]+","+V[1]+","+V[2]+")");
       System.err.println("radius = "+r);
       
       double[] r0 = new double[D];
       
       r0[0]=P[0];
       r0[1]=P[1];
       r0[2]=0.0;
       
       Walker walker = new Walker(r0);
       
       double[] step= new double[D];
       
       step[0]=0.5*r;
       step[1]=0.0;
       step[2]=0.0;//0.2*r;
       
       double[] normal= new double[D];
       double[] d= new double[1];
       double origLength=0.0;
       
       for(int i=0; i<D; i++){
       	origLength+=step[i]*step[i];
       }
       origLength=Math.sqrt(origLength);
     
       
       System.err.println("walker pos = "+walker.r[0]+","+walker.r[1]+","+walker.r[2]+")");
       System.err.println("step = ("+step[0]+","+step[1]+","+step[2]+")");
       
       boolean[] in= new boolean[1];
       double[] offset= new double[]{0.0, 0.0, 0.0};
       double[] p_arr= new double[1];
       
       boolean crosses =diffSim.substrate.crossesMembrane(walker, offset, step, normal, d, false, origLength, in, p_arr);
   	
       System.err.println("crosses membrane is "+crosses);
       
       double[] toBarrier= new double[D];
       double[] amended= new double[D];
       double[] unamended= new double[D];
       
       diffSim.substrate.testAmendment(walker, step, normal, d, origLength, toBarrier, amended, unamended);
       
       System.err.println("toBarrier =("+toBarrier[0]+","+toBarrier[1]+","+toBarrier[2]+")");
       System.err.println("amended =("+amended[0]+","+amended[1]+","+amended[2]+",");
       System.err.println("unamended =("+unamended[0]+","+unamended[1]+","+unamended[2]+",");
       
       
       
   }

    private static void testSingleCylinder(){
    	
    	SimulationParams simParams;
        DiffusionSimulation diffSim;
        
        int tmax=1000;
        
        double p=0.0;

        double L=2E-6;

        Object[] geomParams = new Object[] {new Double(1E-6), new Double(2E-5),
    			new Integer(1), new Integer(1), new Integer(1), new Double(L)};
        double[] stepParams = new double[] {1E-7};

        
        SchemeV0 imParams=SchemeV0.getSchemeV0(7);
        
        simParams = new SimulationParams(100, 100000, 
                p, SimulationParams.SPIKE, SubstrateFactory.CYL_1_INFLAM, geomParams, 
                StepGeneratorFactory.FIXEDLENGTH, 1.0, imParams);
        
        //stepParams=StepGeneratorFactory.getStepParamsArray(StepGeneratorFactory.FIXEDLENGTH,
        //		simParams, imParams);
        
        simParams.setStepParams(stepParams);

        diffSim= new DiffusionSimulation(simParams, imParams);
    	
        SquashyInflammationSubstrate inflam=(SquashyInflammationSubstrate)diffSim.substrate;
        
        
        inflam.P[0][0]= L/2.0;
        inflam.P[0][1]= L/2.0;
        inflam.P[0][2]= L/2.0;
        
        
        diffSim.nextVoxel();
        double[] signal=diffSim.synthScan.getSignals();
        
        for(int i=0; i<signal.length; i++){
        	System.err.println("signal["+i+"] = "+signal[i]);
        }
        
        double[] meanSq= new double[]{0.0, 0.0, 0.0};
        
        for(int i=0; i<diffSim.walker.length; i++){
        	for(int j=0; j<D; j++){
        		meanSq[j]+=(diffSim.walker[i].r[j]-diffSim.walker[i].r0[j])*(diffSim.walker[i].r[j]-diffSim.walker[i].r0[j]);
        	}
        }
        for(int j=0; j<D; j++){
        	meanSq[j]/=diffSim.walker.length;
        	System.err.println("meanSq["+j+"]= "+meanSq[j]);
        }
    }
    
    private static void testPerfectIsotropy(){
        
        CL_Initializer.brownianSimulation= true;
        SimulationParams.sim_N_walkers= 10000;
        SimulationParams.sim_tmax= 100;
        SimulationParams.sim_geomType= SubstrateFactory.EMPTY;
        SimulationParams.sim_initial= SimulationParams.SPIKE;
        CL_Initializer.numVoxels= 1;
        SimulationParams.sim_l= 1e-3;
        CL_Initializer.schemeFile= "facet_scheme_1000.scheme";
        SimulationParams.sim_G= 0.022;
        SimulationParams.sim_G_set= true;
        SimulationParams.sim_delta= 0.032;
        SimulationParams.sim_delta_set= true;
        SimulationParams.sim_DELTA= 0.04;
        SimulationParams.sim_DELTA_set= true;
        OutputManager.outputFile= "perfect.bfloat";
        
        CL_Initializer.initImagingScheme();
        CL_Initializer.initDataSynthesizer();
        
        
        DiffusionSimulation diffsim= (DiffusionSimulation)CL_Initializer.data;
        SyntheticScan scan= diffsim.synthScan;
        
        double[] s= diffsim.stepGenerator.getStep(null);
        double len=0.0;
        for(int i=0; i<s.length; i++){
            len+=s[i]*s[i];
        }
        len=Math.sqrt(len);
        
        double[][] step= new double[SimulationParams.sim_N_walkers][DiffusionSimulation.D];
        
        int n = SimulationParams.sim_N_walkers;
        
        Random rng= new Random(183745634);
        
        for(int i=0; i<n; i++){
            double theta= (double)(2.0*Math.PI*((double)i/(double)n));
            
            //double theta= 2.0*Math.PI*rng.nextDouble();
            
            double x= Math.cos(theta)*len;
            double y= Math.sin(theta)*len;
            
            step[i][0]=x;
            step[i][1]=y;
            step[i][2]=0.0;
            
        }
        
        diffsim.initialiseWalkers();
        Walker[] walker= diffsim.walker;
        
        
        
        
        // first update loop
        for(int i=0; i<walker.length; i++){
            
            /*for(int j=0; j<walker[i].dPhi.length; j++){
                walker[i].dPhi[j]+=scan.getPhaseShift(walker[i].r, 0.0, j, 0.0);
            }*/
            
            double[] newStep= new double[D];
            for(int j=0; j<D; j++){
                newStep[j]= step[i][j];
            }
            
            diffsim.substrate.amend(walker[i], step[i], 0.0, i);
            
            for(int j=0; j<D; j++){
                if(newStep[j]!=step[i][j]){
                    diffsim.logger.warning("step "+i+" amended (dir "+j);
                }
            }
            
            walker[i].makeStep(step[i]);
            
        }
        
        
        
        BufferedWriter phasesWriter;
        
        try{
            phasesWriter= new BufferedWriter(new FileWriter("walkerPhases.csv"));
        }
        catch(IOException ioe){
            throw new LoggedException(ioe);
        }
        
        
        //second update loop
        for(int i=0; i<walker.length; i++){
            
            for(int j=0; j<walker[i].dPhi.length; j++){
                walker[i].dPhi[j]+=scan.getPhaseShift(walker[i], diffsim.dt, j, 0.0);
            }
            
            double[] newStep= new double[D];
            for(int j=0; j<D; j++){
                newStep[j]= step[i][j];
            }
            
            diffsim.substrate.amend(walker[i], step[i], 0.0, i);
            
            for(int j=0; j<D; j++){
                if(newStep[j]!=step[i][j]){
                    diffsim.logger.warning("step "+i+" amended (dir "+j);
                }
            }
            
            walker[i].makeStep(step[i]);
            
        }
        
        double[] signal= diffsim.synthScan.getSignals();
        
        for(int i=0; i<signal.length; i++){
            
            double theta= (double)((double)i/(double)n)*2*Math.PI;
            
            System.err.println(theta+" "+signal[i]);
        }
        
    }
    
    
    public static void main(String[] args){

    	//DiffusionSimulation.testFullScale();
    	
    	//DiffusionSimulation.testReflectionFromCylinders();
    	
    	//DiffusionSimulation.testNonIntersection();
    	
    	//DiffusionSimulation.testSingleCylinder();
        
        DiffusionSimulation.testPerfectIsotropy();
    }
    
}
