package simulation.geometry;

import imaging.Scheme;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.Logger;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;
import simulation.dynamics.StepGeneratorFactory;
import simulation.dynamics.Walker;

public class MeshSubstrate extends Substrate {

	/** dimensionality of space */
	private static final int D= DiffusionSimulation.D;
	
	/** triangles forming the object mesh */
	private final Collection<Triangle> triangles;
	
	/** logging object */
	private final Logger logger= Logger.getLogger(this.getClass().getName());
	
	/** index of the triangle to skip, if we're skipping */
	private int skipping=-1;
	
	/** size of (cubic) unit cell around object */
	private final double[] sep;
	
	/** length scale that largest vertex displacement vector in object is scaled to */
	private final double size;
	
	/** the ray vector that is used as a step in the intracellular routine */
	private final double[] ray;

	/** map of triangles to their exception coplanar lists */
	private final HashMap<Triangle, ArrayList<Triangle>> coplanarMap;
	
	
	/** 
	 * read a substrate from the named PLY file and scale it to the 
	 * specified length. 
	 * 
	 * currently assumes that the object in the PLY file is centred
	 * on the origin. this is simple enough to change. eventually
	 * i'll add an option to specify better configurations such as 
	 * multiple objects, positions and rotations.
	 * 
	 * the object will be scaled so that the longest displacement 
	 * vector to a vertex from the origin takes on the length given
	 * in the scale parameter. 
	 * 
	 * @param fname name of PLY file to read
	 * @param scale length to scale object to
	 */
	public MeshSubstrate(String fname, double size, double sep, SimulationParams simParams){
		
		super(simParams, new double[] {sep, sep, sep});
		
		if(sep<size){
			logger.warning("size of PLY mesh object "+
					size+" is smaller that specifed separation "+sep);
		}

		// store the separation
		this.sep= new double[]{sep, sep, sep};
		
		// store the object size
		this.size=size;

		// assemble the ray vector -- parallel to an axis and slightly larger than object
		this.ray=new double[]{0.0, 0.0, Double.MAX_VALUE};
		

		
		logger.info("reading object mesh from '"+fname+"'");
		triangles =PLYreader.readPLYfile(fname, 1.0, simParams.getP());
		
		/* normalise vertices to desired length 
		 * (longest vertex position vector will
		 * be normalised to length given by scale 
		 * parameter)
		 */
		Iterator<Triangle> triIt=triangles.iterator();
		double maxLen=0.0;
		
		while(triIt.hasNext()){
			Triangle nextTriangle=(Triangle)triIt.next();
			
			for(int i=0; i<3; i++){
				double len=0.0;
				double[] vertex= nextTriangle.getVertex(i);
				for(int j=0; j<D; j++){
					len+=vertex[j]*vertex[j];
				}
				if(len>maxLen){
					maxLen=len;
				}
			}
		}
		
		// take root of max length
		maxLen=Math.sqrt(maxLen);
		
		
		// rescale vertices
		triIt=triangles.iterator();
		while(triIt.hasNext()){
			Triangle nextTriangle= (Triangle)triIt.next();
			for(int i=0; i<3; i++){
				/* getVertex returns the same instance
				 * as internal representation of vertex
				 * so can apply scale and translation
				 * here.
				 */
				double[] vertex=nextTriangle.getVertex(i);
				double scale= (size/maxLen);
				for(int j=0; j<D; j++){
					vertex[j]*=scale;
					vertex[j]+=sep/2.0;
				}
			}
			// recalculate the various internal vectors and 
			// dot products we need to categorise triangle
			nextTriangle.initVectors();
		}
		
		
		this.coplanarMap= PLYreader.getCoplanarListMap();
		
		// read out the verices of the triangles
		Iterator<Triangle> tris= triangles.iterator();
		BufferedWriter writer;

		try{
			writer = new BufferedWriter(new FileWriter("tris.csv"));
		}
		catch(IOException ioe){
			throw new RuntimeException(ioe);
		}
		
		while(tris.hasNext()){
			Triangle tri=(Triangle)tris.next();
			
			try{
				for(int i=0; i<3; i++){
					double[] vert= tri.getVertex(i);
					
					writer.write(vert[0]+","+vert[1]+","+vert[2]+"\n");
				}
			}
			catch(IOException ioe){
				throw new RuntimeException(ioe);
			}

		}
		
		try{
			writer.flush();
			writer.close();
		}
		catch(IOException ioe){
			throw new RuntimeException(ioe);
		}
		
	}
	
	/**
	 * maps the physical walker position into the periodic 
	 * unit cell containing the object
	 * 
	 * @param walkerPos physical continuous space position
	 * 
	 * @return poisiton in cell
	 */
	private double[] getSubstrateCoords(double[] walkerPos, double[] offset){
		
		double[] subsCoords= new double[D];
		
		for(int i=0; i<D; i++){
			double windingNum= Math.floor((walkerPos[i]+offset[i])/sep[i]);
			subsCoords[i]=walkerPos[i]+offset[i]-windingNum*sep[i];
		}
		
		return subsCoords;
	}
	
	public boolean crossesMembrane(Walker walker, double[] offset, double[] stepVector,
			double[] normal, double[] d, boolean skipCurrent,
			double origLength, boolean[] in, double[] p) {
		
		
		double nearest=2.0;
		
		boolean crossing=false;
		
		double[] tempNormal= new double[D];
		double[] tempD= new double[1];
		double[] intDist= new double[1];
		
		Iterator<Triangle> triIt= triangles.iterator();
		
		int i=0;
		
		// map walker position into unit cell
		double[] subsCoords= getSubstrateCoords(walker.r, offset);
		
		// check interactions with all triangles
		while(triIt.hasNext()){
			
			// get the next triangle
			Triangle tri=(Triangle)triIt.next();
			
			// check skipping
			if(skipCurrent){
				if(i==skipping){
					i++;
					continue;
				}				
			}
			
			// check intersection
			if(tri.crosses(subsCoords, stepVector, tempNormal, tempD, false, origLength, intDist, null, p)){
				crossing=true;
				
				// if crossing is nearer than any previous one
				// log the triangle's details
				if(intDist[0]<nearest){
					nearest=intDist[0];             // set the arclength for future comparisons
					for(int j=0; j<D; j++){			// copy the normal over
						normal[j]=tempNormal[j];
					}
					d[0]=tempD[0];					// set the distance to interaction point
					skipping=i;						// set the skipping index
				}
				p[0]=tri.getPermeability(0);
			}
			
			// increment triangle counter
			i++;
		}
		
		if(!crossing){
			skipping=-1;
		}
		
		
		return crossing;
	}

	/**
	 * return peak coord, used as the ncentre of the object
	 */
	public double getPeakCoord() {
		
		return sep[0]/2.0;
	}

	/**
	 * returns the size of the unit cell
	 */
	public double[] getSubstrateSize() {
		// TODO Auto-generated method stub
		return sep;
	}

	/** 
	 * not used, does nothing.
	 */
	public void init() {

	}

	/**
	 * performs the ray test to check if a point is inside the
	 * object or not. This test can cope with non-convex objects
	 * (like sharks) and is almost as simple as the dot product 
	 * test used for convex objects.
	 * 
	 * We project a ray in an arbitrary direction and count the
	 * number of intersections with the surface of the object.
	 * If this number if odd, the point is inside the object.
	 * 
	 * This is only completely reliable for closed surfaces.
	 * any holes will cause it to get it wrong some of the time.
	 * 
	 * @param walker the walker whose position to test.
	 * 
	 * @return true if inside object, false if outside
	 */
	public boolean intracellular(Walker walker) {
		
		// ArrayList of coplanar triangles to
		// skip after intersections are found
		ArrayList<Triangle> skips=new ArrayList<Triangle>();
		
		double[] offset= new double[]{0.0, 0.0, 0.0};
		
		// map walker's position into unit cell
		double[] subsCoords=getSubstrateCoords(walker.r, offset);
		
		// space to store the geometric stuff
		// even though we ignore it here, it still
		// needs to go somewhere
		double[] tempNorm= new double[D];
		double[] tempD= new double[1];
		double[] intDist= new double[1];
		double[] p= new double[1];
		
		// number of intersections
		int n=0;
				
		// triangle iterator
		Iterator<Triangle> triIt=triangles.iterator();
		
		// loop over all triangles
		while(triIt.hasNext()){
			// get triangle
			Triangle triangle=(Triangle)triIt.next();

			if(skips.contains(triangle)){
				continue;
			}
			
			// check for intersection with ray
			if(triangle.crosses(subsCoords, ray, tempNorm, tempD, false, 2.0*size, intDist, null, p)){
				// if yes, add coplanar triangles to the skipping array
				ArrayList<Triangle> newSkips=coplanarMap.get(triangle);
				
				if(newSkips!=null){
					skips.addAll(newSkips);
				}
				
				// increment the intersection counter
				n++;
			}
			
		}
		
		// if n is odd, we're inside
		if(n%2==1){
			return true;
		}
		
		// otherwise outside
		return false;
	}

	/** 
	 * test ply substrate reading and intra cellular routine
	 * 
	 * @param args
	 */
	public static void main(String[] args){
		
		int tmax=100;
		
		Scheme scheme= null;
		
		Object[] geomParams= null;
		
		SimulationParams simParams= new SimulationParams(1, 1000, 0.0, 
                SimulationParams.UNIFORM, 0, geomParams, StepGeneratorFactory.FIXEDLENGTH, 
                1.5, scheme);
		
		// construct substarte from file
		MeshSubstrate meshSubs= new MeshSubstrate("cube.ply", 1.0, 2.0, simParams);
		
		// fish out the triangles
		Collection<Triangle> mesh= meshSubs.triangles;
		
		// and let's have a look at them...
		System.err.println("read "+mesh.size()+" triangles");
		
		int i=0;
		Iterator<Triangle> meshIt= mesh.iterator();
		while(meshIt.hasNext()){
			Triangle triangle= (Triangle)meshIt.next();
			
			System.err.println(++i+" "+triangle);
		}
		
		// now let's have a look at the intracellular routine
		double midPt= meshSubs.getPeakCoord();
		
		double[] pos= new double[]{midPt, midPt, midPt};
		
		Walker walker= new Walker(pos);
		
		System.err.println("midpoint is "+midPt);
		
		System.err.println("intracellular(midpoint) is "+meshSubs.intracellular(walker));
		
		double size= meshSubs.size;
		
		System.err.println("mesh size is "+size);
		
		pos[0]+=1.1*size;
		
		walker = new Walker(pos);
		
		System.err.println("extra cellular point is ("+pos[0]+","+pos[1]+","+pos[2]+")");
		System.err.println("intracellular is "+meshSubs.intracellular(walker));
	}
	
}
