package simulation.geometry;

import java.util.logging.Logger;
import java.io.*;

import simulation.SimulationParams;

/**
 * Converts a VRML file into Triangle objects.
 */
public class VRMLReader {
    private Logger logger= Logger.getLogger(this.getClass().getName());
    
    private static VRMLReader vr;
    
    // increment size (and initial size) of the arrays
    private static final int ARRAY_INC_SIZE = 200;
    
    // initial StringBuffer size
    private static final int SB_INIT_SIZE = 1000;
    
    private Triangle [] triangleList;
    
    // arrays for the verts, normals, indices and size indicators
    private double [][] verts;
    private int vertsArrayInd;
    private double [][] normals;
    private int normalArrayInd;
    private int [] [] faces;
    private int faceInds;
    private int [] [] faceNorms;
    private int faceNormInds;
    
    // flags to ensure numbers are parsed to the correct array
    private boolean inPoints = false;
    private boolean inFaces = false;
    private boolean inNorms = false;
    private boolean inFaceNorms = false;
    
    /*
     * Constructor for VRML Reader
     */
    private VRMLReader() {
    }
    
    /*
     * returns a VRMLReader object.
     * @return a VRMLReader object  
     */
    public static VRMLReader getVRMLReader() {
        if(vr==null) {
            vr = new VRMLReader();
        }
        return vr;
    }
    
    /*
     * loads the specified VRML file and converts to Triangles
     * 
     * @param filename the name to the file to be loaded
     * @return an array of Triangles
     */
    public Triangle [] loadFile(String filename){
        
        // set stuff up!
        resetVariables();
        int geometry = 0;
        String str = "";
        int index = 0;
        boolean finishedObject = false;
        boolean nextObject = false;
        
        // try whacking things into lists!!
        try {
            // load file
            BufferedReader br = new BufferedReader(new FileReader(filename));
            
            // read entire file
            while((str = br.readLine())!=null) {
                /*
                 * look for geometry... we assume that geometry, coords,
                 * normals and indices are on different lines
                 */
                if((str.indexOf("geometry")>-1) || nextObject) {
                    nextObject = false;
                    geometry++;
                    
                    // get points, faces, etc
                    if(str.indexOf("geometry")>-1)
                        str = str.substring(str.indexOf("geometry")+8, str.length());  //need to make sure that pattern matching returns false!
                    do
                    {
                        finishedObject = false;
                        
                        if(str ==null) {
                            //end of file!
                            finishedObject = true;
                        } else if(str.indexOf("point")>-1) {
                            inPoints = true;
                            parseData(br, str.substring(str.indexOf("point")).trim(),",");
                            inPoints = false;
                        } else if (str.indexOf("normal ")>-1) {
                            inNorms = true;
                            parseData(br, str.substring(str.indexOf("normal ")).trim(),",");
                            inNorms = false;
                        } else if (str.indexOf("coordIndex")>-1) {
                            inFaces = true;
                            parseData(br, str.substring(str.indexOf("coordIndex")).trim(),"-1");
                            inFaces = false;
                        } else if (str.indexOf("normalIndex")>-1) {
                            inFaceNorms = true;
                            parseData(br, str.substring(str.indexOf("normalIndex")).trim(),"-1");
                            inFaceNorms = false;
                        } else if (str.indexOf("geometry")>-1) {
                            finishedObject = true;
                            nextObject = true;
                        }
                        
                        if(!finishedObject) {
                            // checked line for all tags, so carry on to next line
                            str = br.readLine();
                        } else {
                            // if finished object, add its components to the triangle list!
                            triangleList = conCatList(triangleList,convertToTriangles());

                            // clear arrays for next object!
                            resetVariables();
                        }
                    }
                    while(!finishedObject);
                    
                    
                }
            }
            br.close();
            return triangleList;
        } catch(IOException ioe) {
            logger.severe("Couldn't open specified .wrl file");
        }
        
        logger.severe("Returning null list of Triangles!");

        // return null list
        return triangleList;
    }
    
    /*
     * reads the data from the file and parses it into the appropriate array
     * @param br the BufferedReader for the file
     * @param s the first string containing vertex/index info from the
     * BufferedReader
     *
     */
    private void parseData(BufferedReader br, String s, String delimiter) throws IOException {
        boolean done = false;
        String str = s;
        
        // find start of data
        int index = str.indexOf('[') + 1;
        
        while(index<1) // if square bracket not found, keep looking until it is!
        {
            str = br.readLine().trim();
            index = str.indexOf('[') + 1;
        }
        
        if(index > str.length()) {
            //end of line, so find start of data
            str = br.readLine();
            index = 0;
        } else {
            str = str.substring(index);
            index = 0;
        }
        
        // concatenate all points onto one line!
        StringBuffer sBuffer = new StringBuffer(SB_INIT_SIZE); //create a string buffer with a capacity of SB_INIT_SIZE characters
        sBuffer.append(str);
        
        while(str.indexOf(']')==-1) {
            str = br.readLine();
            sBuffer.append(str);
        }
        
        str = sBuffer.toString();
        str = str.substring(0,str.indexOf(']')+1).trim();
        
        // start parsing points/vectors
        while(!done) {

            if(str.indexOf(delimiter,index)>-1) {
                if(delimiter.equals("-1"))
                    parseInt3(str.substring(index,str.indexOf(delimiter,index)));
                else
                    parseDouble3(str.substring(index,str.indexOf(delimiter,index)));
                
                index = str.indexOf(delimiter,index)+delimiter.length();
                if(index >= str.indexOf(']') || index == -1) {
                    done = true;
                } else if(index == str.trim().length() || index == -1) {
                    if(str.indexOf(']')>-1)
                        done = true;
                    else {
                        str = br.readLine();
                        index = 0;
                    }
                }
            } else if(index >= str.indexOf(']'))
                done = true;
            else if((str.indexOf(']') - index) > 0) {
                str = str.substring(index,str.indexOf(']')+1).trim();
                index = 0;
                if(str.length() > 4) {
                    if(delimiter.equals("-1"))
                        parseInt3(str.substring(index,str.indexOf(']')));
                    else
                        parseDouble3(str.substring(index,str.indexOf(']')));
                    
                }
                done = true;
            }
        }
    }
    /*
     * parses 3 doubles from a substring.  doubles are space separated
     * @param s the substring containing 3 doubles
     */
    private void parseDouble3(String s) {
        String str = s.trim();
        
        try {
            int indexVal = str.indexOf(" ");
            
            double x = Double.parseDouble(str.substring(0,indexVal));
            
            // get rid of white space
            while(str.charAt(indexVal+1)==' ')
                indexVal++;
            
            double y = Double.parseDouble(str.substring(indexVal,str.indexOf(" ",indexVal+1)));
            
            indexVal = str.indexOf(" ",indexVal+1);
            
            // get rid of white space
            while(str.charAt(indexVal+1)==' ')
                indexVal++;
            
            double z = Double.parseDouble(str.substring(indexVal,str.length()));
            
            if(inPoints) {
                if(vertsArrayInd+1>verts.length)
                    verts = increaseArraySize(verts);
                
                verts[vertsArrayInd][0] = x;
                verts[vertsArrayInd][1] = y;
                verts[vertsArrayInd][2] = z;
                vertsArrayInd++;
            } else if(inNorms) {
                if(normalArrayInd+1>normals.length)
                    normals = increaseArraySize(normals);
                
                normals[normalArrayInd][0] = x;
                normals[normalArrayInd][1] = y;
                normals[normalArrayInd][2] = z;
                normalArrayInd++;
            }
            
        } catch (NumberFormatException nfe) {
	    logger.severe("could not parse doubles from string: "+ str);
        }
    }
    
    /*
     * parses 3 integers from a substring.  The values can be either
     *     a) comma separated or
     *     b) space separated
     *
     * @param s the substring to parse (containing 3 integers)
     */
    private void parseInt3(String s) {
        String str = s.trim();

        try {
            if(str.length()>4) {
                boolean quad = false;
                int firstInd = str.indexOf(',');
                if(firstInd==0) {
                    firstInd=1;
                    
                    while(str.charAt(firstInd+1)==' ')
                        firstInd++;
                } else
                    firstInd=0;
                
                int indexVal = str.indexOf(',', 1);
                
                int vertA = 0;
                int vertB = 0;
                int vertC = 0;
                
                if(indexVal == str.length() || indexVal==-1) {
                    
                    // find white space after first value
                    indexVal = str.indexOf(' ', firstInd+1);
                    
                    // get rid of white space
                    while(str.charAt(indexVal+1)==' ')
                        indexVal++;
                    
                    //assume space separated
                    vertA = Integer.parseInt(str.substring(firstInd,indexVal).trim());
                    
                    firstInd = indexVal;
                    indexVal = str.indexOf(' ', firstInd+1);

                    // get rid of white space
                    while(str.charAt(indexVal+1)==' ')
                        indexVal++;
                    
                    vertB = Integer.parseInt(str.substring(firstInd,indexVal).trim());
                    
                    firstInd = indexVal;
                    indexVal = str.indexOf(' ', firstInd+1);
                    
                    vertC = Integer.parseInt(str.substring(firstInd,str.length()).trim());
                }
                else {
                    vertA = Integer.parseInt(str.substring(firstInd,indexVal).trim());
                    vertB = Integer.parseInt(str.substring(indexVal+1,str.indexOf(',',indexVal+1)).trim());
                    vertC = Integer.parseInt(str.substring(str.indexOf(',',indexVal+1)+1,str.length()-1).trim());
                }
                
                if(inFaces) {
                    if(faceInds+1>faces.length)
                        faces = increaseArraySize(faces);
                    
                    faces[faceInds][0] = vertA;
                    faces[faceInds][1] = vertB;
                    faces[faceInds][2] = vertC;
                    faceInds++;
                } else if(inFaceNorms) {
                    if(faceNormInds+1>faceNorms.length)
                        faceNorms = increaseArraySize(faceNorms);
                    
                    faceNorms[faceNormInds][0] = vertA;
                    faceNorms[faceNormInds][1] = vertB;
                    faceNorms[faceNormInds][2] = vertC;
                    faceNormInds++;
                }
            }
            
        } catch (NumberFormatException nfe) {
            logger.severe("could not parse indices from string: "+ str);
        }
    }

    /*
     * increases the size of a 2D array of doubles
     * @param a the original array
     * @return a new array of doubles containing the original array (plus a
     * bit more space!)
     */  
    private double [][] increaseArraySize(double [][] a) {
        double [][] b = new double [a.length + ARRAY_INC_SIZE][3];
        
        for(int i=0; i<a.length;i++)
            for(int j=0;j<3;j++)
                b[i][j] = a[i][j];
        for(int i=a.length; i<b.length;i++)
            for(int j=0;j<3;j++)
                b[i][j] = 0.0;
        
        return b;
    }
    
    /*
     * increases the size of a 2D array of integers
     *
     * @param a the original array
     * @return a new array of integers containing the original array (plus a
     * bit more space!)
     */  
    private int [][] increaseArraySize(int [][] a) {
        int [][] b = new int [a.length + ARRAY_INC_SIZE][3];
        
        for(int i=0; i<a.length;i++)
            for(int j=0;j<3;j++)
                b[i][j] = a[i][j];
        for(int i=a.length; i<b.length;i++)
            for(int j=0;j<3;j++)
                b[i][j] = 0;
        
        return b;
    }
    
    /*
     * Converts the arrays of points/indices to Triangles
     *
     * @return an array of Triangles.
     */
    private Triangle[] convertToTriangles() {
        Triangle [] list = new Triangle[faceInds];
        
        double [] tempVert1 = new double [3];
        double [] tempVert2 = new double [3];
        double [] tempVert3 = new double [3];
        double [] normalVect = new double [3];
             
        //should be the same amount of face and normal indices!!
        for(int i=0;i<faceInds;i++) {
            tempVert1[0] = verts[faces[i][0]][0];
            tempVert1[1] = verts[faces[i][0]][1];
            tempVert1[2] = verts[faces[i][0]][2];
            
            tempVert2[0] = verts[faces[i][1]][0];
            tempVert2[1] = verts[faces[i][1]][1];
            tempVert2[2] = verts[faces[i][1]][2];
            
            tempVert3[0] = verts[faces[i][2]][0];
            tempVert3[1] = verts[faces[i][2]][1];
            tempVert3[2] = verts[faces[i][2]][2];
            
            
            // calculate normal and normalize!
                
            // find two vectors on the poly
            double [] vect1 = new double [3];
            double [] vect2 = new double [3];
                
            vect1[0] = tempVert1[0] - tempVert2[0];
            vect1[1] = tempVert1[1] - tempVert2[1];
            vect1[2] = tempVert1[2] - tempVert2[2];
                
            vect2[0] = tempVert1[0] - tempVert3[0];
            vect2[1] = tempVert1[1] - tempVert3[1];
            vect2[2] = tempVert1[2] - tempVert3[2];
                
            // cross product
            /*normalVect[0] = (vect1[1]*vect2[2]) - (vect1[2]*vect2[1]);
            normalVect[1] = (vect1[2]*vect2[0]) - (vect1[0]*vect2[2]);
            normalVect[2] = (vect1[0]*vect2[1]) - (vect1[1]*vect2[0]);
                
            // normalize
            double magnitude = Math.sqrt(normalVect[0]*normalVect[0] + normalVect[1]*normalVect[1] + normalVect[2]*normalVect[2]);
            normalVect[0] = normalVect[0]/magnitude;
            normalVect[1] = normalVect[1]/magnitude;
            normalVect[2] = normalVect[2]/magnitude;
            */
            
	    // create a new Triangle and add it to the list
            list[i] = new Triangle(tempVert1,tempVert2,tempVert3, SimulationParams.sim_p);
        }
        return list;
    }
    
    /*
     * Concatenates two lists of Triangles.
     * Used when more than one object present in the VRML file
     *
     * @param a the first array of Triangles
     * @param b the second array of Triangles
     * @return a single array of Triangles containing the elements of a and b
     */
    private Triangle [] conCatList(Triangle [] a, Triangle []b) {
        if(a!=null && b!=null) {
            Triangle [] temp = new Triangle[a.length + b.length];
            
            for(int i=0;i<a.length;i++)
                temp[i] = a[i];
            for(int i=a.length;i<(a.length+b.length);i++)
                temp[i] = b[i-a.length];
            return temp;
        } else if (a==null)
            return b;
        else
            return a;
    }

    /*
     * Clears the arrays and resets all the variables to a default state
     * ready for the next object/file
     */
    private void resetVariables() {
        inPoints = false;
        inFaces = false;
        inNorms = false;
        inFaceNorms = false;
        
        verts = new double [ARRAY_INC_SIZE][3];
        normals = new double [ARRAY_INC_SIZE][3];
        faces = new int [ARRAY_INC_SIZE][3];
        faceNorms = new int [ARRAY_INC_SIZE][3];
        vertsArrayInd = 0;
        normalArrayInd = 0;
        faceInds = 0;
        faceNormInds = 0;
        Triangle [] triangleList = null;
        
        // since all arrays are the same size to start with, do all init in one loop!
        for(int i=0;i<verts.length;i++) {
            for(int j=0;j<3;j++) {
                verts[i][j]=0.0;
                normals[i][j]=0.0;
                faces[i][j]=0;
                faceNorms[i][j]=0;
            }
        }
    }
    
    public static void main(String [] args) {
        VRMLReader vrTemp = VRMLReader.getVRMLReader();
        Triangle [] tr = vrTemp.loadFile(args[0]);
        
        System.out.println("num triangles: " + tr.length + "\n\n");
        
        for(int i = 0;i<tr.length;i++)
            System.out.println((i+1) + ": " + tr[i]);
        
    }
}
