package tractography;

import data.*;
import imaging.*;
import inverters.*;
import misc.DT;
import numerics.*;

import java.util.Random;

/**
 * Each call to #getPDs(int, int, int) returns a new bootstrap estimate of the PDs. 
 * The bootstrap data is inverted using the specified one and two tensor index. 
 *
 *
 * @version $Id: BS_DT_TractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public abstract class BS_DT_TractographyImage extends BootstrapTractographyImage {
 
    private final DiffusionInversion linearDTInversion;

    private final DiffusionInversion oneDTInversion;
    private final DiffusionInversion twoDTInversion;
  
    
  
    /**
     * Construct an image.
     * 
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vc the voxel classification.
     * @param indices the inversion indices for the two-tensor and tensor reconstruction, in that order.
     * If a two-tensor index is not supplied, the default is ModelIndex.POSPOS.
     * @param imPars the imaging scheme for the data.
     * @param r a source of random numbers. The java.util.Random class is not recommended, use
     * tools.MTRandom instead.
     *
     * @see apps.ModelFit
     *
     */
    public BS_DT_TractographyImage(int[] dataDims, double[] voxelDims, int[][][] vc, 
				   ModelIndex[] indices, Scheme imPars, Random r) {

	super(dataDims, voxelDims, 3, vc, imPars, r);


	if (indices.length == 1) {
	    indices = new ModelIndex[] {ModelIndex.POSPOS, indices[0]};
	}

        linearDTInversion = DT_Inversion.getIndexedDT_Inversion(ModelIndex.LDT, imPars);

        oneDTInversion = DT_Inversion.getIndexedDT_Inversion(indices[1], imPars);
        twoDTInversion = new TwoTensorInversion(imPars, indices[0], indices[1]);

    }


  

    /**
     * Gets a new bootstrap estimate at every call.
     * 
     * Gets the eigenvectors for the voxel in the order  <code>{e1, e2, e3, e1...}</code>.
     *
     */
    public final Vector3D[] getEigenvectors(int i, int j, int k) {
        setVectors(i,j,k);
	return vectors[i][j][k];
    }
    
    
 
    protected void setVectors(int i, int j, int k) {

	double[] voxelData = nextBootstrapSample(i,j,k);

        DT[] tensors = null;

        vectors[i][j][k] = new Vector3D[3*numPDs[i][j][k]];
        
  
        // don't bother with background
        if (numPDs[i][j][k] == 0) {
            return;
        }
        
	if (numPDs[i][j][k] == 1) {
	    
	    double[] params = oneDTInversion.invert(voxelData);
	    
            if (params[0] == 2.0) {
                System.err.println("DT inversion failed in voxel " + i + " " + j + " " + k + 
                                   ". Trying linear inversion");
                params = linearDTInversion.invert(voxelData);
            }

	    tensors = new DT[1];

            tensors[0] = new DT(params[2], params[3], params[4], params[5], params[6], params[7]);
	    
	}
	else if (numPDs[i][j][k] == 2) {
	    double[] params = twoDTInversion.invert(voxelData);
	    
	    tensors = new DT[2];

            tensors[0] = new DT(params[4], params[5], params[6], params[7], params[8], params[9]);

            tensors[1] = new DT(params[11], params[12], params[13], 
                                         params[14], params[15], params[16]);
            
	}
	else {
	    throw new UnsupportedOperationException("Can't invert data higher than order 4");
	}

          
        for (int t = 0; t < tensors.length; t++) {
            
            double[][] seig = tensors[t].sortedEigenSystem();
            
            for (int v = 0; v < 3; v++) {
                vectors[i][j][k][v + 3*t] = new Vector3D(seig[1][v], seig[2][v], seig[3][v]);
            }

        }
    }


    /**
     * @return the next bootstrap sample of data.
     *
     */
    protected abstract double[] nextBootstrapSample(int i, int j, int k);




}
