package tractography;

import numerics.Point3D;
import java.util.ArrayList;


/**
 * A free form voxel-based ROI for tractography. The ROI consists of labelled regions.
 * The regions are labelled as integers from 1 upwards. 
 *
 * @author Philip Cook
 * @version $Id: FreeFormROI.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public final class FreeFormROI implements RegionOfInterest {

    private final double xVoxelDim;
    private final double yVoxelDim;
    private final double zVoxelDim;

    private final int xDataDim;
    private final int yDataDim;
    private final int zDataDim;

    private final short[][][] data;

    private final short minRegionIndex;
    private final short maxRegionIndex;

    private final int numberOfRegions;


    /**
     * Create an ROI from an integer volume. Data dimensions are taken from the size of the array.
     *  
     * @param voxelDims voxel dimensions, in mm.
     */
    public FreeFormROI(short[][][] roi, double[] voxelDims) {
	this(roi, voxelDims[0], voxelDims[1], voxelDims[2]);
    }


    /**
     * Create an ROI from an integer volume. Data dimensions are taken from the size of the array.
     * Voxel dimensions should be in mm.
     *  
     */
    public FreeFormROI(short[][][] roi, double xVoxDim, double yVoxDim, double zVoxDim) {

	data = roi;

	xDataDim = data.length;
	yDataDim = data[0].length;
	zDataDim = data[0][0].length;
	
        short tmpMin = Short.MAX_VALUE;
        short tmpMax = 0;

        boolean[] gotRegion = new boolean[Short.MAX_VALUE];

	int numRegions = 0;

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {

		    short intensity = data[i][j][k];

		    if (intensity > tmpMax) {
			tmpMax = intensity;
		    }
		    if (intensity > 0 && intensity < tmpMin) {
			tmpMin = intensity;
		    }
		    
		    if (intensity > 0 && !gotRegion[intensity]) {
			gotRegion[intensity] = true;
			numRegions++;
		    }
		}
	    }
	}

	numberOfRegions = numRegions;
	
	minRegionIndex = tmpMin;
        maxRegionIndex = tmpMax;
	
	xVoxelDim = xVoxDim;
	yVoxelDim = yVoxDim;
	zVoxelDim = zVoxDim;
	
	
    }


    /**
     * Create an ROI from an integer volume. Only the voxels matching the specified index
     * are included in the ROI.
     *  
     */
    public FreeFormROI(short[][][] roi, short index, double[] voxelDims) {
	this(roi, index, voxelDims[0], voxelDims[1], voxelDims[2]);	
    }


    /**
     * Create an ROI from an integer volume. Only the voxels matching the specified index
     * are included in the ROI.
     *  
     */
    public FreeFormROI(short[][][] roi, short index, double xVoxDim, double yVoxDim, double zVoxDim) {

	xDataDim = roi.length;
	yDataDim = roi[0].length;
	zDataDim = roi[0][0].length;

	data = new short[xDataDim][yDataDim][zDataDim];
        
	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    if (roi[i][j][k] == index) {
			data[i][j][k] = index;
		    }
		}
	    }
	}
	
        minRegionIndex = index;
        maxRegionIndex = index;

	xVoxelDim = xVoxDim;
	yVoxelDim = yVoxDim;
	zVoxelDim = zVoxDim;

	numberOfRegions = 1;
	
    }


    /**
     * Get all regions in the volume with a nonzero index. This method can consume a large amount of
     * memory if there are many ROIs.
     *
     *
     */
    public static FreeFormROI[] getRegions(short[][][] roi, double xVoxDim, double yVoxDim, 
                                           double zVoxDim) {
        
        short minIndex = Short.MAX_VALUE;
        short maxIndex = 0;

    	int xDataDim = roi.length;
	int yDataDim = roi[0].length;
	int zDataDim = roi[0][0].length;
     
               
	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
		    if (roi[i][j][k] < minIndex && roi[i][j][k] > 0) {
                        minIndex = roi[i][j][k];
		    }
                    if (roi[i][j][k] > maxIndex) {
                        maxIndex = roi[i][j][k];                        
                    }
		}
	    }
	}


	if (maxIndex == 0) {
	    // no ROIs
	    return new FreeFormROI[0];
	}
        
        boolean[] gotRegion = new boolean[maxIndex - minIndex + 1];

        int numRegions = 0;

        for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
                    if (roi[i][j][k] > 0) {
                        if ( !gotRegion[ roi[i][j][k] - minIndex ]) {
                            numRegions++;
                            gotRegion[ roi[i][j][k] - minIndex ] = true;
                        }
                        
                    }
                }
            }
        }

        FreeFormROI[] regions = new FreeFormROI[numRegions];
        int nextRegion = 0;
        
        for (int r = 0; r < gotRegion.length; r++) {
            if (gotRegion[r]) {
                regions[nextRegion++] = new FreeFormROI(roi, (short)(minIndex + r), xVoxDim, yVoxDim, zVoxDim);
            }
        }
        
        return regions;

    }


    /**
     * Tests to see if a specific voxel is in this region.
     *
     * @param x the x index of the voxel
     * @param y the y index of the voxel
     * @param z the z index of the voxel
     * @return true if the voxel is within the region bounds; false otherwise
     */
    public boolean containsVoxel(int x, int y, int z) {
	return (data[x][y][z] > 0);
    }


    /**
     * Tests to see if a point measured in MM coordinates is in this region.
     *
     * @param point the point to test
     * @return true if the point is within the region bounds; false otherwise
     */
    public boolean containsMMPoint(Point3D point) {

	int xC = (int)(point.x / xVoxelDim);
	int yC = (int)(point.y / yVoxelDim);
	int zC = (int)(point.z / zVoxelDim);
	
	return (data[xC][yC][zC] > 0);

    }


    /**
     * @return all the voxels in the ROI.
     */
    public Voxel[] getVoxels() {

	ArrayList<Voxel> list = new ArrayList<Voxel>(100);
	
	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {

		    if (data[i][j][k] > 0) { 
			list.add(new Voxel(i,j,k));
		    }

		}
	    }
	}


	Voxel[] voxels = new Voxel[list.size()];
	
        return list.toArray(voxels);


    }


    /**
     * @return a point at the centre of every voxel in this region.
     */
    public Point3D[] getSeedPoints() {

	ArrayList<Point3D> list = new ArrayList<Point3D>(100);
	
	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {

		    if (data[i][j][k] > 0) { 
			list.add(new Point3D((i + 0.5) * xVoxelDim,
					 (j + 0.5) * yVoxelDim,
					     (k + 0.5) * zVoxelDim));
		    }

		}
	    }
	}


	Point3D[] points = new Point3D[list.size()];
	
        return list.toArray(points);

    }


    public int numberOfRegions() {
	return numberOfRegions;
    }


    /**
     * Get a specific ROI. 
     *
     * @param index the index of the required region, 
     * where <code>-1 < index < numberOfRegions() </code>.
     */
    public RegionOfInterest getRegion(int index) {
	
	if (index >= 0 && index < maxRegionIndex - minRegionIndex + 1) { 
	    return new FreeFormROI(data, (short)(minRegionIndex + index), xVoxelDim, yVoxelDim, zVoxelDim);
	}
	else {
	    throw new IllegalArgumentException("ROI index out of bounds. This ROI contains " + 
					       numberOfRegions() + " regions");
	}
    }

}
