package tractography;

import numerics.*;

import java.util.Random;

/**
 * Does Probabilistic tractography using neighbour-choice interpolation.
 *
 * @version $Id $
 * @see ProbNeighbourChoiceInterpolator
 * @author  Philip Cook
 * 
 */
public class NC_ProbFibreTracker extends EulerFibreTracker {
   
    protected final NeighbourChoiceInterpolator interpProbRef;

    /**
     * @param image the image within which the tracking will take place.
     * @param stepLength the distance (in mm) to move before performing the next interpolation. 
     * @param ipThresh the minimum absolute value of the cosine of the
     * angle between the tracking directions after traversing one slice thickness. So if slice 
     * thickness is 2.5 mm, and step size is 0.25mm, then the curvature will be assessed every 
     * 10 steps.
     *
     * @param ran the Random for the randomizer and interpolator.
     */
    public NC_ProbFibreTracker(TractographyImage image, double stepLength, double ipThresh, Random ran) {
	
	super(image, stepLength, ipThresh);
	interpolator = new NeighbourChoiceInterpolator(image, ran);
	interpProbRef = (NeighbourChoiceInterpolator)(interpolator);
    }


    /** 
     * Track a path from a single seed point,
     * and wrap result in a #TractCollection.
     * @param seedPos the point (in mm) to track from.
     *  
     */
    public TractCollection trackFromSeed(Point3D seedPos) {
	TractCollection tc = super.trackFromSeed(seedPos);
	interpProbRef.resetRandomization();
	return tc;
    }


    /** 
     * Track a path from a single seed point, following a specified PD.
     *
     * @param seedPos the point (in mm) to track from.
     * @param pd the index of the PD to follow for the initial step.
     *  
     */
    public TractCollection trackFromSeed(Point3D seedPos, int pd) {
	TractCollection tc = super.trackFromSeed(seedPos, pd);
	interpProbRef.resetRandomization();
	return tc;
    }

   
}
