package tractography;

import misc.*;
import numerics.Point3D;

import java.util.ArrayList;
import java.io.*;


/**
 * An ROI for tractography. The ROI consists of a list of points in mm space. There is no restriction
 * on the number of points per voxel.
 * 
 *
 * @author Philip Cook
 * @version $Id: PointListROI.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public final class PointListROI implements RegionOfInterest {

    private final FreeFormROI voxelROI;

    private final Point3D[] points;
    private final Voxel[] voxels;   
    
    private final double xVoxelDim;
    private final double yVoxelDim;
    private final double zVoxelDim;
    
    private final int xDataDim;
    private final int yDataDim;
    private final int zDataDim;


    /**
     * Create an ROI from a list of points in mm coordinates.
     *  
     */
    public PointListROI(Point3D[] pointList, int xDataDim, int yDataDim, int zDataDim,
			double xVoxelDim, double yVoxelDim, double zVoxelDim) { 

	this(pointList, new int[] {xDataDim, yDataDim, zDataDim},
	     new double[] {xVoxelDim, yVoxelDim, zVoxelDim});

    } 

    /**
     * Create an ROI from a list of points in mm coordinates.
     *  
     */
    public PointListROI(Point3D[] pointList, int[] dataDims, double[] voxelDims) {    

	xDataDim = dataDims[0];
	yDataDim = dataDims[1];
	zDataDim = dataDims[2];

	xVoxelDim = voxelDims[0];
	yVoxelDim = voxelDims[1];
	zVoxelDim = voxelDims[2];
    	
	points = new Point3D[pointList.length];

	System.arraycopy(pointList, 0, points, 0, points.length);

	// voxels with seed points
	short[][][] roi = new short[xDataDim][yDataDim][zDataDim];
	
	voxels = new Voxel[points.length];
	
	for (int i = 0; i < points.length; i++) {
	    
	    int xC = (int)(points[i].x / xVoxelDim);
	    int yC = (int)(points[i].y / yVoxelDim);
	    int zC = (int)(points[i].z / zVoxelDim);

	    voxels[i] = new Voxel(xC, yC, zC);

	    roi[xC][yC][zC] = 1;
	}

	voxelROI = new FreeFormROI(roi, voxelDims);
    }



    /**
     * Create an ROI from a list of points, in mm coordinates, stored as text in the format
     * <code>x y z\nx y z\n...</code>.
     *  
     */
    public static PointListROI readPoints(String file, int[] dataDims, double[] voxelDims) {

	
	Point3D[] pointList = null;

	ArrayList<Point3D> tmpPoints = new ArrayList<Point3D>(100);
	
	try {

	    InputStreamReader isr = new InputStreamReader(new FileInputStream(file));
	    
	    BufferedReader reader = new BufferedReader(isr);
	    
	    String line = reader.readLine();

	    readLines: 
	    while (line != null) {

		String[] elements = line.split("\\s+");

		if (elements.length < 3) {
		    //blank line
		    break readLines;
		}

		Point3D point = new Point3D(Double.parseDouble(elements[0]), 
					    Double.parseDouble(elements[1]),
					    Double.parseDouble(elements[2]));

		tmpPoints.add(point);

		line = reader.readLine();
	    }

	    pointList = new Point3D[tmpPoints.size()];
	
	    tmpPoints.toArray(pointList);

	    isr.close();
	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}

	return new PointListROI(pointList, dataDims, voxelDims);
	
    }


    /**
     * Tests to see if a specific voxel is in this region.
     *
     * @param x the x index of the voxel
     * @param y the y index of the voxel
     * @param z the z index of the voxel
     * @return true if the voxel is within the region bounds; false otherwise
     */
    public boolean containsVoxel(int x, int y, int z) {
	return voxelROI.containsVoxel(x, y, z);
    }


    /**
     * Tests to see if <code>point</code> is within a voxel that contains one or more seed
     * points. Note that this does <b>not</b> test <code>point</code> for equality with any of 
     * the seed points in the ROI.
     *
     * @param point the point to test
     * @return true if the point is within the region bounds; false otherwise
     */
    public boolean containsMMPoint(Point3D point) {
	return voxelROI.containsMMPoint(point);
    }


    /**
     * @return all the voxels in the ROI. The array has a one to one correspondence 
     * to the array returned by <code>getSeedPoints()</code>, therefore there may be
     * duplicate voxels in the array.
     */
    public Voxel[] getVoxels() {

	Voxel[] copy = new Voxel[voxels.length];
	
	System.arraycopy(voxels, 0, copy, 0, voxels.length);
	
	return copy;
    	
    }



    /**
     * @return a point at the centre of every voxel in this region.
     */
    public Point3D[] getSeedPoints() {

	Point3D[] copy = new Point3D[points.length];

	System.arraycopy(points, 0, copy, 0, points.length);

	return copy;
    }



    /**
     * @return 1.
     */
    public int numberOfRegions() {
	return 1;
    }



    /**
     * There is only one ROI in this class, so the index must be 1 and the 
     * returned reference is <code>this</code>.
     */
    public RegionOfInterest getRegion(int index) {
	
	if (index == 0) { 
	    return this;
	}
	else {
	    throw new IllegalArgumentException("ROI index out of bounds. This ROI contains " + 
					       numberOfRegions() + " regions");
	}
    }

}
