package tractography;

import data.*;
import misc.*;
import numerics.*;

import java.io.*;
import java.util.logging.*;
import java.util.zip.*;

/**
 * Provides an interface for input of Tract objects.
 *
 * @author Philip Cook
 * @version $Id: TractSource.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public class TractSource {


    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.tractography.TractSource");

    private final DataInputStream din;

    private final boolean inputIsVoxels;

    private Tract nextTract = null;

    private boolean reachedEndOfFile;
    private boolean noMoreTracts;
  

    private final double xVoxelDim;
    private final double yVoxelDim;
    private final double zVoxelDim;


    /**
     * Construct a source from the file or stdin.
     * 
     * @param filename name of the file. If <code>null</code>, tracts are read from stdin.
     *
     * @param tractType either "raw" or "voxels".
     *
     * @param voxelDims only used if input is "voxels"; in which case the returned tracts have
     * a point at the centre of each voxel, plus two extra ones to determine the exact tangent at 
     * the seed point.
     *
     * @see tractography.VoxelList#toTract
     */
    public TractSource(String filename, String tractType, double[] voxelDims) {
	this(filename, tractType, voxelDims[0], voxelDims[1], voxelDims[2]);
    }
    

    public TractSource(String filename, String tractType, double xVoxelDim, double yVoxelDim,
		       double zVoxelDim) {

	int bufferSize = OutputManager.FILEBUFFERSIZE;

	this.xVoxelDim = xVoxelDim;
	this.yVoxelDim = yVoxelDim;
	this.zVoxelDim = zVoxelDim;

	if (tractType.equals("voxels")) {
	    inputIsVoxels = true;
	}
	else if (tractType.equals("raw")) {
	    inputIsVoxels = false;
	}
	else {
	    throw new LoggedException("Unrecognized input model : " + tractType);
	}

	try {
	    if (filename == null) {
		din = new DataInputStream(new BufferedInputStream(System.in, bufferSize));
        	logger.info("reading data from standard input");
	    }
	    else {
		
		if (filename.endsWith(".gz")) {
		    FileInputStream fin = new FileInputStream(filename);
		    
		    din = 
			new DataInputStream(new GZIPInputStream(new BufferedInputStream(fin, bufferSize)));
		}
		else {
		    FileInputStream fin = new FileInputStream(filename);
		    
		    din = new DataInputStream(new BufferedInputStream(fin, bufferSize));
		    
		}
	    }
	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}

	init();

	
    }

    private void init() {
        reachedEndOfFile = false;
        noMoreTracts = false;

        // Read in the first voxel ready for the first data request.
        try {
            readNextTract();
	    
	    if (reachedEndOfFile) {
		noMoreTracts = true;
		throw new LoggedException("Input contains no tracts");
	    }

        }
        catch (Exception e) {
            throw new LoggedException(e);
        }

    }

    private void readNextTract() {
	if (inputIsVoxels) {
	    nextTract = readTractFromVoxels();
	}
	else {
	    nextTract = readTractFromRaw();
	}
    }


    private Tract readTractFromVoxels() {

	boolean gotWholeTract = true;

	try {
	    
	    
	    short numVoxels = din.readShort();
	    
	    gotWholeTract = false;
	    
	    short seedPointIndex = din.readShort();

	    Vector3D tangent = new Vector3D(din.readShort(), din.readShort(), din.readShort());
	    Vector3D negTangent = new Vector3D(din.readShort(), din.readShort(), din.readShort());
	    
	    Voxel[] voxels = new Voxel[numVoxels];
	    
	    for (int v = 0; v < numVoxels; v++) {
		voxels[v] = new Voxel(din.readShort(), din.readShort(), din.readShort());
	    }
	    
	    gotWholeTract = true;

	    return new VoxelList(voxels, seedPointIndex, xVoxelDim, yVoxelDim, zVoxelDim, tangent, 
				 negTangent).toTract();
	}
	catch (Exception e) {
	    if (e instanceof EOFException) {
		if (!gotWholeTract) {
		    throw new LoggedException("EOF before whole tract was read");
		}
		reachedEndOfFile = true;
		return null;
	    }
	    else {
		throw new LoggedException("Unexpected " + e + " . Check that input is in VOXEL format");
	    }
	}


    }

    

    private Tract readTractFromRaw() {
	
	boolean gotWholeTract = true;

	try {

	    Tract zeroToSeed = new Tract(100, 100.0); // points 0 to seedPointIndex
	    Tract seedToEnd = new Tract(100, 100.0); // points seedPointIndex to end
	
	    int numPoints = (int)din.readFloat();  // should get EOF here after last tract

	    gotWholeTract = false;
	
	    int seedPointIndex = (int)din.readFloat();
	
	    Point3D[] points = new Point3D[numPoints];
	
	    for (int p = 0; p < numPoints; p++) {
		double x = din.readFloat();
		double y = din.readFloat();
		double z = din.readFloat();
	    
		points[p] = new Point3D(x,y,z);
	    }
	    gotWholeTract = true;
	
	    for (int p = seedPointIndex; p >= 0; p--) {
		zeroToSeed.addPoint(points[p]);
	    }
	
	    for (int p = seedPointIndex; p < numPoints; p++) {
		seedToEnd.addPoint(points[p]);
	    }
	
	    seedToEnd.joinTract(zeroToSeed);

	    return seedToEnd;

	}
	catch (Exception e) {
	    if (e instanceof EOFException) {
		if (!gotWholeTract) {
		    throw new LoggedException("EOF before whole tract was read");
		}
		reachedEndOfFile = true;
		return  null;
	    }
	    else {
		throw new LoggedException("Unexpected " + e + " . Check that input is in RAW format");
	    }
	}
	
    }


    
    /**
     * 
     *
     *  @throws LoggedException if more tracts are read than the file contains.
     */
    public Tract nextTract() {

	if (noMoreTracts) {
	    throw new LoggedException("No more tracts in input");
	}

	Tract current = nextTract;
	
	readNextTract();

	if (reachedEndOfFile) {
            noMoreTracts = true;
        }

	return current;
    }


    /**
     * @return true if there are more tracts. When this returns false, further calls
     *
     */
    public boolean more() {
	return !noMoreTracts;
    }



}
