package tractography;

import data.*;
import imaging.*;
import misc.DT;
import numerics.*;


/**
 * Superclass for tractography images. Images store the data for tracking, including PICo PDF parameters.
 *
 * @version $Id: TractographyImage.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * @author  Philip Cook
 * 
 */
public class TractographyImage {

    
    /** Size of image in x dimension. */
    protected final int xDataDim;

    /** Size of image in y dimension. */
    protected final int yDataDim;

    /** Size of image in z dimension. */
    protected final int zDataDim;

    /** Size (mm) of x voxel length. */
    protected final double xVoxelDim;

    /** Size (mm) of y voxel length. */
    protected final double yVoxelDim;

    /** Size (mm) of z voxel length. */
    protected final double zVoxelDim;
    
    /** Number of PDs in each voxel */
    protected final int[][][] numPDs;

    // list of unit vectors. These may be just PDs, or all eigenvectors
    protected final Vector3D[][][][] vectors; 

    protected boolean[][][] isotropicMask = null;

    protected final int vecsPerPD;
   

    /**
     * Basic constructor initialises dimensions only.
     *
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vectorsPerPD the number of vectors the image contains per principal direction. 
     * 
     */
    public TractographyImage(int[] dataDims, double[] voxelDims, int vectorsPerPD) {
	
	xVoxelDim = voxelDims[0];
	yVoxelDim = voxelDims[1];
	zVoxelDim = voxelDims[2];

	xDataDim = dataDims[0];
	yDataDim = dataDims[1];
	zDataDim = dataDims[2];

	vecsPerPD = vectorsPerPD;

	vectors = new Vector3D[xDataDim][yDataDim][zDataDim][];
	numPDs = new int[xDataDim][yDataDim][zDataDim];

    }


    /**
     * Matlab constructor.
     *
     * @param vectors vector data. If PDs only, there should be one vector per voxel. If eigen systems,
     * they should be ordered {e1, e2, e3}.
     * @param dataDims array of data dimensions {xDataDim, yDataDim, zDataDim}.
     * @param voxelDims array of voxel dimensions (in mm) {xVoxelDim, yVoxelDim, zVoxelDim}.
     * @param vectorsPerPD the number of vectors the image contains per principal direction. 
     * 
     */
    public TractographyImage(Vector3D[][][][] vectors, double[] voxelDims, int vectorsPerPD) {
	
	xVoxelDim = voxelDims[0];
	yVoxelDim = voxelDims[1];
	zVoxelDim = voxelDims[2];

	xDataDim = vectors.length;
	yDataDim = vectors[0].length;
	zDataDim = vectors[0][0].length;
	
	vecsPerPD = vectorsPerPD;

	this.vectors = vectors;
	numPDs = new int[xDataDim][yDataDim][zDataDim];

        for (int k = 0; k < zDataDim; k++) {
            for (int j = 0; j < yDataDim; j++) {
                for (int i = 0; i < xDataDim; i++) {
		    numPDs[i][j][k] = vectors[i][j][k].length;
		}
	    }
	}	
	
	computeIsotropicMask();
    }



    /**
     * @return the princpal direction or directions in the voxel.
     */
    public Vector3D[] getPDs(int i, int j, int k) {
	

	if (numPDs[i][j][k] == 1) {
	    return new Vector3D[] {vectors[i][j][k][0]};
	}
        else if (numPDs[i][j][k] == 0) {
            return new Vector3D[] {};
        }
	else { // if (numPDs[i][j][k] == 2) {

            Vector3D[] pds = new Vector3D[numPDs[i][j][k]];
            
            for (int x = 0; x < numPDs[i][j][k]; x++) {
                pds[x] = vectors[i][j][k][x * vecsPerPD];
            }

            return pds;
	}
 
    }


    /**
     * @return the princpal direction or directions in the voxel, given an estimated local fiber orientation.
     * If the PDs in the voxel are not dependent on the fiber orientation, then this method returns the
     * same thing as getPDs(int, int, int).
     *
     */
    public Vector3D[] getPDs(int i, int j, int k, Vector3D fibreOrientation) {
	return getPDs(i,j,k);
    }



    /**
     * @return the number of PDs in this voxel.
     *
     */
    public final int numberOfPDs(int i, int j, int k) {
        return numPDs[i][j][k];
    }
   

    /** Size of image in x dimension. */
    public final int xDataDim() {
	return xDataDim;
    }

    /** Size (mm) of x voxel length. */
    public final double xVoxelDim() {
	return xVoxelDim;
    }

    /** Size of image in y dimension. */
    public final int yDataDim() {
	return yDataDim;
    }

    /** Size (mm) of y voxel length. */
    public final double yVoxelDim() {
	return yVoxelDim;
    }

    /** Size of image in z dimension. */
    public final int zDataDim() {
	return zDataDim;
    }

    /** Size (mm) of z voxel length. */
    public final double zVoxelDim() {
	return zVoxelDim;
    }

    /** Array of voxel dimensions in mm. */
    public final double[] getVoxelDims() {
	return new double[] {xVoxelDim, yVoxelDim, zVoxelDim};
    }

    /** Array of data dimensions. */
    public final int[] getDataDims() {
	return new int[] {xDataDim, yDataDim, zDataDim};
    }


    /**
     * Computes mask for tracking. A voxel in the mask is <code>true</code> 
     * if streamlines should terminate on entry to the voxel. This method creates a brain / background
     * mask. A voxel is isotropic if the number of PDs in the voxel is zero.
     * 
     *
     */
    public final void computeIsotropicMask() {

	isotropicMask = new boolean[xDataDim][yDataDim][zDataDim];

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
			
                    if (numPDs[i][j][k] == 0) {
                        isotropicMask[i][j][k] = true;
                    }
                    
                }
            }
        }
		
    }


    /**
     * Computes mask for tracking. A voxel in the mask is <code>true</code> 
     * if streamlines should terminate on entry to the voxel. Background voxels are always 
     * <code>true</code>, voxels with multiple fibres are always <code>false</code>.
     *
     * @param anisMap an image containing the some quantitative measurement of the 
     * anisotropy of the diffusion in the voxel. 
     * @param threshold voxels with one PD and anisotropy below this value will be set 
     * to <code>true</code>.
     * 
     */
    public final void computeIsotropicMask(double[][][] anisMap, double threshold) {

        isotropicMask = new boolean[xDataDim][yDataDim][zDataDim];

	for (int k = 0; k < zDataDim; k++) {
	    for (int j = 0; j < yDataDim; j++) {
		for (int i = 0; i < xDataDim; i++) {
					
		    if (numPDs[i][j][k] == 0) {
			isotropicMask[i][j][k] = true;
		    }
		    else if (numPDs[i][j][k] > 1) {
			isotropicMask[i][j][k] = false;
		    }
		    else if (anisMap[i][j][k] < threshold) {
			isotropicMask[i][j][k] = true;
		    }
		    
		}
	    }
	}
	
    }


    /**
     * Gets a boolean mask image for tracking. A voxel in the mask is <code>true</code> 
     * if streamlines should terminate on entry to the voxel.
     */
    public final boolean[][][] getIsotropicMask() {
	return isotropicMask;
    }




}
