package edu.jhu.ece.iacl.plugins.hardi;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import edu.jhu.bme.smile.commons.textfiles.TextFileReader;
import edu.jhu.ece.iacl.algorithms.hardi.QBall;
import edu.jhu.ece.iacl.io.CubicVolumeReaderWriter;
import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.io.ModelImageReaderWriter;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.structures.image.ImageDataFloat;

public class QBallProjection extends ProcessingAlgorithm{ 

	/****************************************************
	 * Input Parameters 
	 ****************************************************/
	private ParamFileCollection sphericalHarmonicCoeffsReal;	// SLAB-enabled A 4D volume with one tensor estimated per pixel
	private ParamFileCollection sphericalHarmonicCoeffsImag;	// SLAB-enabled A 4D volume with one tensor estimated per pixel
	private ParamFile gradsTable;		// .grad or .dpf file with a list of gradient directions
	private ParamBoolean outputImag;
	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamFileCollection reconReal;	// SLAB-enabled A 4D volume with one tensor estimated per pixel
	private ParamFileCollection reconImag;	// SLAB-enabled A 4D volume with one tensor estimated per pixel

	private static final String rcsid =
		"$Id: QBallProjection.java,v 1.5 2009/03/27 01:28:44 bennett Exp $";
	private static final String cvsversion =
		"$Revision: 1.5 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");


	protected void createInputParameters(ParamCollection inputParams) {

		/****************************************************
		 * Step 1. Set Plugin Information 
		 ****************************************************/
		inputParams.setName("Q-Ball: SH Projection");
		inputParams.setLabel("Q-Ball: SH Projection");	
		inputParams.setCategory("Modeling.Diffusion.QBall");
		inputParams.setPackage("IACL");

		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://sites.google.com/site/jhupami/");
		info.add(new AlgorithmAuthor("Bennett Landman","landman@jhu.edu","http://sites.google.com/site/bennettlandman/"));
		info.setDescription("Log-linear minium mean squared error tensor estimation.");
		info.setAffiliation("Johns Hopkins University, Department of Biomedical Engineering");
		info.setVersion(revnum);	
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Step 2. Add input parameters to control system 
		 ****************************************************/
		sphericalHarmonicCoeffsReal = new ParamFileCollection("Spherical Harmonic Coefficients",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		sphericalHarmonicCoeffsReal.setName("QBall-RealPart");
		inputParams.add(sphericalHarmonicCoeffsReal);
		sphericalHarmonicCoeffsImag = new ParamFileCollection("Spherical Harmonic Coefficients",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		sphericalHarmonicCoeffsImag.setName("QBall-ImagPart");
		inputParams.add(sphericalHarmonicCoeffsImag);
		inputParams.add(gradsTable=new ParamFile("Directions Along Which to Project the SH Model",new FileExtensionFilter(new String[]{"grad","dpf"})));		
		inputParams.add(outputImag=new ParamBoolean("Output Imaginary Part of Reconstuction?",false));
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system 
		 ****************************************************/
		reconReal = new ParamFileCollection("QBall Projection Real Part",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		reconReal.setName("QBallProj-Real");		
		outputParams.add(reconReal);			
		reconImag = new ParamFileCollection("QBall Projection Imaginary Part",new FileExtensionFilter(ModelImageReaderWriter.supportedFileExtensions));
		reconImag.setName("QBallProj-Imag");
		reconImag.setMandatory(false);
		outputParams.add(reconImag);			

	}
	protected void execute(CalculationMonitor monitor) {
		/****************************************************
		 * Step 1. Indicate that the plugin has started.
		 * 		 	Tip: Use limited System.out.println statements
		 * 			to allow end users to monitor the status of
		 * 			your program and report potential problems/bugs
		 * 			along with information that will allow you to 
		 * 			know when the bug happened.  
		 ****************************************************/
		System.out.println("Q-Ball Proj: Start");

		/****************************************************
		 * Step 2. Loop over input slabs
		 ****************************************************/
		List<File> shReal = sphericalHarmonicCoeffsReal.getValue();
		List<File> shImag = sphericalHarmonicCoeffsImag.getValue();

		CubicVolumeReaderWriter rw  = CubicVolumeReaderWriter.getInstance();
		ArrayList<File> realOutVols = new ArrayList<File>();
		ArrayList<File> imagOutVols = new ArrayList<File>();
		for(int jSlab=0;jSlab<shReal.size();jSlab++) {
			/****************************************************
			 * Step 2. Parse the input data 
			 ****************************************************/
			ImageData shRealData=rw.read(shReal.get(jSlab));
			String sourceName = shRealData.getName();
			ImageDataFloat shRealFloat=new ImageDataFloat(shRealData);
			shRealData.dispose();
			shRealData=null;
			ImageData shImagData=rw.read(shImag.get(jSlab));			
			ImageDataFloat shImagFloat=new ImageDataFloat(shImagData);
			shImagData.dispose();
			shImagData=null;


			float [][]grads=null;
			TextFileReader text = new TextFileReader(gradsTable.getValue());
			try {
				grads  = text.parseFloatFile();
			} catch (IOException e) { 

				throw new RuntimeException("QBall: Unable to parse grad-file");
			}

			/****************************************************
			 * Step 3. Perform limited error checking 
			 ****************************************************/
			// If there are 4 columns in the gradient table, remove the 1st column (indecies)
			if(grads[0].length==4) {
				float [][]g2 = new float[grads.length][3];
				for(int i=0;i<grads.length;i++) 
					for(int j=0;j<3;j++)
						g2[i][j]=grads[i][j+1];
				grads=g2;
			}

			if(grads[0].length!=3)
				throw new RuntimeException("QBall: Invalid gradient table. Must have 3 or 4 columns.");

			int SHorder = QBall.getOrderFromNumberOfCoeff(shRealFloat.getComponents());

			/****************************************************
			 * Step 4. Run the core algorithm. Note that this program 
			 * 		   has NO knowledge of the MIPAV data structure and 
			 * 		   uses NO MIPAV specific components. This dramatic 
			 * 		   separation is a bit inefficient, but it dramatically 
			 * 		   lower the barriers to code re-use in other applications.  		  
			 ****************************************************/
			float [][][][]reconReal = new float[shRealFloat.getRows()][shRealFloat.getCols()][shRealFloat.getSlices()][grads.length];
			float [][][][]reconImag;
			if(outputImag.getValue())
				reconImag = new float[shRealFloat.getRows()][shRealFloat.getCols()][shRealFloat.getSlices()][grads.length];
			else 
				reconImag=null;

			QBall.projectSphericalHarmonics(reconReal,reconImag,
					shRealFloat.toArray4d(),shImagFloat.toArray4d(),
					grads,SHorder);			

			/****************************************************
			 * Step 5. Retrieve the image data and put it into a new
			 * 			data structure. Be sure to update the file information
			 * 			so that the resulting image has the correct
			 * 		 	field of view, resolution, etc.  
			 ****************************************************/

			ImageData  out= (new ImageDataFloat(reconReal)); reconReal=null;
			out.setHeader(shRealFloat.getHeader());
			shRealFloat.dispose();
			shRealFloat=null;			
			out.setName(sourceName+"_QBallProjReal");		
			File outputSlab = rw.write(out, getOutputDirectory());
			realOutVols.add(outputSlab);
			out.dispose();

			if(outputImag.getValue()) {
				out= (new ImageDataFloat(reconImag)); reconImag=null;
				out.setHeader(shImagFloat.getHeader());
				shImagFloat.dispose();
				shImagFloat=null;			
				out.setName(sourceName+"_QBallProjImag");		
				outputSlab = rw.write(out, getOutputDirectory());
				imagOutVols.add(outputSlab);
				out.dispose();
				out=null;		
			}

		}

		reconReal.setValue(realOutVols);
		if(outputImag.getValue())
			reconImag.setValue(imagOutVols);
		/****************************************************
		 * Step 6. Let the user know that your code is finished.  
		 ****************************************************/
		System.out.println("QBall: FINISHED");
	}
}
