package edu.jhu.ece.iacl.plugins.labeling.simulation;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import javax.vecmath.Point2i;

import edu.jhu.bme.smile.commons.textfiles.TextFileReader;
import edu.jhu.ece.iacl.io.CubicVolumeReaderWriter;
import edu.jhu.ece.iacl.io.FileExtensionFilter;
import edu.jhu.ece.iacl.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.structures.geom.GridPt;
import edu.jhu.ece.iacl.structures.image.ImageData;
import edu.jhu.ece.iacl.algorithms.manual_label.simulation.SurfaceRater;
import edu.jhu.ece.iacl.algorithms.manual_label.simulation.VolumeRater;

public class AlgorithmSimulateVolumeRater extends ProcessingAlgorithm {
	
	//input params
	private ParamVolume volParam;
	private ParamInteger numRaters;
	private ParamOption connectivity;
	private ParamFloat performance;
	
	private ParamOption typeErr;
	
	private ParamFileCollection confusionfile;
	private ParamFileCollection perimerrfile;
	private ParamFileCollection growprobfile;
	private ParamFile labelfile;
	private ParamFile boundaryfile;
	
	//output params
	private ParamFileCollection raterVols;
	
	//other useful objects
	private TextFileReader text;
	
	
	
	private static final String rcsid =
		"$Id: AlgorithmSimulateVolumeRater.java,v 1.3 2009/05/28 15:22:39 bogovic Exp $";
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = 
		cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = 
		"Simulate Volume Rater.";
	private static final String longDescription = "Simulates volume raters either by a confusion matrix or perimeter error";
	

	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(volParam=new ParamVolume("Truth Volume"));
		
		inputParams.add(labelfile=new ParamFile("Labels", new FileExtensionFilter(new String[]{"txt","csv"})));
		labelfile.setMandatory(false);
		inputParams.add(confusionfile=new ParamFileCollection("Confusion Matrix", new FileExtensionFilter(new String[]{"txt","csv"})));
		confusionfile.setMandatory(false);
		
		inputParams.add(performance=new ParamFloat("Rater Performance",0.0f,1.0f,0.9f));
		
		inputParams.add(boundaryfile=new ParamFile("Boundaries", new FileExtensionFilter(new String[]{"txt","csv"})));
		boundaryfile.setMandatory(false);
		inputParams.add(perimerrfile=new ParamFileCollection("Permimeter Boundary Error", new FileExtensionFilter(new String[]{"txt","csv"})));
		perimerrfile.setMandatory(false);
		inputParams.add(growprobfile=new ParamFileCollection("Boundary Grow Probabilty File", new FileExtensionFilter(new String[]{"txt","csv"})));
		growprobfile.setMandatory(false);
		
		inputParams.add(typeErr=new ParamOption("Error Type",new String[]{"Perimeter","Confusion"}));

		inputParams.add(numRaters=new ParamInteger("Number of Raters to Generate",1));
		inputParams.add(connectivity=new ParamOption("Connectivity",new String[]{"6","18","26"}));
		
		inputParams.setLabel("Simulate Volume Raters");
		inputParams.setName("simVolRaters");
		inputParams.setCategory("IACL.Labeling.Sim");

	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(raterVols=new ParamFileCollection("Rater Volumes"));
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		System.out.println("Starting");
		
		//set the output directory and create it if it doesn't exist
		File dir = new File(this.getOutputDirectory()+File.separator+edu.jhu.ece.iacl.utility.FileUtil.forceSafeFilename(this.getAlgorithmName()));
		try{
			if(!dir.isDirectory()){
				(new File(dir.getCanonicalPath())).mkdir();
			}
		}catch(IOException e){ e.printStackTrace(); }
		
		//get ready
		
		GridPt.Connectivity conn;
		if(connectivity.getIndex()==0){
			conn=GridPt.Connectivity.SIX;
		}else if(connectivity.getIndex()==1){
			conn=GridPt.Connectivity.EIGHTEEN;
		}else{
			conn=GridPt.Connectivity.TWENTYSIX;
		}
		
		int num = numRaters.getInt();
		float totproberr = performance.getFloat();
		ImageData truthvol = volParam.getImageData();
		
		/* Parse Labels*/ 
		int[] labels =null;
		if(labelfile.getValue()!=null){
			float[][] labellist =null;
			text = new TextFileReader(labelfile.getValue());
			try {
				labellist  = text.parseFloatFile();
			} catch (IOException e) { 
				throw new RuntimeException("Unable to parse grad-file");
			}
			//check the format of the label file and error check
			if(labellist.length>0 && labellist[0].length==1){
				labels = new int[labellist.length];
				for(int i=0; i<labellist.length; i++){
					labels[i]=(int)labellist[i][0];
				}
			}else if(labellist[0].length>0 && labellist.length==1){
				labels = new int[labellist[0].length];
				for(int i=0; i<labellist[0].length; i++){
					labels[i]=(int)labellist[0][i];
				}
			}else{
				System.err.println("Invalid label file");
			}
		}else{
			labels = VolumeRater.getLabelList(truthvol);
		}
		
		
		/* Parse Boundaries*/ 
		ArrayList<Point2i> bounds=null;
		int[][] boundarray = null;
		if(boundaryfile.getValue()!=null){
			text = new TextFileReader(boundaryfile.getValue());
			try{
				boundarray  = text.parseIntFile();
			}catch(IOException e){
				e.printStackTrace();
			}
			System.out.println(boundarray.length +", " + boundarray[0].length);
			if(boundarray.length>0 && boundarray[0].length==2){
				bounds = new ArrayList<Point2i>(boundarray.length);
				for(int i=0; i<boundarray.length; i++){
					bounds.add(new Point2i(boundarray[i][0],boundarray[i][1]));
				}
			}else if(boundarray[0].length>0 && boundarray.length==2){
				bounds = new ArrayList<Point2i>(boundarray[0].length);
				for(int i=0; i<boundarray[0].length; i++){
					bounds.add(new Point2i(boundarray[0][i],boundarray[1][i]));
				}
			}else{
				System.err.println("Invalid Boundary file - must be Nx2 or 2xN");
			}
		}
		
		/*Ensure the lengths of relevant file collections are the same*/
		int numFiles=1;
		if(typeErr.getIndex()==0){
			if(perimerrfile!=null && growprobfile !=null){
				if(perimerrfile.size()>0|| growprobfile.size()>0){
					if(perimerrfile.size()==growprobfile.size()){
						numFiles=perimerrfile.size();
					}else{
						System.err.println("Perimeter error parameters must have the same number of files input");
					}
				}else{
					System.err.println("All perimeter error parameters need file inputs");
				}
			}
		}else{
			if(confusionfile!=null && confusionfile.size()>0){
				numFiles=confusionfile.size();
			}
		}
		
		int N = labels.length; //number of labels
		
		System.out.println("NUMFILES: " +numFiles);
	
		ArrayList<File> vollist = null;
		if(numFiles==0){
			vollist = new ArrayList<File>(num);
		}else{
			vollist = new ArrayList<File>(num*numFiles);
		}
		
		
		for(int n=0; n<numFiles; n++){
			System.out.println("file: " +n);
			float[] perErr = null;
			float[] growProb = null;
			float[][] conf=null;
			String addtoname = "";
			if(typeErr.getIndex()==1){
				/* Parse Confusion Matrix */ 

				if(confusionfile.getValue(n)!=null){
					text = new TextFileReader(confusionfile.getValue(n));
					addtoname=confusionfile.getValue(n).getName();
					addtoname=addtoname.substring(0, addtoname.length()-4);
					try {
						conf  = text.parseFloatFile();
					} catch (IOException e) { 
						throw new RuntimeException("Unable to parse grad-file");
					}
				}else{
					/* fake it */
					float max = performance.getFloat();
					float offterm = (1-max)/(N-1);
					conf = new float[N][N];
					for(int i=0; i<N; i++){
						for(int j=0; j<N; j++){
							if(i==j){
								conf[i][j]=max;
							}else{
								conf[i][j]=offterm;
							}
						}
					}
				}
			}else{
				/* Parse Permimeter Error Vector 
				 * and grow probability vector*/ 
				if(perimerrfile.getValue(n)!=null){
					//boundErrProb
					float[][] pererlist = null;
					text = new TextFileReader(perimerrfile.getValue(n));

					try{
						pererlist  = text.parseFloatFile();
					}catch(IOException e){
						e.printStackTrace();
					}
					if(pererlist.length>0 && pererlist[0].length==1){
						perErr = new float[pererlist.length];
						for(int i=0; i<pererlist.length; i++){
							perErr[i]=pererlist[i][0];
						}
					}else if(pererlist[0].length>0 && pererlist.length==1){
						perErr = new float[pererlist[0].length];
						for(int i=0; i<pererlist[0].length; i++){
							perErr[i]=pererlist[0][i];
						}
					}else{
						System.err.println("Invalid Perimeter Error Prob file");
					}

					//grow prob file
					pererlist = null;
					text = new TextFileReader(growprobfile.getValue(n));
					addtoname=growprobfile.getValue(n).getName();
					addtoname=addtoname.substring(0, addtoname.length()-4);
					try{
						pererlist  = text.parseFloatFile();
					}catch(IOException e){
						e.printStackTrace();
					}
					if(pererlist.length>0 && pererlist[0].length==1){
						growProb = new float[pererlist.length];
						for(int i=0; i<pererlist.length; i++){
							growProb[i]=pererlist[i][0];
						}
					}else if(pererlist[0].length>0 && pererlist.length==1){
						growProb = new float[pererlist[0].length];
						for(int i=0; i<pererlist[0].length; i++){
							growProb[i]=pererlist[0][i];
						}
					}else{
						System.err.println("Invalid Perimeter Growth Prob file");
					}

				}else{
					/* fake it */
					//					perover = new float[N];
					//					int NperimErr = N;
					//					for(int i=0; i<NperimErr; i++){
					//					perover[i]=1;
					//					}

					bounds = VolumeRater.findBoundaries(truthvol,conn);
					bounds.trimToSize();
					int boundnum = bounds.size();
					perErr = new float[boundnum];
					growProb = new float[boundnum];
					for(int i=0; i<boundnum; i++){
						growProb[i]=0.7f;
						perErr[i] = 1f/((float)boundnum);
					}
				}
			}


			//init volume rater object
			VolumeRater vr = null;
			if(typeErr.getIndex()==0){
				vr = new VolumeRater(truthvol, totproberr, bounds, perErr, growProb);
			}else{
				vr = new VolumeRater(truthvol, totproberr, labels, conf);
			}
			vr.setConnectivity(conn);
			//generate rater volumes
			for(int i=0; i<num; i++){	
				vr.setName(truthvol.getName()+"_"+addtoname+"_"+Integer.toString(i));
				File volout = null;
				System.out.println("ErrorType: " + typeErr.getValue());
				if(typeErr.getIndex()==0){
					System.out.println("Perim");
					vr.genPerimRaterVolume();
					volout=vr.writeRaterVolume(dir);
				}else{
					System.out.println("Conf");
					vr.genConfRaterVolume();
					volout=vr.writeRaterVolume(dir);
				}
				vr.reset();
				vollist.add(volout);
			}
			
		}
		System.out.println("vollist");
		System.out.println(vollist);
		raterVols.setValue(vollist);
	}

}
