package edu.jhmi.rad.medic.structures;

import java.io.*;
import java.util.*;
import java.util.zip.*;
import java.lang.*;
import java.net.URL;

/**
 *
 *  This class handles a LUT for critical points in 3D topology
 *	with the connectivity from a given filename
 *	
 *	@version    June 2005
 *	@author     Pierre-Louis Bazin
 *
 */

public class CriticalPointLUT2D {
	
	private static final int B0 = 1;
	private static final int B1 = 2;
	private static final int B2 = 4;
	private static final int B3 = 8;
	private static final int B4 = 16;
	private static final int B5 = 32;
	private static final int B6 = 64;
	private static final int B7 = 128;
	private static final int B8 = 256;
	
	private static final int MAX = 200;

	private BitSet 		isRegular;
	
	private	String		filename;	// file to open
	
	private static final boolean	debug = true;
	
	public CriticalPointLUT2D(String filename_, int size) {
		isRegular = new BitSet(size);
		filename = filename_;
	}
	
	public void finalize() {
		isRegular = null;
	}
	
	/**
	 *  add a new value
	 */
	public final void set(int ind, boolean val) { isRegular.set(ind, val); }
	
	/**
	 *  get a value
	 */
	 public final boolean get(int ind) { return isRegular.get(ind); }
	
	 /** translate a pattern into the key number */
	 public final int keyFromPattern(byte[][] img, int x, int y) {
		 int key=0;
		 
		 if (img[x-1][y-1]>img[x][y]) key += B0;
		 if (img[x-1][y  ]>img[x][y]) key += B1;
		 if (img[x-1][y+1]>img[x][y]) key += B2;
		 if (img[x  ][y-1]>img[x][y]) key += B3;
		 if (img[x  ][y+1]>img[x][y]) key += B4;
		 if (img[x+1][y-1]>img[x][y]) key += B5;
		 if (img[x+1][y  ]>img[x][y]) key += B6;
		 if (img[x+1][y+1]>img[x][y]) key += B7;
		  
		 return key;
	 }
	 
	 /** translate a pattern into the key number */
	 public final int keyFromPattern(byte[][][] img, int x, int y, int k) {
		 int key=0;
		 
		 if (img[x-1][y-1][k]>img[x][y][k]) key += B0;
		 if (img[x-1][y  ][k]>img[x][y][k]) key += B1;
		 if (img[x-1][y+1][k]>img[x][y][k]) key += B2;
		 if (img[x  ][y-1][k]>img[x][y][k]) key += B3;
		 if (img[x  ][y+1][k]>img[x][y][k]) key += B4;
		 if (img[x+1][y-1][k]>img[x][y][k]) key += B5;
		 if (img[x+1][y  ][k]>img[x][y][k]) key += B6;
		 if (img[x+1][y+1][k]>img[x][y][k]) key += B7;
		  
		 return key;
	 }
	 
	 /** translate a pattern into the key number */
	 public final int keyFromPattern(boolean[][] img, int x, int y) {
		 int key=0;
		 
		 if (img[x-1][y-1]) key += B0;
		 if (img[x-1][y  ]) key += B1;
		 if (img[x-1][y+1]) key += B2;
		 if (img[x  ][y-1]) key += B3;
		 if (img[x  ][y+1]) key += B4;
		 if (img[x+1][y-1]) key += B5;
		 if (img[x+1][y  ]) key += B6;
		 if (img[x+1][y+1]) key += B7;
		  
		 return key;
	 }
	 
	 /** translate a pattern into the key number */
	 public final int keyFromPattern(byte[] img, int idx, int dx, int dy, int dz) {
		 int key=0;
		 
		 if (img[idx-dx-dy   ]>img[idx]) key += B0;
		 if (img[idx-dx      ]>img[idx]) key += B1;
		 if (img[idx-dx+dy   ]>img[idx]) key += B2;
		 if (img[idx   -dy   ]>img[idx]) key += B3;
		 if (img[idx   +dy   ]>img[idx]) key += B4;
		 if (img[idx+dx-dy   ]>img[idx]) key += B5;
		 if (img[idx+dx      ]>img[idx]) key += B6;
		 if (img[idx+dx+dy   ]>img[idx]) key += B7;
		 
		 return key;
	 }
	 
	 /** translate a key into the corresponding pattern */
	 public final byte[][] patternFromKey(int key) {
		 byte[][] img = new byte[3][3];
		 
		 img[2][2] = (byte)(key/B7); key -= img[2][2]*B7;
		 img[2][1] = (byte)(key/B6); key -= img[2][1]*B6;
		 img[2][0] = (byte)(key/B5); key -= img[2][0]*B5;
		 img[1][2] = (byte)(key/B4); key -= img[1][2]*B4;
		 img[1][0] = (byte)(key/B3); key -= img[1][0]*B3;
		 img[0][2] = (byte)(key/B2); key -= img[0][2]*B2;
		 img[0][1] = (byte)(key/B1); key -= img[0][1]*B1;
		 img[0][0] = (byte)(key/B0); key -= img[0][0]*B0;
		 
		 return img;
	 }

	 public final void loadPattern() {
		 //URL res = CriticalPointLUT2D.class.getResource( filename );
		 //String current = System.getProperty("user.dir");
		 //System.out.println("path: "+current);
		 //System.setProperty("user.dir", res.getPath());
		 //System.out.println("pattern path: "+System.getProperty("user.dir"));
		 try {
			 //File f = new File( res.getFile() );
			 //File f = new File( filename );
			 //FileInputStream fis = new FileInputStream( f );
			 //if (debug) System.out.println("Opening LUT: "+res.toString());
			 //InputStream fis = res.openStream();
			 InputStream fis = CriticalPointLUT2D.class.getResourceAsStream( filename );
			 byte[] list = new byte[B8];
			
			 fis.read(list);
			 fis.close();
			 int N = 0;
			 for (int n=0;n<B8;n++) {
				 isRegular.set( n, (list[n]==1) );
			 	 if (list[n]==1) N++;
			 }
			 if (debug) System.out.println("Simple points: "+N);
			 
			 list = null;
		 } catch (FileNotFoundException e) {
			 System.out.println("File not found:");
			 System.out.println(e.getMessage());
		 } catch (IOException e ) {
			 System.out.println("i/o exception:");
			 System.out.println(e.getMessage());
		 }
	 }
	 
	 public final boolean loadCompressedPattern() {
		 /*
		 URL res = null;
		 try {
			res = CriticalPointLUT2D.class.getResource( filename );
		 } catch (Exception e) {
			System.err.println("error:");
            System.err.println(e.getMessage()+ "\n");
		 }
		 */
		 //String current = System.getProperty("user.dir");
		 //System.out.println("path: "+current);
		 //System.setProperty("user.dir", res.getPath());
		 //System.out.println("pattern path: "+System.getProperty("user.dir"));
		 try {
			 //File f = new File( res.getFile() );
			 //File f = new File( filename );
			 //FileInputStream fis = new FileInputStream( f );
			 //if (debug) System.out.println("Opening LUT: "+res.toString());
			 //FileInputStream fis = new FileInputStream( res.getPath() );
			 //FileInputStream fis = res.openStream();
			 InputStream fis = CriticalPointLUT2D.class.getResourceAsStream( filename );
			 GZIPInputStream gzfis = new GZIPInputStream(fis, B8);
			 //GZIPInputStream gzfis = new GZIPInputStream(res.openStream(), B26);
			 byte[] list = new byte[B8];
			
			 gzfis.read(list, 0, B8);
			 gzfis.close();
			 int N = 0;
			 for (int n=0;n<B8;n++) {
				 isRegular.set( n, (list[n]==1) );
				 if (list[n]==1) N++;
			 }
			 if (debug) System.out.println("Simple points: "+N);
			 
			 list = null;
			 return true;
		 } catch (FileNotFoundException e) {
			 System.out.println("File not found:");
			 System.out.println(e.getMessage());
			 isRegular = null;
			 return false;
		 } catch (IOException e ) {
			 System.out.println("i/o exception:");
			 System.out.println(e.getMessage());
			 isRegular = null;
			 return false;
		 }
	 }
	 
	 public final String getFilename() {
		 URL res = CriticalPointLUT2D.class.getResource( filename );
		 return res.toString();
	 }
}
