/**
 * JIST Extensions for Computer-Integrated Surgery
 *
 * Center for Computer-Integrated Surgical Systems and Technology &
 * Johns Hopkins Applied Physics Laboratory &
 * The Johns Hopkins University
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 * @author Blake Lucas
 */
package edu.jhu.cs.cisst.plugins.utilities.surface;

import javax.vecmath.Point3f;
import javax.vecmath.Point3i;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import edu.jhu.cs.cisst.algorithms.geometry.surface.*;
import edu.jhu.cs.cisst.algorithms.segmentation.gac.DistanceField3D;
import edu.jhu.cs.cisst.algorithms.util.DataOperations;
import edu.jhu.ece.iacl.algorithms.CommonAuthors;
import edu.jhu.ece.iacl.algorithms.volume.DistanceField;
import edu.jhu.ece.iacl.jist.algorithms.graphics.GeomUtil;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.NormalGenerator;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;

public class PlugInConvertSurfaceToLevelSet extends ProcessingAlgorithm {
	protected ParamSurface surfParam;
	protected ParamVolume levelsetParam;
	protected ParamDouble cellsizeParam;
	protected ParamPointFloat minPointParam;
	protected ParamPointFloat maxPointParam;
	protected ParamPointInteger dimsParam;
	protected ParamBoolean useDimsParam;
	protected ParamBoolean autoBoundsParam;
	protected ParamSurface orientedSurfaceParam;
	protected ParamMatrix reshapeMatrixParam;
	protected ParamSurface isoSurfParam;

	protected void createInputParameters(ParamCollection inputParams) {
		inputParams.add(surfParam = new ParamSurface("Surface"));
		inputParams.add(cellsizeParam = new ParamDouble("Cell Size", 1));
		inputParams.add(minPointParam = new ParamPointFloat("Min Point",
				new Point3f(0, 0, 0)));
		inputParams.add(maxPointParam = new ParamPointFloat("Max Point",
				new Point3f(256, 256, 256)));
		inputParams.add(dimsParam = new ParamPointInteger("Dimensions",
				new Point3i(256, 256, 256)));
		inputParams
				.add(useDimsParam = new ParamBoolean("Use Dimensions", true));
		inputParams
				.add(autoBoundsParam = new ParamBoolean("Auto-Bounds", true));

		inputParams.setName("surf_to_level_set");
		inputParams.setLabel("Surface to Level Set");
		inputParams.setCategory("Utilities.Surface");
		inputParams.setPackage("CISST");
		AlgorithmInformation info = getAlgorithmInformation();
		info.add(CommonAuthors.blakeLucas);
		info.setVersion(MeshDistanceHash.getVersion());
		info.setDescription("Converts a surface to level set.");
		info.setStatus(DevelopmentStatus.Release);
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(levelsetParam = new ParamVolume("Level Set"));
		outputParams.add(isoSurfParam = new ParamSurface("Iso-Surface"));
		outputParams.add(orientedSurfaceParam = new ParamSurface(
				"Oriented Surface"));
		outputParams.add(reshapeMatrixParam = new ParamMatrix(
				"Volume to Surface Transform", 4, 4));
	}

	/**
	 * Cross.
	 * 
	 * @param pa
	 *            the pa
	 * @param pb
	 *            the pb
	 * @param pc
	 *            the pc
	 * 
	 * @return the vector3f
	 */
	private static Vector3f cross(Point3f pa, Point3f pb, Point3f pc) {
		Point3f p1 = new Point3f((float) (pb.x - pa.x), (float) (pb.y - pa.y),
				(float) (pb.z - pa.z));
		Point3f p2 = new Point3f((float) (pc.x - pa.x), (float) (pc.y - pa.y),
				(float) (pc.z - pa.z));
		Vector3f p3 = new Vector3f(p1.y * p2.z - p1.z * p2.y, p1.z * p2.x
				- p1.x * p2.z, p1.x * p2.y - p1.y * p2.x);
		return p3;
	}

	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		EmbeddedSurface surf = surfParam.getSurface();
		TriangleMesh mesh = new TriangleMesh(surf);
		MeshDistanceHash hash;
		if (autoBoundsParam.getValue()) {
			hash = new MeshDistanceHash(mesh, cellsizeParam.getDouble());
		} else {
			if (useDimsParam.getValue()) {
				hash = new MeshDistanceHash(mesh, cellsizeParam.getDouble(),
						minPointParam.getValue(), dimsParam.getValue());
			} else {
				hash = new MeshDistanceHash(mesh, cellsizeParam.getDouble(),
						minPointParam.getValue(), maxPointParam.getValue());
			}
		}
		monitor.observe(hash);
		ImageDataFloat df = hash.getDistanceField();
		DistanceField3D dfs = new DistanceField3D();
		df = dfs.solve(df, 10.0f);
		levelsetParam.setValue(df);
		isoSurfParam.setValue(hash.getIsoSurface());
		reshapeMatrixParam.setValue(hash.getTransform());

		levelsetParam.setValue(df);
		float[][][] dfMat = df.toArray3d();
		int rows = df.getRows();
		int cols = df.getCols();
		int slices = df.getSlices();

		surf.transform(hash.getTransform().inverse());
		int indexCount = surf.getIndexCount();
		Point3f centroid = new Point3f();
		int vid1, vid2, vid3;
		float[][][][] gradient = DataOperations.gradient(dfMat, false);
		int flip = 0;
		double[][] faceData = new double[indexCount / 3][1];
		for (int fid = 0; fid < indexCount; fid += 3) {
			Point3f p1 = surf.getVertex(vid1 = surf.getCoordinateIndex(fid));

			Point3f p2 = surf
					.getVertex(vid2 = surf.getCoordinateIndex(fid + 1));

			Point3f p3 = surf
					.getVertex(vid3 = surf.getCoordinateIndex(fid + 2));

			centroid.x = 0.333333333333f * (p1.x + p2.x + p3.x);
			centroid.y = 0.333333333333f * (p1.y + p2.y + p3.y);
			centroid.z = 0.333333333333f * (p1.z + p2.z + p3.z);

			Vector3f norm = cross(p1, p2, p3);

			Vector3f grad = DataOperations.interpolateVector(centroid.x,
					centroid.y, centroid.z, gradient, rows, cols, slices);
			faceData[fid / 3][0] = DataOperations.interpolate(centroid.x,
					centroid.y, centroid.z, dfMat, rows, cols, slices);
			if (norm.dot(grad) < 0) {
				flip++;
				surf.setCoordinateIndex(fid, vid3);
				surf.setCoordinateIndex(fid + 2, vid1);
			}
		}
		System.out.println("Flipped " + flip + " normals to point outward.");
		String name = surf.getName();
		surf = new EmbeddedSurface(surf.getVertexCopy(), surf.getIndexCopy());
		//surf.setCellData(faceData);
		surf.setName(name);
		surf.transform(hash.getTransform());
		orientedSurfaceParam.setValue(surf);
	}

}
