package edu.jhu.ece.iacl.algorithms.manual_label.simulation;

import java.awt.List;
import java.io.File;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.NavigableSet;
import java.util.Random;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.vecmath.Point2f;
import javax.vecmath.Point2i;

import edu.jhu.ece.iacl.jist.io.ImageDataReaderWriter;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataInt;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import edu.jhu.ece.iacl.utility.ArrayUtil;
/**
 * A class to produce synthetic / random label images
 * to be used by segmentation algorithms
 * 
 * Some methods are ports of levelset.c
 * downloadable from:
 * http://www.mathworks.com/matlabcentral/fileexchange/29447-multiphase-level-set-image-segmentation
*/
public class SyntheticLabeling {
	
	private static final double M_PI = Math.PI; 
	private static final double ang60 = Math.PI/3;
	private static final double[] angles = new double[]{ang60,2*ang60,3*ang60,4*ang60,5*ang60};
	
	public static int[][] genRandom(int nx, int ny, int nobj){
		return genRandom(nx,ny,nobj,new Random());
	}
	public static int[][] genRandom(int nx, int ny, int nobj, Random rand){
		int[][] out = new int[nx][ny];
		for(int x=0; x<nx; x++)for(int y=0; y<ny; y++){
			out[x][y]=rand.nextInt(nobj);
		}
		return out;
	}
	public static int[][][] genRandom(int nx, int ny, int nz, int nobj){
		return genRandom(nx,ny,nz,nobj,new Random());
	}
	public static int[][][] genRandom(int nx, int ny, int nz, int nobj, Random rand){
		int[][][] out = new int[nx][ny][nz];
		for(int x=0; x<nx; x++)for(int y=0; y<ny; y++)for(int z=0; z<nz; z++){
			out[x][y][z]=rand.nextInt(nobj);
		}
		return out;
	}
	
	public static int[] genObjLabelsRand(int npts, int nobj){
		Random r = new Random();
		int[] labels = new int[npts];
		for(int i=0; i<npts; i++){
			labels[i]=r.nextInt(nobj);
		}
		return labels;
	}
	
	public static int[] genObjLabels(int npts, int nobj){
		int[] labels = new int[npts];
		int l = 0;
		for(int i=0; i<npts; i++){
			labels[i]=l;
			l++;
			if(l==nobj){
				l=0;
			}
		}
		return labels;
	}
	
	public static int[][] genOverlapMask(int nobj, float circsize){
		return null;
	}
	
public static int[][] circlesFromPoints(ArrayList<Point2f> points, int nx, int ny, float circsize){
		int[] labels = new int[points.size()];
		for(int i=1; i<=labels.length; i++)
			labels[i-1]=i;
			
		return circlesFromPoints(points,labels,nx,ny,circsize);
	}

	public static int[][] circlesFromPoints(ArrayList<Point2f> points, int[] labels, int nx, int ny, float circsize){
		
		int ncirc = points.size();
		
		int[][] out = new int[nx][ny];
		byte[][] circmask = genCircleMask(circsize);
		int N = circmask.length;
		
		for(int i=0; i<ncirc; i++){
			
			int xc = Math.round(points.get(i).x - N/2);
			int yc = Math.round(points.get(i).y - N/2);

			int myobj = labels[i];

			for(int xx=0; xx<N; xx++)for(int yy=0; yy<N; yy++){
				if(circmask[xx][yy]==1 && xc+xx<nx && yc+yy<ny && xc+xx>=0 && yc+yy>=0){
					out[xc+xx][yc+yy] = myobj;
				}
			}
		}
		
		return out;
	}
	
	public static int[][] circlesFromPoints(Point2f[] points, int nx, int ny, float circsize){
		int[] labels = new int[points.length];
		for(int i=1; i<=labels.length; i++)
			labels[i-1]=i;
		
		return circlesFromPoints(points,labels,nx,ny,circsize);
	}
	
	public static int[][] circlesFromPoints(Point2f[] points, int[] labels, int nx, int ny, float circsize){
		
		int ncirc = points.length;
		
		int[][] out = new int[nx][ny];
		byte[][] circmask = genCircleMask(circsize);
		int N = circmask.length;
		
		for(int i=0; i<ncirc; i++){
			
			int xc = Math.round(points[i].x - N/2);
			int yc = Math.round(points[i].y - N/2);

			int myobj = labels[i];

			for(int xx=0; xx<N; xx++)for(int yy=0; yy<N; yy++){
				if(circmask[xx][yy]==1){
					out[xc+xx][yc+yy] = myobj;
				}
			}
		}
		
		return out;
	}
	
	public static int[][] genUniformCircleEverywhere(int nx, int ny, int nobj, float circrad, float ol, float bnd, boolean rand){
		
		ArrayList<Point2f> pts = new ArrayList<Point2f>();
		
		float rowoff = 2*(circrad)*(1-ol);
		
		pts.add(new Point2f(bnd+circrad, bnd+circrad));
		pts.add(new Point2f(bnd+circrad+rowoff, bnd+circrad));
		
		Point2f newpt = rotate(pts.get(0),pts.get(1),angles[0]);
		
		
		System.out.println(pts.get(0));
		System.out.println(pts.get(1));
		System.out.println(newpt);
		
		float xoff = newpt.x-pts.get(0).x;
		float yoff = newpt.y-pts.get(0).y;
		
		float x = pts.get(0).x;
		float y = pts.get(0).y;
		
		boolean usept1x = false;
		float startx = pts.get(1).x;
		
		for(y=pts.get(1).y; y<ny; y+=yoff){
//			System.out.println("y: " + y);
			for(x=startx; x<nx; x+=rowoff){
				pts.add(new Point2f(x,y));
			}
			
			// update x
			if(usept1x){
				startx = pts.get(0).x;
				usept1x = false;
			}else{
				startx = pts.get(0).x - xoff;
				usept1x = true;
			}
		}
		
//		System.out.println(pts);
		int[] labels = null;
		if(rand){
			labels = genObjLabelsRand(pts.size(),nobj);
		}else{
			labels = genObjLabels(pts.size(),nobj);
		}
		
		int[][] out = circlesFromPoints(pts,labels,nx,ny,circrad);
		
		return out;
	}
	
	public static int[][] genUniformCircles(int nx, int ny, int nobj, float circrad, float ol, float bnd){
		int[][] out;
		
		Point2f[] cntrpnts = genTriangularTessellation((1-ol)*2*circrad,nobj-1);
		float minx = Float.MAX_VALUE;
		float maxx = Float.MIN_VALUE;
		float miny = Float.MAX_VALUE;
		float maxy = Float.MIN_VALUE;
		
		for(int i=0; i<cntrpnts.length; i++){
			if(cntrpnts[i].x<minx){ minx=cntrpnts[i].x; }
			if(cntrpnts[i].x>maxx){ maxx=cntrpnts[i].x; }
			if(cntrpnts[i].y<miny){ miny=cntrpnts[i].y; }
			if(cntrpnts[i].y>maxy){ maxy=cntrpnts[i].y; }
		}
		
		float tilestepx = 2*(circrad) + bnd + maxx-minx;
		float tilestepy = 2*(circrad) + bnd + maxy-miny;
		
		System.out.println("minx: " + minx);
		System.out.println("maxx: " + maxx);
		System.out.println("miny: " + miny);
		System.out.println("maxy: " + maxy);
		
		System.out.println("tilestepx: " + tilestepx);
		System.out.println("tilestepy: " + tilestepy);
		
		ArrayList<Point2f> pnts = new ArrayList<Point2f>(cntrpnts.length);
		ArrayList<Integer> labs = new ArrayList<Integer>(cntrpnts.length);
		for(int i=0; i<cntrpnts.length; i++){
//			float x =0; 
//			float y =0;
			float thisx = circrad+bnd+cntrpnts[i].x-minx;
			float thisy = circrad+bnd+cntrpnts[i].y-miny;
			for(int x = 0; x<nx; x+=tilestepx) for(int y = 0; y<ny; y+=tilestepy){
				Point2f mypt = new Point2f(x+thisx, y+thisy);
				pnts.add(mypt);
				labs.add(i+1);
			}
		}
		System.out.println(pnts);
		
		int npts = pnts.size();
		int[] labels = new int[npts];
		int l = 1;
		for(int i=0; i<npts; i++){
			labels[i]=labs.get(i);
		}
		
		out = circlesFromPoints(pnts,labels,nx,ny,circrad);
		
		return out;
	}
	
	public static Point2f[] genTriangularTessellation(float r, int npts){
		Point2f[] pts = new Point2f[npts];
		int ptcount = 0;
		float eps = (0.25f*r)*(0.25f*r);
		
		System.out.println("eps: " + eps);
		
		pts[0] = new Point2f(0,0);
		ptcount++;

		DistComparator dc = new DistComparator(eps);
		TreeSet<Point2f> allptset = new TreeSet<Point2f>(dc);
		TreeSet<Point2f> addptset = new TreeSet<Point2f>(dc);
		
		allptset.add(pts[0]);
		while(ptcount< npts){
			
			// add all rotated points
			Point2f firstadd = new Point2f(pts[ptcount-1].x + r, pts[ptcount-1].y);
			if(allptset.add(firstadd)){
				addptset.add(firstadd);
//				System.out.println("added: " + firstadd);
			}
				
			for(int i=0; i<angles.length; i++){
				Point2f addnow = rotate(pts[ptcount-1],firstadd, angles[i]);
				if(allptset.add(addnow)){
//					System.out.println("added: " + addnow);
					addptset.add(addnow);
				}
			}
//			System.out.println("\nallptset: " + allptset);
//			System.out.println("\naddptset: " + addptset);
			
			
			Point2f candidate = addptset.pollFirst();
			
			boolean addme = true;
			for(int j=0; j<ptcount; j++){
				float d = 0;
				d = (candidate.x-pts[j].x)*(candidate.x-pts[j].x) + (candidate.y-pts[j].y)*(candidate.y-pts[j].y);
				if(d<eps){
					addme=false;
				}
			} 
			
			if(addme){
				pts[ptcount] = candidate;
//				System.out.println("\ntook: " + pts[ptcount]);
				ptcount++;
			}
			
		
		}
		
//		System.out.println("\nFINAL POINTS:");
//		for(int i=0; i<pts.length; i++){
//			System.out.println(pts[i]);
//		}
		
		return pts;
	}
	
	static class DistComparator implements Comparator<Point2f>{
		float r;
		public DistComparator(float rad){this.r = rad;}
		
		@Override
		public int compare(Point2f a, Point2f b){
			float da = a.x*a.x + a.y*a.y;
			float db = b.x*b.x + b.y*b.y;
			float dab = (a.x - b.x)*(a.x - b.x) + (a.y - b.y)*(a.y - b.y);
			if(dab>r){
				if(da<db)
					return -1;
				else if(db<da)
					return 1;	
				else
					return 1;
			}else{
				return 0;
			}
		}
	}
	
	
	
	public static int[][] genUniformCircles1(int nx, int ny, int nobj, float circsize, float ol){
		int[][] out = new int[nx][ny];
		byte[][] circmask = genCircleMask(circsize);
		int N = circmask.length;
		int obj = 1;
		
		System.out.println("circmask size: " + circmask.length + " x " + circmask[0].length);
		
		int xs=0;
		int ys=0;
		int x =0;
		int y =0;
		while(xs<nx){
			ys=0;
			while(ys<ny){
			
				for(x=0; (x<N && xs+x<nx); x++)for(y=0; y<N && ys+y<ny; y++){
					if(circmask[x][y]==1){
						out[xs+x][ys+y] = obj;
					}
				}
				
				obj++;
				if(obj==nobj) obj = 1;
				
				ys+=N;
			}
			
			xs+=N;
		}
		
		return out;
	}
	
	/**
	 * 
	 * @param nx
	 * @param ny
	 * @param nobj
	 * @param frac
	 * @param ol
	 * @param oljitter - distance of centers of circles vary uniformly by a factor of (oljitter)*basedist to (1+oljitter)*basedist 
	 * @return
	 */
	public static int[][] genRandomCircles(int nx, int ny, int nobj, float frac, float ol, float oljitter){
		int[][] out = new int[nx][ny];
		Random rand = new Random();
		
		int ntries = 500;
		
		int ncirc = nobj-1;
		
		float rad  = (float)(0.6f*Math.sqrt((nx*ny*frac)/(Math.PI*ncirc*(1-ol))));
		float diam = 2*rad;
		float basedist = (1-ol)*diam;
		float mindist  = (1-ol-oljitter)*diam - 2;
		
		if(oljitter>1f || oljitter<0f){
			System.err.println("ol jitter must be >1 & <0");
			return null;
		}
		
		float maxjitter = oljitter*diam;
//		float jittermult = oljitter*basedist/2;
		
		System.out.println("ol: " + ol);
		System.out.println("rad: " + rad);
		System.out.println("diam: " + diam);
		System.out.println("basedist: " + basedist);
		System.out.println("maxjitter: " + maxjitter);
		
		
		float bnd = diam; 
		
		byte[][] circmask = genCircleMask(rad);
		int N = circmask.length;
		System.out.println("circmask size: " + circmask.length + " x " + circmask[0].length);
		
		ArrayList<Integer> lablist = new ArrayList<Integer>(nobj);
		for(int i=1; i<nobj; i++){ lablist.add(i); }
		
		
		// the initial point has to be in bounds
		int t = 0;
		boolean notdone = true;
		
		float x1=-1f, y1=-1f, vx=-1f, vy=-1f;
		while(t<ntries && notdone){
			System.out.println("try: " + t);
			
			notdone = false;
			x1 = bnd + (nx - 2*bnd)*rand.nextFloat();
			y1 = bnd + (ny - 2*bnd)*rand.nextFloat();

			vx = rand.nextFloat();
			vy = rand.nextFloat();
			float vm = (float)Math.sqrt(vx*vx + vy*vy);
			
			//add jitter
			float thisdist = basedist + (rand.nextFloat()-0.5f)*2*maxjitter;
			vx = vx*thisdist/vm;
			vy = vy*thisdist/vm;
			
			

			if(!isInBounds(x1+vx,y1+vy,nx,ny,bnd)){
				vx = -vx;
				if(!isInBounds(x1+vx,y1+vy,nx,ny,bnd)){
					vy = -vy;
					if(!isInBounds(x1+vx,y1+vy,nx,ny,bnd)){
						vx = -vx;
						if(!isInBounds(x1+vx,y1+vy,nx,ny,bnd)){
//							System.out.println("OOB init - try again");
							notdone = true;
						}
					}
				}
			}
			t++;
		}
		
		ArrayList<Point2f> points = new ArrayList<Point2f>(ncirc);
		points.add(new Point2f(x1,y1));
		points.add(new Point2f(x1+vx,y1+vy));
		
		// stores the indices of of other points this point is "connected" to
		ArrayList<int[]> edges = new ArrayList<int[]>(ncirc);
		
		// stores the number of points each point is "connected" to
		int[] pointedgecounts = new int[ncirc];
		
		edges.add(new int[ncirc]);
		edges.add(new int[ncirc]);
		
		edges.get(0)[0] = 1; pointedgecounts[0]++;
		edges.get(1)[0] = 0; pointedgecounts[1]++;
		
		t = 0;
		int n=2;
		while(t<ntries && n<ncirc){
			System.out.println("try: " + t);
			// pick a point
			int ptind = rand.nextInt(n);
			
			// find an edge w/ that point
//			System.out.println("blah in: " + pointedgecounts[ptind]);
			int edgeind = rand.nextInt(pointedgecounts[ptind]);
			
			
			
			// get a nbr of it.
			int nbrind = edges.get(ptind)[edgeind];
			
			Point2f proppt = rotate(points.get(ptind),points.get(nbrind),angles[rand.nextInt(5)]);
			float thisdist = basedist + (rand.nextFloat()-0.5f)*2*maxjitter;
			rescale(points.get(ptind),proppt,thisdist);
			
//			System.out.println("thispt: " + points.get(ptind));
//			System.out.println("nbrpt : " + points.get(nbrind));
//			System.out.println("proppt: " + proppt);
			if(!isInBounds(proppt.x,proppt.y,nx,ny,bnd)){
//				System.out.println("OOB " + bnd  + " ... try again");
				continue;				
			}
			// check to see if this region has been covered already
			boolean isrepeat = false;
			for(int i=0; i<n; i++){
				if(i!=ptind && i!=nbrind){
					if(dist(proppt.x,proppt.y,points.get(i).x,points.get(i).y) < mindist){
//						System.out.println("Point already exists ... try again");
						isrepeat = true;
						break;
					}
				}
			}
			if(isrepeat) continue;
			
			// add this point
			points.add(proppt);
			System.out.println("points: " + points);
			
			// add this points to edges for its "seed points"
			edges.get(ptind)[pointedgecounts[ptind]] 	= n; pointedgecounts[ptind]++;
			edges.get(nbrind)[pointedgecounts[nbrind]] 	= n; pointedgecounts[nbrind]++;
			
			// add the "seed points" as edges of this point
			edges.add(new int[ncirc]);
			edges.get(n)[0]=ptind;
			edges.get(n)[1]=nbrind;
			pointedgecounts[n]+=2;
			
			n++;
			
			t++;
		}
		
		
		System.out.println("points: " + points);
		System.out.println("points len: " + points.size());
		
		// drop down the circles at all the (rounded) points
		for(int i=0; i<ncirc; i++){
			System.out.println(lablist);
			
			int xc = Math.round(points.get(i).x - N/2);
			int yc = Math.round(points.get(i).y - N/2);
			
			int myind = -1;
			int myobj = -1;
//			if(lablist.size()==1){
//				myind = 0;
//			}else{
				myind = rand.nextInt(ncirc-i);
//			}
			
			myobj = lablist.get(myind);
			
			lablist.remove(myind);

			for(int xx=0; xx<N; xx++)for(int yy=0; yy<N; yy++){
				if(circmask[xx][yy]==1){
					out[xc+xx][yc+yy] = myobj;
				}
			}
		}
		
		
		return out;
	}
	
	public static boolean isInBounds(float x, float y, int nx, int ny, float bnd){
		boolean inbnds = true;
		
		if(x<bnd){ inbnds = false; }
		if(y<bnd){ inbnds = false; }
		if(nx-x<bnd){ inbnds = false; }
		if(ny-y<bnd){ inbnds = false; }
		
		return inbnds;
	}
	
	public static Point2f rotate(float x1, float y1, float x2, float y2, double ang){
		Point2f out = new Point2f();
		
		double cosang = Math.cos(ang);
		double sinang = Math.cos(ang);
		
		out.x = (float)(x1 + (x2-x1)*cosang -  (y2-y1)*sinang); 
		out.y = (float)(y1 + (x2-x1)*sinang +  (y2-y1)*cosang);
		
		return out;
	}
	
	public static Point2f rotate(Point2f pt1, Point2f pt2, double ang){
		Point2f out = new Point2f();
		
		double cosang = Math.cos(ang);
		double sinang = Math.sin(ang);
		
		out.x = (float)(pt1.x + (pt2.x-pt1.x)*cosang -  (pt2.y-pt1.y)*sinang); 
		out.y = (float)(pt1.y + (pt2.x-pt1.x)*sinang +  (pt2.y-pt1.y)*cosang);
		
		return out;
	}
	
	public static void rescale(Point2f fixed, Point2f moving, float newdist){
		float vx = moving.x-fixed.x;
		float vy = moving.y-fixed.y;
		double vm = Math.sqrt(vx*vx + vy*vy);
		vx *= (newdist/vm);
		vy *= (newdist/vm);
		
		moving.x = fixed.x + vx;
		moving.y = fixed.y + vy;
	}
	
	public static byte[][] genCircleMask(float rad){
		float diam = 2*rad;
		int N = (int)Math.ceil(diam);
		byte[][] mask = new byte[N][N];
		for(int x=0; x<N; x++)for(int y=0; y<N; y++){
			if(dist(x, y, rad, rad)<rad){
				mask[x][y]=1;
			}
		}
		return mask;
	}
	
	/**
	 * Initializes multiphase levelsets in two ways:
	 * if inittype == 1, the initialization consists of large offset circles for each phases
	 * else the initialization has many small offset circles
	 * @param nx dim x
	 * @param ny dim y
	 * @param nobj number of objects
	 * @param inittype type of initialization
	 * @return
	 */
	public static float[][] initseg(int nx, int ny, int nobj,
			int inittype){
		int N = nx*ny;
		float[][] init = new float[nobj][N];
		
		float distance = 0f;
		
		int count_pos = 0;
		int count_neg = 0;
		int value_min = -100;
		int value_max = 100;
		int coeff = (value_max-value_min);
		double offset = 0;
		float vc = 0f;
		float hc = 0f;
		
		if(inittype==1){
			for(int x = 0; x<nx; x++)for(int y=0; y<ny; y++){
				for(int i=0; i<nobj; i++){
					offset = i * 2 * M_PI /nobj;
					vc = (float)(nx/2*(1-Math.sin(offset)/2.0));
					hc = (float)(ny/2*(1-Math.cos(offset)/2.0));
					
					distance = dist(x, y, vc, hc)/((float) Math.sqrt((nx*nx/4)+(ny*ny/4)));
					init[i][y+ny*x] = (float)(coeff * Math.exp(-distance*distance/.1)+value_min);
				}
			}	
		}else{
			
			int pas=1;
			int x= 5;
			int d=(int)((nx-pas*x)/x);
			int pos=(int)(pas + (d/2));
			int y=(int)((ny)/(d+pas));

			int[] fin = new int[x*y];
			int h = 0;
			int i=0;
			int j=0;
			int l1=0,l2=0;

			float min;

			for(i=0;i<x;i++) for(j=0;j<y;j++) {
				fin[j + y*i]=h;
				h+=1; 
				if (h==nobj) h=0; 
			}

			for(int k=0;k<nobj;k++){
				for(int v=0;v<ny;v++) for(h=0;h<nx;h++) {

					min=9999;
					 for(i=0;i<x;i++) for(j=0;j<y;j++) {
						if( ((v-(pos+j*(pas+d)))*(v-(pos+j*(pas+d)))+(h-(pos+i*(pas+d)))*(h-(pos+i*(pas+d)))<min)&(fin[j + i*y]==k)){
							min=(v-(pos+j*(pas+d)))*(v-(pos+j*(pas+d)))+(h-(pos+i*(pas+d)))*(h-(pos+i*(pas+d)));
							l1=i;l2=j;
						}
					}

					init[k][v+ ny*h]= (float)(((d/2)-dist(v,h,(pos+l2*(pas+d)),(pos+l1*(pas+d))))/Math.sqrt(ny*ny + nx*nx));

				}
			} 
		}
		
		return init;
	}
	
	/**
	 * Initializes labels in two ways:
	 * if inittype == 1, the initialization consists of large offset circles for each phases
	 * else the initialization has many small offset circles
	 * @param nx dim x
	 * @param ny dim y
	 * @param nobj number of objects
	 * @param inittype type of initialization
	 * @return
	 */
	public static int[] initsegLabels(int nx, int ny, int nobj, int inittype){
		int N = nx*ny;
		int[] init = new int[N];
		
		float distance = 0f;
		
		int count_pos = 0;
		int count_neg = 0;
		int value_min = -100;
		int value_max = 100;
		int coeff = (value_max-value_min);
		double offset = 0;
		float vc = 0f;
		float hc = 0f;
		float val = 0f;
		
		if(inittype==1){
			for(int x = 0; x<nx; x++)for(int y=0; y<ny; y++){
				for(int i=0; i<nobj; i++){
					offset = i * 2 * M_PI /(nobj-1);
					vc = (float)(nx/2*(1-Math.sin(offset)/2.0));
					hc = (float)(ny/2*(1-Math.cos(offset)/2.0));
					
					distance = dist(x, y, vc, hc)/((float) Math.sqrt((nx*nx/4)+(ny*ny/4)));
					
					val = (float)(coeff * Math.exp(-distance*distance/.1)+value_min);
					if(val>0){
						init[y+ny*x] = i;
					}
					
				}
			}	
		}else{
			
			int pas=1;
			int x= 5;
			int d=(int)((nx-pas*x)/x);
			int pos=(int)(pas + (d/2));
			int y=(int)((ny)/(d+pas));

			int[] fin = new int[x*y];
			int h = 0;
			int i=0;
			int j=0;
			int l1=0,l2=0;

			float min;

			for(i=0;i<x;i++) for(j=0;j<y;j++) {
				fin[j + y*i]=h;
				h+=1; 
				if (h==nobj) h=0; 
			}

			for(int k=0;k<nobj;k++){
				for(int v=0;v<ny;v++) for(h=0;h<nx;h++) {

					min=9999;
					 for(i=0;i<x;i++) for(j=0;j<y;j++) {
						if( ((v-(pos+j*(pas+d)))*(v-(pos+j*(pas+d)))+(h-(pos+i*(pas+d)))*(h-(pos+i*(pas+d)))<min)&(fin[j + i*y]==k)){
							min=(v-(pos+j*(pas+d)))*(v-(pos+j*(pas+d)))+(h-(pos+i*(pas+d)))*(h-(pos+i*(pas+d)));
							l1=i;l2=j;
						}
					}

					val = (float)(((d/2)-dist(v,h,(pos+l2*(pas+d)),(pos+l1*(pas+d))))/Math.sqrt(ny*ny + nx*nx));

					if(val>0){
						init[y+ny*x] = k;
					}
				}
			} 
		}
		
		return init;
	}
	
	public static float dist(float x_a, float y_a, float x_b, float y_b){
		return (float)Math.sqrt(((x_a-x_b)*(x_a-x_b)+(y_a-y_b)*(y_a-y_b)));
	}
	
	
	
	
	public static void main(String[] args){
//		File outdir = new File("/iacl/cruise09/john/tests/synthLabel");
		File outdir = new File("/iacl/cruise09/john/mgdm/compareMethods/testdat");
//		File outdir = new File("C:\\Users\\John\\Documents\\Research\\tests\\synthLabeling");
		
		File tempfile = new File("/iacl/cruise09/john/multiphase/testdat2/obj8_2.xml");
//		File tempfile = new File("C:\\Users\\John\\Documents\\Research\\multiphase\\testdat2\\obj32_2.xml");
		
		ImageDataReaderWriter rw = ImageDataReaderWriter.getInstance();
		
		ImageData templateimg = rw.read(tempfile);
		ImageHeader hdr = templateimg.getHeader();
		int nx = templateimg.getRows();
		int ny = templateimg.getCols();
		int nz = templateimg.getSlices();
		
		int nobj = 32;
		float circsize = 6f;
		float ol = 0.16f;
		float bnd = 3f;
		
		System.out.println("nx	: " + nx);
		System.out.println("ny	: " + ny);
		System.out.println("nobj: " + nobj);
		
//		DistComparator dc = new DistComparator(0.1f);
//		System.out.println(dc.compare(new Point2f(2f,0f), new Point2f(3.0f,0f)));
		
//		DistComparator dc = new DistComparator(0.1f);
//		TreeSet<Point2f> allptset = new TreeSet<Point2f>(dc);
//		
//		allptset.add(new Point2f(2f,0f));
//		System.out.println(allptset);
////		Point2f f = allptset.first();
//		
//		allptset.remove(new Point2f(0f,2.05f));
//		System.out.println(allptset);
		
//		int[][] result2d = ArrayUtil.reshape2D(SyntheticLabeling.initsegLabels(nx,ny,nobj,2),nx,ny,true);
//		int[][] result2d = SyntheticLabeling.genUniformCircles(nx,ny,nobj,5);
//		int[][] result2d = SyntheticLabeling.genRandomCircles(nx,ny,nobj,0.7f,0.2f);
		
//		Point2f[] pts = genTriangularTessellation(1f, nobj);
		
//		int[][] result2d = genUniformCircles(nx, ny, nobj, circsize, ol, bnd);
		
		int[][] result2d = genUniformCircleEverywhere(nx, ny, nobj, circsize, ol, bnd, false);
		
//		ImageDataInt imgout = new ImageDataInt(result2d);
//		imgout.setName("Synthlabel_"+nobj);
//		imgout.setHeader(hdr);
		
//		rw.write(imgout, outdir);
		System.out.println("done");
		System.exit(0);
	}


	
}