package edu.jhu.ece.iacl.algorithms.registration;

import edu.jhu.ece.iacl.jist.structures.image.ImageData;


//N-Dimensional Histogram - Recursive Tree Implementation
//If it is a Joint Histogram between subjects and targets, by convention subjects are placed first
public class NDimHistogramTree{
	private NDimHistogramTree[] children;
	private int[] data;
	private int currentDimNum;
	private int numOfBins;
	
	public NDimHistogramTree(int numOfDimensions, int numOfBins) {		
		currentDimNum = numOfDimensions-1;
		this.numOfBins = numOfBins;
		
		if(currentDimNum > 0){//If not on the last dimension, recursively add one dimension lower
			children = new NDimHistogramTree[numOfBins];
			for (int i = 0; i < numOfBins; i ++){
				children[i] = new NDimHistogramTree(numOfDimensions-1, numOfBins);
			}
			data = null;
		}
		else{
			children = null;
			data = new int[numOfBins];
		}
	}
	
	public NDimHistogramTree(int numOfSub, int numOfTar, int numOfBins) {
		this(numOfSub + numOfTar, numOfBins);
	}
	
	//get bin
	public int get(int[] index){
		if(currentDimNum > 0) return children[index[currentDimNum]].get(index);
		else return data[index[currentDimNum]]; 
	}
	public int get(int[] subIndexes, int[] tarIndexes){
		return get(combineIndexes(subIndexes, tarIndexes));
	}

	//set bin
	public void set(int[] index, int val){
		if(currentDimNum > 0) children[index[currentDimNum]].set(index,val);
		else data[index[currentDimNum]] = val;
	}
	public void set(int[] subIndexes, int[] tarIndexes, int val){
		set(combineIndexes(subIndexes, tarIndexes), val);
	}

	//raise bin by one
	public void increment(int[] index){
		if(currentDimNum > 0) children[index[currentDimNum]].increment(index);
		else data[index[currentDimNum]]++;
	}
	public void increment(int[] subIndexes, int[] tarIndexes){
		increment(combineIndexes(subIndexes, tarIndexes));
	}
	
	//drop bin by one	
	public void decrement(int[] index){
		if(currentDimNum > 0) children[index[currentDimNum]].decrement(index);
		else data[index[currentDimNum]]--;
	}
	public void decrement(int[] subIndexes, int[] tarIndexes){
		decrement(combineIndexes(subIndexes, tarIndexes));
	}
	
	
	
	public int[] combineIndexes(int[] subIndexes, int[] tarIndexes){
		int[] combIndexes = new int[subIndexes.length + tarIndexes.length];
		for (int i = 0; i < combIndexes.length; i++){
			if(i < subIndexes.length) combIndexes[i] = subIndexes[i];
			else combIndexes[i] = tarIndexes[i-subIndexes.length];
		}
		return combIndexes;
		
	}
	
	//given subject and target images, fill up the joint histogram
	public void fillJointHistogram(ImageData[] subjects, ImageData[] targets, int[] boundingBox){
		int totalLength = subjects.length + targets.length; 
		if((totalLength) != (currentDimNum + 1)){
			System.out.format("Subjects/Target Length Does Not Match Histogram Size\n");
			return;
		}
		int[] index = new int[totalLength];
		int bin; 
		clearHist();
		
		for (int i = boundingBox[0]; i <= boundingBox[1]; i++) 
			for (int j = boundingBox[2]; j <= boundingBox[3]; j++) 
				for (int k = boundingBox[4]; k <= boundingBox[5]; k++) {

					//get bin index for all subjects and targets
					for(int ch = 0; ch < totalLength; ch++ ){
						if (ch < subjects.length) bin = subjects[ch].getUByte(i, j, k);
						else bin = targets[ch - subjects.length].getUByte(i, j, k);
						
						//Make sure is in bounds
						if (bin >= numOfBins){
							bin = numOfBins - 1;
							System.out.format("Value outside of bin range\n");
						}
						if (bin < 0) {
							bin = 0;
							System.out.format("Value outside of bin range\n");
						}
						index[ch] = bin;
					}
					//increment bin
					increment(index);
		}		
	}
	
	public double calcEntropy(){
		return calcEntropy(getTotalVol());
	}
	public double calcEntropy(int totalVol){
		double entropy = 0;
		double tmp;
		for(int i = 0; i < numOfBins; i++){
			if(currentDimNum > 0) entropy += children[i].calcEntropy(totalVol);
			else {
				if (data[i] > 0) {
					tmp = ((double) data[i]) / totalVol;
					entropy -= tmp * Math.log(tmp);
				}
			}
		}
		
		return entropy;
	}
	
	public NDimHistogramTree clone(){
		NDimHistogramTree clone = new NDimHistogramTree(currentDimNum+1, numOfBins);
		if(currentDimNum > 0){
			for (int i = 0; i < numOfBins; i++){
				clone.children[i] = children[i].clone();			
			}
		}
		else{
			for (int i = 0; i < numOfBins; i++){
				clone.data[i] = data[i]; 
			}
		}
		return clone;
	}

	public void copy(NDimHistogramTree copyFrom){
		if((currentDimNum != copyFrom.currentDimNum) || (numOfBins != copyFrom.numOfBins)){
			System.out.format("Invalid Copy - Objects Don't Match\n");
			return;
		}
		for (int i = 0; i < numOfBins; i++){
			if(currentDimNum > 0) children[i].copy(copyFrom.children[i]);			
			else data[i] = copyFrom.data[i];
		}
	}
	
	public int getDimensionNum() {return currentDimNum;}
	public int getNumOfBins() {return numOfBins;}
	public int getTotalVol() {
		int totalVol = 0;
		for(int i = 0; i < numOfBins; i++){
			if(currentDimNum > 0) totalVol += children[i].getTotalVol();
			else totalVol += data[i];
		}
		return totalVol;
	}
	public void clearHist(){
		for(int i = 0; i < numOfBins; i++){
			if(currentDimNum > 0) children[i].clearHist();
			else data[i] = 0;
		}
	}

	
}
