package edu.umcu.algorithms.quantitative;

import edu.jhu.bme.smile.commons.math.StatisticsDouble;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

public class WildBootstrapT1{
	/* Bootstrap function takes as input:
	 * 	-	Fitobject
	 * 	-	Fit parameters
	 *	-	Data 
	 *	- 	Fitted data
	 *	-	number of repeats
	 * and outputs 
	 * 	-	mean values of fit output
	 *	-	std devs of values of fit outputs
	 *	-	mean SSq errors / residuals
	 */
	double [][] BResults = new double [4][3];
	public double[][] WildBootstrapT1(T1Fitting T1fit, double[] x, double[] y, double TR, String nrpar, int bootreps){
		/*
		 * layout of the wild bootstrap method:
		 * 1- run fit and retrieve residuals
		 * 2- rescale residuals by multiply residuals by Rademacher function and Heteroscedasticity consistend
		 *  covariance matrix estimator (HCCME), if applicable.
		 * 3- create bootstrap sample by multiplying the model by the input parameters and adding random samples of
		 * 		rescaled residuals.
		 * 4- fit bootstrapped sample and store data
		 * 5- repeat step 3 and 4 lot of times
		 * 6- calculate means and standard deviations of fitted values  
		 * */	
		
		// Run initial fit:
		double[] FitResult = T1fit.T1Fit(x, y, TR, nrpar);

		// calculate residuals
		double[] residuals = new double [x.length];
		/* Get function value from I=I0*(1.+(k-1.)*EXP(-TI/T1)-k*EXP(-TR/T1))*/
		double[] v = new double [x.length];
		for (int i=0; i<x.length; i++){
			v[i]=FitResult[0]*(1.+(FitResult[2]-1.)*Math.exp(-x[i]/FitResult[1])-FitResult[2]*Math.exp(-TR/FitResult[1]));
			residuals[i] = y[i] - v[i];
		}

		
		/* HCCME is 1 if all residuals can be considered equal, otherwise HCCME can take a lot of forms
		 * see also: T. Zhu et al. NeuroImage 40 (2008) p.1144-1156
		 * here we assume either equal residuals or
		 * HCCME = sqrt(n/(n-k))
		 * where n = number of samples, and k the number of fitted variables
		 */
		//double HCCME = Math.sqrt(x.length/(x.length-3));
		double HCCME = 1.0;
		

		double[] bfit = new double[3];
		double[] bI0fit = new double[bootreps];
		double[] bt1fit = new double[bootreps];
		double[] bkfit = new double[bootreps];
		double[] bssqfit = new double[bootreps];
		
		//Start bootstrap loop		
		for( int i=0; i<bootreps; i++){
			//make bootstrap sample
			double[] bsample = new double [x.length];
			for(int n=0; n<x.length; n++){
				//get Rademacher value
				int f = 1;
				if(Math.round(Math.random())==0){f = -1;}
				double e = residuals[(int) Math.round(Math.random()*(x.length-1))]; //pick a residual
				bsample[n] = v[n]+(HCCME*e*f);
			}
			//run the fit again
			bfit = T1fit.T1Fit(x, bsample, TR, nrpar);
			bI0fit[i] = bfit[0];
			bt1fit[i] = bfit[1];
			bkfit[i] = bfit[2];
			bssqfit[i] = bfit[3];
			
		}
		
		BResults[0][0] = StatisticsDouble.mean(bI0fit);
		BResults[0][1] = StatisticsDouble.std(bI0fit);
		BResults[1][0] = StatisticsDouble.mean(bt1fit);
		BResults[1][1] = StatisticsDouble.std(bt1fit);
		BResults[2][0] = StatisticsDouble.mean(bkfit);
		BResults[2][1] = StatisticsDouble.std(bkfit);
		BResults[3][0] = StatisticsDouble.mean(bssqfit);
		BResults[3][1] = StatisticsDouble.std(bssqfit);
		
		
		return BResults;
	}
	
}