package edu.umcu.plugins.utilities.roi;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import JSci.maths.ArrayMath;
import Jama.Matrix;

import edu.jhu.ece.iacl.algorithms.CommonAuthors;
import edu.jhu.ece.iacl.algorithms.dti.ParV4Info;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.io.StringArrayXMLReaderWriter;
import edu.jhu.ece.iacl.jist.io.StringReaderWriter;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamObject;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.geom.BndBox;
import edu.jhu.ece.iacl.jist.structures.geom.PT;
import edu.jhu.ece.iacl.jist.structures.geom.Polyhedron;
import edu.jhu.ece.iacl.jist.structures.geom.TriangleSigned;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataDouble;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.umcu.algorithms.volume.SParInfo;

/*
 * @author Daniel Polders (zigurana@gmail.com)
 * This is a very rough attempt to get a mask out of a spectroscopy SPAR (Philips) file.
 * The aim is to provide a means to correlate imaging information to spectroscopy results.
 * This module reads in the planning parameters (offsets, angulations, FOV) for both the imaging
 * and spectro sequence, and defines a mask in image space that has non-zero values for the location
 * the spectro voxel.
 * This approach assumes that these plannings are perfect, and that there was only one spectro voxel,
 * no multi-voxel or SPI sequences allowed ATM. Any ideas on how to tackle partial volume effects are
 * more than welcome!
 * 
 * ~Jan 2011
 */

public class SPAR_mask extends ProcessingAlgorithm {
	/****************************************************
	 * Input Parameters
	 ****************************************************/
	private ParamFile Img_parfile;
	private ParamVolume Img_volume;		
	private ParamFile Spec_sparfile;
	

	/****************************************************
	 * Output Parameters
	 ****************************************************/
	private ParamVolume OutputMask;

	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");
	private static final String shortDescription = "Parses Phillips .par file and .spar file and returns volume with spectroscopy voxel.";
	private static final String longDescription = "The module takes two datasets, one imaging and one (single voxel) spectroscopy, and finds\n" +
			"the location of the spectrovoxel in image space.\n" +
			"Inputs are:\n" +
			"- a PAR or REC file of the imaging volume\n" +
			"- an SPAR file of the spectro dataset\n" +
			"  Output is a mask in image space containing 1's at the locations of the spectroscopy voxel.\n" +
			"!NB: this module assumes that the subject was in the same position, and orientation, it will only take into account differences in planning.\n" +
			"	This is a -very- simple implementation, it does not even check if the spectro volume is inside the imaging volume (yet).";

	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Step 1. Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("UMCU");
		inputParams.setCategory("Utilities");
		inputParams.setLabel("SPAR Mask");
		inputParams.setName("SPAR_Mask");

		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.nitrc.org/projects/jist/");
		info.setAffiliation("UMC Utrecht");
		info.add(new AlgorithmAuthor("Daniel Polders", "daniel.polders@gmail.com", ""));
		info.setDescription(longDescription);
		info.setLongDescription(longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.ALPHA);

		/****************************************************
		 * Step 2. Add input parameters to control system
		 ****************************************************/
		inputParams.add(Img_parfile = new ParamFile("Imaging ParV4 File", new FileExtensionFilter(new String[]{"par","PAR","parv2","rec","REC"})));
		inputParams.add(Img_volume = new ParamVolume("Imaging Volume"));
		inputParams.add(Spec_sparfile = new ParamFile("Spectroscopy SPAR File", new FileExtensionFilter(new String[]{"spar","SPAR"})));
	}


	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Step 1. Add output parameters to control system
		 ****************************************************/
		outputParams.add(OutputMask = new ParamVolume("SpectroMask",VoxelType.DOUBLE));
	}

	@Override
	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute();
	}
	
	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute() {
			/****************************************************
			 * Step 1. Indicate that the plugin has started.
			 ****************************************************/
			this.setLabel("Initializing");
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Start");
			
			/****************************************************
			 * Step 2. Parse the input data
			 ****************************************************/
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Found PAR file: "+Img_parfile.getURI().getPath());
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Found Imaging volume: "+Img_volume.getURI().getPath());
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Found SPAR file: "+Spec_sparfile.getURI().getPath());

			
			/****************************************************
			 * Step 3. Setup memory for the computed volumes
			 ****************************************************/
			//ImageData vol1 = Img_volume.getImageData(); 
			int r = Img_volume.getImageData().getRows();
			int c = Img_volume.getImageData().getCols();
			int s = Img_volume.getImageData().getSlices();
			
			ImageData mask = new ImageDataDouble(r,c,s);
			mask.setHeader(Img_volume.getImageData().getHeader());
			mask.setName(Img_volume.getImageData().getName()+"_"+Spec_sparfile.getValue().getName()+"_Mask");
			StringArrayXMLReaderWriter saxml = new StringArrayXMLReaderWriter();
	
			/****************************************************
			 * Step 4. Run the core algorithm. Note that this program
			 * 		   has NO knowledge of the MIPAV data structure and
			 * 		   uses NO MIPAV specific components. This dramatic
			 * 		   separation is a bit inefficient, but it dramatically
			 * 		   lower the barriers to code re-use in other applications.
			 ****************************************************/
			
			// read in image par information
			ParV4Info img_info = new ParV4Info(Img_parfile.getURI().getPath());
			img_info.getInfo();
			double[] img_ang =img_info.getSliceAngulation(); // returns slice angulations: AP, FH, RL
			double[] img_off = img_info.getOffCentre(); // returns slice offcenters: AP, FH, RL

			// read in spectro spar information
			SParInfo spec_info = new SParInfo(Spec_sparfile.getURI().getPath());
			spec_info.getInfo();
			double[] spe_ang = spec_info.getVoxelAngulation(); //format: ap,rl,fh
			double[] spe_off = spec_info.getOffCentre(); //format: ap,rl,fh
			double[] spe_sizes = spec_info.getSizes(); //format: ap,rl,fh
			
			if(!img_info.getDate().equals(spec_info.getDate())){
				JistLogger.logError(2, "Scan Dates of Imaging volume ("+img_info.getDate()+") and spectro measurement ("+spec_info.getDate()+") are not equal!");
				return;
			}
			if(Img_volume.getImageData().getHeader().getAxisOrientation()[0].name()!="R2L_TYPE"||
					Img_volume.getImageData().getHeader().getAxisOrientation()[1].name()!="A2P_TYPE"||
					Img_volume.getImageData().getHeader().getAxisOrientation()[2].name()!="I2S_TYPE"){
						JistLogger.logError(2, getClass().getCanonicalName()+": Axis orientations ("+Img_volume.getImageData().getHeader().getAxisOrientation()[0]+","
							+Img_volume.getImageData().getHeader().getAxisOrientation()[1]+","+Img_volume.getImageData().getHeader().getAxisOrientation()[2]+") are not supported, aborting.");
						return;
					}
			
			/* reorder angulations and offcenters to reflect RL-AP-FH*/
			double[] img_ang2 = {img_ang[2],img_ang[0], img_ang[1]};
			double[] img_off2 = {img_off[2],img_off[0], img_off[1]};
			double[] spe_ang2 = {spe_ang[1],spe_ang[0], spe_ang[2]};
			double[] spe_off2 = {spe_off[1],spe_off[0], spe_off[2]};
			double[] spe_sizes2 ={spe_sizes[1],spe_sizes[0], spe_sizes[2]};
			
			JistLogger.logOutput(4, getClass().getCanonicalName()+": image angulations (RL-AP-FH):("+img_ang2[0]+","+img_ang2[1]+","+img_ang2[2]+") degrees");
			JistLogger.logOutput(4, getClass().getCanonicalName()+": image offcenters (RL-AP-FH):("+img_off2[0]+","+img_off2[1]+","+img_off2[2]+") millimeter");
			JistLogger.logOutput(4, getClass().getCanonicalName()+": spectro angulations (RL-AP-FH):("+spe_ang2[0]+","+spe_ang2[1]+","+spe_ang2[2]+") degrees");
			JistLogger.logOutput(4, getClass().getCanonicalName()+": spectro offcenters (RL-AP-FH):("+spe_off2[0]+","+spe_off2[1]+","+spe_off2[2]+") millimeter");
			JistLogger.logOutput(4, getClass().getCanonicalName()+": spectro Voxel size (RL-AP-FH):("+spe_sizes2[0]+","+spe_sizes2[1]+","+spe_sizes2[2]+") millimeter");

			
			/* calc rotation matrices from angulations */
			double[][] Rot_img = CalcRotationMatrix(img_ang2);
			double[][] Rot_spe = CalcRotationMatrix(spe_ang2);
			
			/* Get new coordinates of spectro-box in image space */
				//new offcenter for spectro voxel
			double[] offcenter = new double[3];
			offcenter[0] = spe_off2[0]-img_off2[0];
			offcenter[1] = spe_off2[1]-img_off2[1];
			offcenter[2] = spe_off2[2]-img_off2[2];
			
				//offcenter2 = Rot_img * offcenter;
			double[] offcenter2 = new double[3];
			offcenter2[0] = Rot_img[0][0]*offcenter[0]+Rot_img[1][0]*offcenter[1]+Rot_img[2][0]*offcenter[2];
			offcenter2[1] = Rot_img[0][1]*offcenter[0]+Rot_img[1][1]*offcenter[1]+Rot_img[2][1]*offcenter[2];
			offcenter2[2] = Rot_img[0][2]*offcenter[0]+Rot_img[1][2]*offcenter[1]+Rot_img[2][2]*offcenter[2];
			JistLogger.logOutput(4, getClass().getCanonicalName()+": New midpoint of spectro voxel is: ("+offcenter2[0]+","+offcenter2[1]+","+offcenter2[2]+") millimeter");
			
			
				//rot3 = R_img'*R_spec;
			double[][] rot3 = new double[3][3];
			double[][] Rot_imgt = ArrayMath.transpose(Rot_img);
			for(int i = 0; i < 3; i++) {
		      for(int j = 0; j < 3; j++) {
		        for(int k = 0; k < 3; k++){
		        	rot3[i][j] += Rot_imgt[i][k]*Rot_spe[k][j];
		        }
		      }
			}
		    /* calc new corner coords of spectro voxel in image space (in mm) */
		    Matrix Rot3_matrix = new Matrix(rot3);
		    JistLogger.logOutput(4, getClass().getCanonicalName()+": Rot3_matrix matrix calculated");
		    for(int i = 0; i < 3; i++) {
		      JistLogger.logOutput(4, getClass().getCanonicalName()+": {"+Rot3_matrix.getArray()[i][0]+" "+Rot3_matrix.getArray()[i][1]+" "+Rot3_matrix.getArray()[i][2]+"}");
		    }
		    double Rot3_matrix_det = Rot3_matrix.det();
		    
		    JistLogger.logOutput(4, getClass().getCanonicalName()+": Rot3_matrix_det = "+Rot3_matrix.det());
		    
		    Matrix Rot3_matrix_inv =Rot3_matrix.inverse();
		    
		    JistLogger.logOutput(4, getClass().getCanonicalName()+": Rot3_matrix INVERSE matrix calculated");
		    for(int i = 0; i < 3; i++) {
			      JistLogger.logOutput(4, getClass().getCanonicalName()+": {"+Rot3_matrix_inv.getArray()[i][0]+" "+Rot3_matrix_inv.getArray()[i][1]+" "+Rot3_matrix_inv.getArray()[i][2]+"}");
			}
		    
		    //Difference between inverse and transpose, should be 0
		    Matrix difference = Rot3_matrix.transpose().minus(Rot3_matrix_inv);
		    JistLogger.logOutput(4, getClass().getCanonicalName()+": Difference between inverse and transpose, should be 0's");
		    for(int i = 0; i < 3; i++) {
		    	JistLogger.logOutput(4, getClass().getCanonicalName()+": {"+difference.getArray()[i][0]+" "+difference.getArray()[i][1]+" "+difference.getArray()[i][2]+"}");
			}
		    
		    	// get sizes of spectro volume wrt to midpoint
		    double[] size_half = new double[3];
		    size_half[0] = spe_sizes2[0]/2;
		    size_half[1] = spe_sizes2[1]/2;
		    size_half[2] = spe_sizes2[2]/2;
		    Matrix size_halfM = new Matrix(size_half,1);
		    
		    	//Unit vectors along the axis
		    	// This is actually not a unit cube, but a cube with length 2!! (with midpoint being 0,0,0)
		    double[][] c1={{-1, 0, 0},{0,-1,0},{0, 0,-1}};
		    double[][] c2={{-1, 0, 0},{0,-1,0},{0, 0,+1}};
		    double[][] c3={{-1, 0, 0},{0,+1,0},{0, 0,-1}};
		    double[][] c4={{-1, 0, 0},{0,+1,0},{0, 0,+1}};
		    double[][] c5={{+1, 0, 0},{0,-1,0},{0, 0,-1}};
		    double[][] c6={{+1, 0, 0},{0,-1,0},{0, 0,+1}};
		    double[][] c7={{+1, 0, 0},{0,+1,0},{0, 0,-1}};
		    double[][] c8={{+1, 0, 0},{0,+1,0},{0, 0,+1}};
		    Matrix C1 = new Matrix(c1);
		    Matrix C2 = new Matrix(c2);
		    Matrix C3 = new Matrix(c3);
		    Matrix C4 = new Matrix(c4);
		    Matrix C5 = new Matrix(c5);
		    Matrix C6 = new Matrix(c6);
		    Matrix C7 = new Matrix(c7);
		    Matrix C8 = new Matrix(c8);
		    
		    
		   double[] corner1 = ArrayMath.add(Rot3_matrix.times(C1).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner2 = ArrayMath.add(Rot3_matrix.times(C2).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner3 = ArrayMath.add(Rot3_matrix.times(C3).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner4 = ArrayMath.add(Rot3_matrix.times(C4).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner5 = ArrayMath.add(Rot3_matrix.times(C5).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner6 = ArrayMath.add(Rot3_matrix.times(C6).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner7 = ArrayMath.add(Rot3_matrix.times(C7).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   double[] corner8 = ArrayMath.add(Rot3_matrix.times(C8).times(size_halfM.transpose()).getRowPackedCopy(),offcenter2);
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": Following corners calculated (in mm):");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner1: ("+corner1[0]+", "+corner1[1]+", "+corner1[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner2: ("+corner2[0]+", "+corner2[1]+", "+corner2[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner3: ("+corner3[0]+", "+corner3[1]+", "+corner3[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner4: ("+corner4[0]+", "+corner4[1]+", "+corner4[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner5: ("+corner5[0]+", "+corner5[1]+", "+corner5[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner6: ("+corner6[0]+", "+corner6[1]+", "+corner6[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner7: ("+corner7[0]+", "+corner7[1]+", "+corner7[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": corner8: ("+corner8[0]+", "+corner8[1]+", "+corner8[2]+")");
		   
		   // Convert coordinates from mm to (fractional) voxels from midpoint
		   	float[] Img_resolutions = mask.getHeader().getDimResolutions();
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Resolutions of imaging volume are: "+Img_resolutions[0]+","+Img_resolutions[1]+","+Img_resolutions[2]+" "+mask.getHeader().getUnitsOfMeasure()[0]);
			for(int i=0; i<3; i++){
				corner1[i]=corner1[i]/Img_resolutions[i];
				corner2[i]=corner2[i]/Img_resolutions[i];
				corner3[i]=corner3[i]/Img_resolutions[i];
				corner4[i]=corner4[i]/Img_resolutions[i];
				corner5[i]=corner5[i]/Img_resolutions[i];
				corner6[i]=corner6[i]/Img_resolutions[i];
				corner7[i]=corner7[i]/Img_resolutions[i];
				corner8[i]=corner8[i]/Img_resolutions[i];
			}
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": Following corners calculated (in voxels):");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C1: ("+corner1[0]+", "+corner1[1]+", "+corner1[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C2: ("+corner2[0]+", "+corner2[1]+", "+corner2[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C3: ("+corner3[0]+", "+corner3[1]+", "+corner3[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C4: ("+corner4[0]+", "+corner4[1]+", "+corner4[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C5: ("+corner5[0]+", "+corner5[1]+", "+corner5[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C6: ("+corner6[0]+", "+corner6[1]+", "+corner6[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C7: ("+corner7[0]+", "+corner7[1]+", "+corner7[2]+")");
		   JistLogger.logOutput(4, getClass().getCanonicalName()+": C8: ("+corner8[0]+", "+corner8[1]+", "+corner8[2]+")");	
		
			// get offset from image midpoint to image origin (in voxels)
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Dimensions of imaging volume are: "+r+" rows, "+c+" columns, "+s+" slices.");
			double[] offcenter3 = new double[3];
			offcenter3[0] = r/2;
			offcenter3[1] = c/2;
			offcenter3[2] = s/2;
			JistLogger.logOutput(4, getClass().getCanonicalName()+": Offset from image origin in voxels: ("+offcenter3[0]+", "+ offcenter3[1]+ ", "+offcenter3[2]+")");
			
		    //PT punt = new PT(corner1[0]+offcenter2[0], corner1[1]+offcenter2[1], corner1[2]+offcenter2[2]);
//		    PT PT000 = new PT(corner1[0]+offcenter3[0], corner1[1]+offcenter3[1], corner1[2]+offcenter3[2]);
//	        PT PT001 = new PT(corner2[0]+offcenter3[0], corner2[1]+offcenter3[1], corner2[2]+offcenter3[2]);
//	        PT PT010 = new PT(corner3[0]+offcenter3[0], corner3[1]+offcenter3[1], corner3[2]+offcenter3[2]);
//	        PT PT011 = new PT(corner4[0]+offcenter3[0], corner4[1]+offcenter3[1], corner4[2]+offcenter3[2]);
//	        PT PT100 = new PT(corner5[0]+offcenter3[0], corner5[1]+offcenter3[1], corner5[2]+offcenter3[2]);
//	        PT PT101 = new PT(corner6[0]+offcenter3[0], corner6[1]+offcenter3[1], corner6[2]+offcenter3[2]);
//	        PT PT110 = new PT(corner7[0]+offcenter3[0], corner7[1]+offcenter3[1], corner7[2]+offcenter3[2]);
//	        PT PT111 = new PT(corner8[0]+offcenter3[0], corner8[1]+offcenter3[1], corner8[2]+offcenter3[2]);

			///downcast to ints
		    PT PT000 = new PT(Math.round(corner1[0]+offcenter3[0]), Math.round(corner1[1]+offcenter3[1]), Math.round(corner1[2]+offcenter3[2]));
	        PT PT001 = new PT(Math.round(corner2[0]+offcenter3[0]), Math.round(corner2[1]+offcenter3[1]), Math.round(corner2[2]+offcenter3[2]));
	        PT PT010 = new PT(Math.round(corner3[0]+offcenter3[0]), Math.round(corner3[1]+offcenter3[1]), Math.round(corner3[2]+offcenter3[2]));
	        PT PT011 = new PT(Math.round(corner4[0]+offcenter3[0]), Math.round(corner4[1]+offcenter3[1]), Math.round(corner4[2]+offcenter3[2]));
	        PT PT100 = new PT(Math.round(corner5[0]+offcenter3[0]), Math.round(corner5[1]+offcenter3[1]), Math.round(corner5[2]+offcenter3[2]));
	        PT PT101 = new PT(Math.round(corner6[0]+offcenter3[0]), Math.round(corner6[1]+offcenter3[1]), Math.round(corner6[2]+offcenter3[2]));
	        PT PT110 = new PT(Math.round(corner7[0]+offcenter3[0]), Math.round(corner7[1]+offcenter3[1]), Math.round(corner7[2]+offcenter3[2]));
	        PT PT111 = new PT(Math.round(corner8[0]+offcenter3[0]), Math.round(corner8[1]+offcenter3[1]), Math.round(corner8[2]+offcenter3[2]));		    

	        JistLogger.logOutput(4, getClass().getCanonicalName()+": Corners of SpectroVolume in voxels:");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT000 ("+PT000.x +", "+PT000.y +", "+PT000.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT001 ("+PT001.x +", "+PT001.y +", "+PT001.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT010 ("+PT010.x +", "+PT010.y +", "+PT010.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT011 ("+PT011.x +", "+PT011.y +", "+PT011.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT100 ("+PT100.x +", "+PT100.y +", "+PT100.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT101 ("+PT101.x +", "+PT101.y +", "+PT101.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT110 ("+PT110.x +", "+PT110.y +", "+PT110.z +")");
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": PT111 ("+PT111.x +", "+PT111.y +", "+PT111.z +")");
	        
	        		
	        /* create box with these coordinates*/
		    Vector v = new Vector();
	        //left
	        v.add(new TriangleSigned(PT000,PT001,PT010));
	        v.add(new TriangleSigned(PT011,PT001,PT010));
	        //right
	        v.add(new TriangleSigned(PT100,PT101,PT110));
	        v.add(new TriangleSigned(PT111,PT101,PT110));
	        //down
	        v.add(new TriangleSigned(PT000,PT001,PT100));
	        v.add(new TriangleSigned(PT101,PT001,PT100));
	        //up
	        v.add(new TriangleSigned(PT010,PT011,PT110));
	        v.add(new TriangleSigned(PT111,PT011,PT110));
	        //out
	        v.add(new TriangleSigned(PT000,PT010,PT100));
	        v.add(new TriangleSigned(PT110,PT010,PT100));
	        //in
	        v.add(new TriangleSigned(PT001,PT011,PT101));
	        v.add(new TriangleSigned(PT111,PT011,PT101));
	        Polyhedron Spectrovoxel = new Polyhedron(v);
	        
	        // Use bounding box of Spectrovoxel to get range of loops: 
	        BndBox box = Spectrovoxel.getBndBox();
	        int min_rows =(int) Math.floor(box.getMinPt().x);
	        int min_cols =(int) Math.floor(box.getMinPt().y);
	        int min_sli=(int) Math.floor(box.getMinPt().z);
	        int max_rows=(int) Math.ceil(box.getMaxPt().x);
	        int max_cols =(int) Math.ceil(box.getMaxPt().y);
	        int max_sli =(int) Math.ceil(box.getMaxPt().z);
	        
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": min_rows:"+min_rows);
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": max_rows:"+max_rows);
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": min_cols:"+min_cols);
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": max_cols:"+max_cols);
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": min_sli:"+min_sli);
	        JistLogger.logOutput(4, getClass().getCanonicalName()+": max_sli:"+max_sli);
	        /* use box to create mask*/
	        int cnt = 0;
	        this.setLabel("Selecting voxels");
	        this.setTotalUnits(max_rows-min_rows);
			for(int i=min_rows; i < max_rows; i++){
				for (int j=min_cols; j < max_cols; j++){
					for(int k=min_sli; k < max_sli; k++){
						PT p = new PT((double)i+0.5,(double)j+0.5,(double)k+0.5); // plus 0.5 to get to "middle" of voxel
						int hit = 0;
						int miss = 0 ;
						//for (int n=0; n < 10; n++){
							if(Spectrovoxel.contains(p)){
								hit ++;	
							}else{
								miss++;
							}
						//}
						//JistLogger.logOutput(4, getClass().getCanonicalName()+": Test if ("+i+","+j+","+k+") is inside Spectrovoxel: (Hit / Miss)= "+hit+" / "+ miss);
						if (hit > 0){
							mask.set(i, j, k, 1); 
				        	cnt++;
				        }
					}
				}
				this.setCompletedUnits(i-min_rows);
			}
	        
			JistLogger.logOutput(2, getClass().getCanonicalName()+": Done creating mask, "+cnt+" voxels were selected.");
			
			/****************************************************
			 * Step 5. Retrieve the image data and put it into a new
			 * 			data structure. Be sure to update the file information
			 * 			so that the resulting image has the correct
			 * 		 	field of view, resolution, etc.
			 ****************************************************/
			this.setLabel("Saving Volume");
			OutputMask.setValue(mask);
		}
	}



	private double[][] CalcRotationMatrix(double[] ang){
		//assumes angles to be given in degrees, so its first converted to radians
		double[] angrad = new double[3];
		angrad[0] = Math.toRadians(ang[0]);
		angrad[1] = Math.toRadians(ang[1]);
		angrad[2] = Math.toRadians(ang[2]);
		
		
//		double [][] R_rl = {{1,0,0},{0,Math.cos(angrad[0]), -Math.sin(angrad[0])},{0,Math.sin(angrad[0]), -Math.cos(angrad[0])}};
//		double [][] R_ap = {{Math.cos(angrad[1]), 0, Math.sin(angrad[1])},{0,1,0},{-Math.sin(angrad[1]), 0, Math.cos(angrad[1])}};
//		double [][] R_fh = {{Math.cos(angrad[2]),-Math.sin(angrad[2]),0},{Math.sin(angrad[2]), Math.cos(angrad[2]),0},{0,0,1}};
		
		// Matlab snippet:
		//R_rl=[1 0 0;0 cosd(angles(1)) -sind(angles(1)); 0 sind(angles(1)) cosd(angles(1))];
	    //R_ap=[cosd(angles(2)) 0 sind(angles(2)); 0 1 0; -sind(angles(2)) 0 cosd(angles(2))];
	    //R_fh=[cosd(angles(3)) -sind(angles(3)) 0; sind(angles(3)) cosd(angles(3)) 0; 0 0 1];
	    //R=  R_rl*R_ap*R_fh;
//		%     R_onestep = [c2*c3, -c1*s3+s1*s2*c3, s1*s3+c1*s2*c3;...
//		%                  c2*s3, c1*c3+s1*s2*s3, -s1*c3+c1*s2*s3;...
//		%                  -s2, s1*c2, c1*c2];
//		    R_onestep = [cosd(angles(2))*cosd(angles(3)), -cosd(angles(1))*sind(angles(3))+sind(angles(1))*sind(angles(2))*cosd(angles(3)), sind(angles(1))*sind(angles(3))+cosd(angles(1))*sind(angles(2))*cosd(angles(3));...
//		        cosd(angles(2))*sind(angles(3)), cosd(angles(1))*cosd(angles(3))+sind(angles(1))*sind(angles(2))*sind(angles(3)), -sind(angles(1))*cosd(angles(3))+cosd(angles(1))*sind(angles(2))*sind(angles(3));...
//		                 -sind(angles(2)), sind(angles(1))*cosd(angles(2)), cosd(angles(1))*cosd(angles(2))];
		double s1 = Math.sin(angrad[0]);
		double s2 = Math.sin(angrad[1]);
		double s3 = Math.sin(angrad[2]);
		double c1 = Math.cos(angrad[0]);
		double c2 = Math.cos(angrad[1]);
		double c3 = Math.cos(angrad[2]);
		//double[][] RotMatrix = new double[3][3];
		double [][] RotMatrix = {{(c2*c3), (-c1*s3+s1*s2*c3), (s1*s3+c1*s2*c3)},
				     			 {(c2*s3), (c1*c3+s1*s2*s3), (-s1*c3+c1*s2*s3)},
				     			 {(-s2), (s1*c2), (c1*c2)}};
		return RotMatrix;
		
	}
}
