package edu.vanderbilt.VUIIS.plugins;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;

import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.PipeAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.pipeline.parser.MipavPluginParser;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipavWrapper;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.model.algorithms.registration.AlgorithmDemonsLite;
import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.view.dialogs.ActionDiscovery;


public class DemonsRegCEST extends ProcessingAlgorithm{
	/****************************************************
	 * Declare Input Parameters
	 ****************************************************/
	private ParamVolume cestVol;	// 4-D Volume containing CEST data with a range of offsets
	private ParamVolume anatVol;
	private ParamInteger levels;
	private ParamInteger iter;
	private ParamFloat smooth;
	private ParamFloat scale;
	private ParamOption reg;
	private ParamOption outputType;
		
	/****************************************************
	 * Declare Output Parameters
	 ****************************************************/
	private ParamVolume regVol; // An Image Volume containing CEST Data
	private ParamVolumeCollection transVol;

	/****************************************************
	 * Declare Plugin Information Variables
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Demons Registration for CEST Volumes";
	private static final String longDescription = "\nInput 1: 4-D CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies] " +
	"\nInput 2: 3-D Anatomical Volume (x,y,z):" +
	" [rows,columns,slices]"+
	"\nOutput 1: 4-D Registered CEST Volume (x,y,z,t): " +
	" [rows,columns,slices,offset frequencies]" +
	"\nOutput 2: Execution Time [sec]; " +
	"\n\tThe time to run the algorithm.";


	protected void createInputParameters(ParamCollection inputParams) {
		/****************************************************
		 * Set Plugin Information
		 ****************************************************/
		inputParams.setPackage("VUIIS");
		inputParams.setCategory("Registration");
		inputParams.setLabel("CEST Demons Registration");
		inputParams.setName("CEST Demons Registration");

		AlgorithmInformation info = getAlgorithmInformation();
		info.add(new AlgorithmAuthor("Blake Dewey","blake.e.dewey@vanderbilt.edu",""));
		info.setAffiliation("Vanderbilt University");
		info.setDescription(shortDescription + longDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.add(new Citation(""));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		/****************************************************
		 * Add input parameters to control system
		 ****************************************************/
		ParamCollection mainParams = new ParamCollection("Main");
		ParamCollection advParams = new ParamCollection("Advanced");
		mainParams.add(cestVol = new ParamVolume("CEST Volume (4D)",null,-1,-1,-1,-1));
		mainParams.add(anatVol = new ParamVolume("Anatomical Volume (3D)",null,-1,-1,-1,1));
		anatVol.setMandatory(false);
		mainParams.add(outputType = new ParamOption("Output Type", new String[]{"Transformed Image","Transformation Field","All Images"}));
		advParams.add(levels = new ParamInteger("Pyramid Levels",4));
		advParams.add(iter = new ParamInteger("Number of Iterations",20));
		advParams.add(smooth = new ParamFloat("Smoothing Kernal",(float)2.0));
		advParams.add(scale = new ParamFloat("Scale",(float)1.0));
		advParams.add(reg = new ParamOption("Regularization Type",new String[]{"Diffusion-Like","Fluid-Like"}));
		
		inputParams.add(mainParams);
		inputParams.add(advParams);
	}


	protected void createOutputParameters(ParamCollection outputParams) {
		/****************************************************
		 * Add output parameters to control system
		 ****************************************************/
		outputParams.add(regVol = new ParamVolume("Registered CEST Volume (4D)",VoxelType.FLOAT,-1,-1,-1,-1));
		outputParams.add(transVol = new ParamVolumeCollection("Transformation Volumes",VoxelType.FLOAT,-1,-1,-1,-1));
	}


	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		/****************************************************
		 * Create a Wrapper to Enclose the Algorithm
		 ****************************************************/
		AlgorithmWrapper wrapper=new AlgorithmWrapper();
		monitor.observe(wrapper);
		wrapper.execute(this);
	}
	

	protected class AlgorithmWrapper extends AbstractCalculation {
		protected void execute(ProcessingAlgorithm parent) {
			this.setLabel("PARSING INPUTS");
			
			/****************************************************
			 * Indicate that the Plugin has started.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PLUGIN STARTED");
			
			/****************************************************
			 * Parse the input data
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "PARSING INPUT DATA");
			this.setLabel("PARSING INPUTS");
			ImageData cest = cestVol.getImageData();	// Extract CEST Data from Volume
			int rcest = cest.getRows(); int ccest = cest.getCols(); int scest = cest.getSlices(); int tcest = cest.getComponents();	// Assign Variables for Dimensions of Volume
			ImageData anat = null;
			
			if(anatVol.getImageData() != null){	// Only extract anat data if Mask Volume is not Empty
				anat = anatVol.getImageData();	// Extract Anat data from Volume
				int ranat = anat.getRows(); int canat = anat.getCols(); int sanat = anat.getSlices();	// Assign Anat Dimension Variables
				JistLogger.logOutput(JistLogger.CONFIG, "ANAT VOLUME ADDED");
				// Check if cest and mask sizes match
				if(rcest != ranat || ccest != canat || scest != sanat){
					JistLogger.logError(JistLogger.SEVERE,"Image and Anat Volume Sizes do not match. Aborting.");
					return;
				}
			}
			
			String regString = null;
			String outputString = null;
			switch(reg.getIndex()){
			case 0:
				regString = "diffusion-like";
				break;
			case 1:
				regString = "fluid-like";
				break;
			}
			switch(outputType.getIndex()){
			case 0:
				outputString = "transformed_image";
				break;
			case 1:
				outputString = "transformation_field";
				break;
			case 2:
				outputString = "all_images";
				break;
			}
			
			
			JistLogger.logOutput(JistLogger.CONFIG, "INPUTS CHECKED");
			/****************************************************
			 * Setup memory for the computed volumes
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "ALLOCATING MEMORY");
			ImageData regData = new ImageDataFloat(rcest,ccest,scest,tcest);
			regData.setHeader(cest.getHeader());
			ImageData source = null;
			ImageData target = null;
			ImageData regimg = null;
			ImageData transimg = null;
			ImageData tmp1 = null;
			ImageData tmp2 = null;
			/****************************************************
			 * Run the core algorithm(s).
			 ****************************************************/
			this.setLabel("DEMONS CALC");
			this.setTotalUnits(tcest);
			for(int m=0;m<tcest;m++){
				this.setCompletedUnits(m);
				if(m==0 && anatVol.getImageData() == null){
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								regData.set(i,j,k,0,cest.getFloat(i,j,k,0));
							}
						}
					}
				}
				else{
				target = new ImageDataFloat(rcest,ccest,scest);
				target.setHeader(cest.getHeader());
				source = new ImageDataFloat(rcest,ccest,scest);
				source.setHeader(cest.getHeader());
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								if(m != 0){
									target.set(i,j,k,regData.getFloat(i,j,k,m-1));
								}
								else{
									target.set(i,j,k,anat.getFloat(i,j,k));
								}
								source.set(i,j,k,cest.getFloat(i,j,k,m));
							}
						}
					}
					
//					File dir = new File(getOutputDirectory()+"/demons_"+m);
//					dir.mkdir();
//					
//					ActionDiscovery plugin = null;
//					try {
//						plugin = (ActionDiscovery) Class.forName("gov.nih.mipav.view.dialogs.JDialogDemonsLite").newInstance();
//						
//					} catch (InstantiationException e) {
//						// TODO Auto-generated catch block
//						e.printStackTrace();
//					} catch (IllegalAccessException e) {
//						// TODO Auto-generated catch block
//						e.printStackTrace();
//					} catch (ClassNotFoundException e) {
//						// TODO Auto-generated catch block
//						e.printStackTrace();
//					}
//					
//					PipeAlgorithm pa = MipavPluginParser.parsePipeAlgorithm(plugin);
//					ProcessingAlgorithm pra = pa.getAlgorithm();
//					ParamVolume pm1 = (ParamVolume)pra.getInput().getFirstChildByName("input_image_1");
//					ParamVolume pm2 = (ParamVolume)pra.getInput().getFirstChildByName("input_image_2");
//					ParamString pm3 = (ParamString)pra.getInput().getFirstChildByName("output_type");
//					pra.setOutputDirectory(dir);
//					pm3.setValue("all_images");
//					pm1.setValue(source);
//					pm2.setValue(target);
//			
//					pra.runAlgorithm(monitor);
//					
//					regimg = new ImageDataFloat(((ParamVolume)pra.getOutput().getFirstChildByName("result_image")).getImageData());
//					transimg = new ImageDataFloat(((ParamVolumeCollection)pra.getOutput().getFirstChildByName("Unexpected Outputs")).getImageDataList().get(0));
					
					target.setHeader(cest.getHeader());
					target.setName(cest.getName()+"_target_"+m);
					ModelImage targetImg = target.getModelImageCopy();
					source.setHeader(cest.getHeader());
					source.setName(cest.getName()+"_source_"+m);
					ModelImage srcImg = source.getModelImageCopy();
					
					AlgorithmDemonsLite demons = new AlgorithmDemonsLite(srcImg,targetImg,levels.getInt(),iter.getInt(),smooth.getFloat(),scale.getFloat(),regString,outputString);
					demons.runAlgorithm();
					
					tmp1 = new ImageDataFloat(rcest,ccest,scest);
					tmp2 = new ImageDataFloat(rcest,ccest,scest);
					switch(outputType.getIndex()){
					case 0:
						tmp1 = new ImageDataMipavWrapper(demons.getResultImages()[0]);
						tmp1.setName(cest.getName()+"_reg_"+m);
						tmp1.setHeader(cest.getHeader());
						break;
					case 1:
						tmp2 = new ImageDataMipavWrapper(demons.getResultImages()[0]);
						tmp2.setName(cest.getName()+"_trans_"+m);
						tmp2.setHeader(cest.getHeader());
						break;
					case 2:
						tmp1 = new ImageDataMipavWrapper(demons.getResultImages()[0]);
						tmp1.setName(cest.getName()+"_reg_"+m);
						tmp1.setHeader(cest.getHeader());
						tmp2 = new ImageDataMipavWrapper(demons.getResultImages()[1]);
						tmp2.setName(cest.getName()+"_trans_"+m);
						tmp2.setHeader(cest.getHeader());
						break;
					}
					
					regimg = new ImageDataFloat(rcest,ccest,scest);
					regimg.setName(cest.getName()+"_reg_"+m);
					regimg.setHeader(cest.getHeader());
					transimg = new ImageDataFloat(rcest,ccest,scest);
					transimg.setName(cest.getName()+"_trans_"+m);
					transimg.setHeader(cest.getHeader());
					for(int i=0;i<rcest;i++){
						for(int j=0;j<ccest;j++){
							for(int k=0;k<scest;k++){
								regimg.set(i,j,k,tmp1.getFloat(i,j,k));
								transimg.set(i,j,k,tmp2.getFloat(i,j,k));
								if(outputType.getIndex() == 0 || outputType.getIndex() == 2){
									regData.set(i,j,k,m,regimg.getFloat(i,j,k));
								}
								else{
									regData.set(i,j,k,m,cest.getFloat(i,j,k,m));
								}
							}
						}
					}
					
					if(outputType.getIndex() == 1 || outputType.getIndex() == 2){
						transVol.add(transimg);
					}
					else{
						ImageData tmp = new ImageDataFloat(rcest,ccest,scest);
						tmp.setName(cest.getName()+"_trans");
						tmp.setHeader(cest.getHeader());
						transVol.add(tmp);
					}
					
//					dir.delete();
//					pra.finalize();
//					pa = null;
//					pra = null;
//					pm1 = null;
//					pm2 = null;
					demons.finalize();
					targetImg.disposeLocal();
					srcImg.disposeLocal();
					demons = null;
					tmp1 = null;
					tmp2 = null;
					regimg = null;
					transimg = null;
					targetImg = null;
					srcImg = null;
					target = null;
					source = null;
				}
			}

			/****************************************************
			 * Retrieve the image data and put it into a new
			 * data structure.
			 ****************************************************/
			JistLogger.logOutput(JistLogger.INFO, "SETTING UP EXPORTS");
			if(outputType.getIndex() == 0 || outputType.getIndex() == 2){
				regData.setName(cest.getName()+"_demons");	// Set CEST volume Name
			}
			else{
				regData.setName(cest.getName());	// Set CEST volume Name
			}
			regData.setHeader(cest.getHeader());	// Set CEST volume Header
			regVol.setValue(regData);	// Assigns Local Variable to Output Parameter
			JistLogger.logOutput(JistLogger.INFO, "FINISHED");
		}
	}
}
