package edu.vanderbilt.masi.algorithms.labelfusion;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

public class MSObservationVolumes {

	private int r,c,s;
	private int numRaters;
	private boolean useIntensity;
	private short[][][][][] obs; // rows x columns x slices x raters x labels
	private float[][][][][] vals;
	private int[][][][] oObs;  // rows x columns x slices x raters
	private float[][][][] ims; // rows x columns x slices x raters
	private boolean[][][] cons; // rows x columns x slices (consensus voxels)
	private float[][][] target; // rows x columns x slices
	private float[][][] currIm; // rows x columns x slices
	private int[] croppingRegionX;
	private int[] croppingRegionY;
	private int[] croppingRegionZ;
	private String[] classes;
	private JistPreferences prefs;
	private int searchVolume;
	private int[] sv;
	private int patchVolume;
	private int[] pv;
	private float msdWeight;
	private float lnccWeight;
	private float searchWeight;
	private float selectThresh;
	private int patchCount;
	private float[] dimres;
	private int[] origDims;
	private float[][][] df;
	private float[][][] targetMeans;
	private float[][][] targetSTDs;
	private float[][][] workingMeans;
	private float[][][] workingSTDs;
	private float[] svSTD; // Search volume standard deviation
	private int numConsensus;
	private int numVoxels;
	private int numToProcess;
	private int maxLabel;

	public MSObservationVolumes(ImageData targetIn, List<ImageData> regIms, 
			List<ImageData> regLabs,String[] cl,int searchVolume_in,int patchVolume_in
			,float msdWeight_in,float lnccWeight_in, float searchWeight_in,float selectThresh_in,int patchCount_in){
		classes = cl;
		prefs = JistPreferences.getPreferences();
		maxLabel = -1;
		searchVolume = searchVolume_in;
		patchVolume = patchVolume_in;
		if(msdWeight_in > 0)
			msdWeight = (float) (1/Math.pow(msdWeight_in,2));
		else
			msdWeight = 0;
		if(lnccWeight_in > 0)
			lnccWeight = (float) (1/Math.pow(lnccWeight_in,2));
		else
			lnccWeight = 0; 
		searchWeight = searchWeight_in;
		selectThresh = selectThresh_in;
		patchCount = patchCount_in;

		// Get dimensions
		r = targetIn.getRows();
		c = targetIn.getCols();
		s = targetIn.getSlices();
		origDims = new int[3];
		origDims[0] = r;
		origDims[1] = c;
		origDims[2] = s;
		numRaters = regLabs.size();
		dimres = targetIn.getHeader().clone().getDimResolutions();

		target = loadImage(targetIn);

		// Make sure everything is there that we expect
		useIntensity = true;
		if(regIms.size() != regLabs.size()){
			JistLogger.logError(JistLogger.WARNING,"Error the number of Image Volumes and Label Volumes was not equal");
			JistLogger.logError(JistLogger.WARNING,"Proceeding without non-local correspondance");
			useIntensity = false;
		}

		// Load Data
		JistLogger.logOutput(JistLogger.SEVERE, "Loading Volumes");
		JistLogger.logFlush();
		int origLevel = prefs.getDebugLevel();
		prefs.setDebugLevel(JistLogger.SEVERE);
		checkImages(regLabs,regIms);
		loadLabels(regLabs);
		if(useIntensity)
			loadImages(regIms);
		prefs.setDebugLevel(origLevel);

		JistLogger.logOutput(JistLogger.INFO, String.format("%d Images of size [%d,%d,%d] have been loaded", numRaters,r,c,s));
		JistLogger.logFlush();
		System.gc();

		// Process Some Things to Make this More Intelligent
		determineCroppingRegion();
		mapSpace();
		determineConsensusVoxels();
		normalizeImages();
		// Lets run the magic
		runNLC();
	}

	private void setDistFactor(float [] sp_stdevs) {

		df = new float[2*sv[0]+1][2*sv[1]+1][2*sv[2]+1];

		float ffx = (float) (-1 / Math.pow(sp_stdevs[0], 2));
		float ffy = (float) (-1 / Math.pow(sp_stdevs[1], 2));
		float ffz = (float) (-1 / Math.pow(sp_stdevs[2], 2));

		int svxf = 2*sv[0]+1;
		int svyf = 2*sv[1]+1;
		int svzf = 2*sv[2]+1;

		for (int x = 0; x < svxf; x++)
			for (int y = 0; y < svyf; y++)
				for (int z = 0; z < svzf; z++)
					df[x][y][z] = (float)
						(Math.exp(ffx * Math.pow((float)(sv[0]-x), 2)) *
						 Math.exp(ffy * Math.pow((float)(sv[1]-y), 2)) *
						 Math.exp(ffz * Math.pow((float)(sv[2]-z), 2)));
	}

	private void runNLC(){
		JistLogger.logOutput(JistLogger.INFO, "-> Allocating Memory");
		JistLogger.logFlush();
		obs = new short[r][c][s][numRaters][];
		vals = new float[r][c][s][numRaters][];
		pv = new int[3];
		sv = new int[3];
		svSTD = new float[3];
		for(int i=0;i<3;i++){
			pv[i] = (int) (patchVolume / dimres[i]);
			sv[i] = (int) (searchVolume / dimres[i]);
			svSTD[i] = (float) (1/searchWeight) / dimres[i];
		}
		setDistFactor(svSTD);
		int numPatches = (2*sv[0]+1)*(2*sv[1]+1)*(2*sv[2]+1);
		if(numPatches < patchCount)
			patchCount = numPatches;
		JistLogger.logOutput(JistLogger.INFO, String.format("The resolutions of the image are [%f,%f,%f]", dimres[0],dimres[1],dimres[2]));
		JistLogger.logOutput(JistLogger.INFO, String.format("Keeping %d patches out of %d",patchCount,numPatches));
		JistLogger.logOutput(JistLogger.INFO, String.format("The image size is [%d %d %d]", r,c,s));
		JistLogger.logOutput(JistLogger.INFO, String.format("Search Volume Radius (voxels): [%d %d %d]", sv[0], sv[1], sv[2]));
		JistLogger.logOutput(JistLogger.INFO, String.format("Search Volume Dimensions (voxels): [%d %d %d]", 2*sv[0]+1, 2*sv[1]+1, 2*sv[2]+1));
		JistLogger.logOutput(JistLogger.INFO, String.format("Search Volume Standard Deviations [%f %f %f]", svSTD[0],svSTD[2],svSTD[2]));
		JistLogger.logOutput(JistLogger.INFO, String.format("Patch Volume Radius (voxels): [%d %d %d]", pv[0], pv[1], pv[2]));
		JistLogger.logOutput(JistLogger.INFO, String.format("Patch Volume Dimensions (voxels): [%d %d %d]", 2*pv[0]+1, 2*pv[1]+1, 2*pv[2]+1));
		JistLogger.logOutput(JistLogger.INFO, String.format("The maximum observed label was %d", maxLabel));
		JistLogger.logOutput(JistLogger.INFO, String.format("The coefficient for LNCC is %f", lnccWeight));
		JistLogger.logOutput(JistLogger.INFO, String.format("The coefficient for MSD is %f", msdWeight));
		JistLogger.logFlush();
		targetMeans = new float[r][c][s];
		targetSTDs  = new float[r][c][s];
		set_LNCC_parameters(target,targetMeans,targetSTDs);
		for(int i=0;i<numRaters;i++){
			currIm = snagImage(i);
			workingMeans = new float[r][c][s];
			workingSTDs = new float[r][c][s];
			set_LNCC_parameters(currIm,workingMeans,workingSTDs);
			runNLCIm(i);
		}
	}

	private void runNLCIm(int idx){
		JistLogger.logOutput(JistLogger.INFO, String.format("-> Running Non-Local Correspondence on volume %d", idx));
		int prev = -1;
		int curr;
		int numDone = 0;
		int oneTenth = numToProcess/10;
		for(int i=0;i<r;i++){
			for(int j=0;j<c;j++){
				for(int k=0;k<s;k++){
					if(!cons[i][j][k]){
						curr = numDone/oneTenth;
						if(curr != prev){
							printStatusBar(curr,10);
							prev = curr;
						}
						runNLCVox(i,j,k,idx);
						numDone++;
					}
				}
			}
		}
	}

	private void runNLCVox(int x,int y,int z,int idx){
		int[] svs = new int[3]; // search volume starts
		int[] sve = new int[3]; // search volume ends
		svs[0] = Math.max(x - sv[0], 0);
		svs[1] = Math.max(y - sv[1], 0);
		svs[2] = Math.max(z - sv[2], 0);
		sve[0] = Math.min(x + sv[0], r-1);
		sve[1] = Math.min(y + sv[1], c-1);
		sve[2] = Math.min(z + sv[2], s-1);
		float diff;
		int lab;
		HashMap<Integer,Float> labMap = new HashMap<Integer,Float>();
		// Calculate for all locations in patch
		for(int xi=svs[0];xi<=sve[0];xi++)
			for(int yi=svs[1];yi<=sve[1];yi++)
				for(int zi=svs[2];zi<=sve[2];zi++){
					diff = runNLCPatch(x,y,z,xi,yi,zi);
					lab = oObs[xi][yi][zi][idx];
					if(!labMap.containsKey(lab))
						labMap.put(lab, 0f);
					float val = labMap.get(lab);
					val += diff;
					labMap.put(lab, val);
				}
		// Normalize
		float norm = 0f;
		int num = 0;
		for(int i:labMap.keySet()){
			float val = labMap.get(i);
			if(val > 0){
				norm += val;
				num++;
			}
		}
		if(norm <=0){
			JistLogger.logError(JistLogger.INFO,String.format("Voxel [%d %d %d] had a normalization constant of %f", x,y,z,norm));
			JistLogger.logError(JistLogger.INFO,String.format("There were %d labels nearby", labMap.size()));
			String str1 = "The labels were [ ";
			String str2 = "The weights were[ ";
			for(int i:labMap.keySet()){
				str1 += i;
				str1 += " ";
				str2 += labMap.get(i);
				str2 += " ";
			}
			str1 += "]";
			str2 += "]";
			JistLogger.logError(JistLogger.INFO, str1);
			JistLogger.logError(JistLogger.INFO, str2);
			for(int xi=svs[0];xi<=sve[0];xi++)
				for(int yi=svs[1];yi<=sve[1];yi++)
					for(int zi=svs[2];zi<=sve[2];zi++){
						diff = runNLCPatch(x,y,z,xi,yi,zi);
						lab = oObs[xi][yi][zi][idx];
						if(!labMap.containsKey(lab))
							labMap.put(lab, 0f);
						float val = labMap.get(lab);
						val += diff;
						labMap.put(lab, val);
						JistLogger.logError(JistLogger.INFO, String.format("[%d %d %d] %d %f",xi,yi,zi,lab,diff));
						
					}
		}
		short[] labs = new short[num];
		float[] valS = new float[num];
		num=0;
		for(int i:labMap.keySet()){
			float val = labMap.get(i);
			if(val > 0){
				labs[num] = (short) i;
				valS[num] = val/norm;
				num++;
			}
		}
		// Put the Labels where they need to go
		obs[x][y][z][idx] = labs;
		vals[x][y][z][idx] = valS;
	}

	private float runNLCPatch(int x,int y,int z,int xi, int yi, int zi){
		float diffLNCC = 1f;
		float diffMSD  = 1f;
		float diffDist = 1f;
		if(lnccWeight > 0){
			diffLNCC = get_LNCC_diff(x,y,z,xi,yi,zi,target,currIm);
			//diffLNCC = (float) Math.exp(diffLNCC * (1/lnccWeight)); /This worked better for some reason
			diffLNCC = (float) Math.exp(diffLNCC * lnccWeight);
		}
		if(msdWeight > 0){
			diffMSD = get_MSD_diff(x,y,z,xi,yi,zi,target,currIm);
			//diffMSD = (float) Math.exp(diffMSD * (1/msdWeight)); /This worked better for some reason
			diffMSD = (float) Math.exp(diffMSD * msdWeight);
		}
		diffDist = df[Math.abs(x-xi)][Math.abs(y-yi)][Math.abs(z-zi)];
		float diff = (float) (diffDist * diffMSD *diffLNCC);
		return diff;
	}

	private float get_LNCC_diff(int x,
			int y,
			int z,
			int xi,
			int yi,
			int zi,
			float [][][] target,
			float [][][] im) {


		float diff = 0;
		float count = 0;
		float tval, ival;
		float tmean = targetMeans[x][y][z];
		float tstd = targetSTDs[x][y][z];
		float imean = workingMeans[xi][yi][zi];
		float istd = workingSTDs[xi][yi][zi];

		int xm = Math.min(Math.min(x, xi), r - Math.max(x, xi) - 1);
		int ym = Math.min(Math.min(y, yi), c - Math.max(y, yi) - 1);
		int zm = Math.min(Math.min(z, zi), s - Math.max(z, zi) - 1);

		int xs = (xm < pv[0]) ? xm : pv[0];
		int ys = (ym < pv[1]) ? ym : pv[1];
		int zs = (zm < pv[2]) ? zm : pv[2];
		
		for (int xp = -xs; xp <= xs; xp++)
			for (int yp = -ys; yp <= ys; yp++)
				for (int zp = -zs; zp <= zs; zp++) {
					tval = target[x+xp][y+yp][z+zp];
					ival = im[xi+xp][yi+yp][zi+zp];
					diff += (tval - tmean)*(ival - imean);
					count++;
				}
		diff /= count * (tstd * istd);

		// correct some minor errors
		if (diff < -1)
			diff = -1;
		if (diff > 1)
			diff = 1;

		return(diff-1);
	}

	private float get_MSD_diff(int x,
			int y,
			int z,
			int xi,
			int yi,
			int zi,
			float [][][] target,
			float [][][] im) {
		float diff = 0;
		float tval, ival;

		int xm = Math.min(Math.min(x, xi), r - Math.max(x, xi) - 1);
		int ym = Math.min(Math.min(y, yi), c - Math.max(y, yi) - 1);
		int zm = Math.min(Math.min(z, zi), s - Math.max(z, zi) - 1);

		int xs = (xm < pv[0]) ? xm : pv[0];
		int ys = (ym < pv[1]) ? ym : pv[1];
		int zs = (zm < pv[2]) ? zm : pv[2];

		for (int xp = -xs; xp <= xs; xp++)
			for (int yp = -ys; yp <= ys; yp++)
				for (int zp = -zs; zp <= zs; zp++){
					tval = target[x+xp][y+yp][z+zp];
					ival = im[xi+xp][yi+yp][zi+zp];
					diff += (tval - ival) * (tval - ival);
				}
		diff /= (2*xs+1) * (2*ys+1) * (2*zs+1) ; 

		return(-diff);

	}

	private void set_LNCC_parameters(float [][][] im,
			float [][][] means,
			float [][][] stds) {

		JistLogger.logOutput(JistLogger.INFO, "-> Calculating image LNCC parameters");

		float val, meanval, stdval, count;

		for (int x = 0; x < r; x++)
			for (int y = 0; y < c; y++)
				for (int z = 0; z < s; z++) {

					meanval = 0; stdval = 0; count = 0;

					// set the current region of interest
					int xl = Math.max(x-pv[0], 0);
					int xh = Math.min(x+pv[0], r-1);
					int yl = Math.max(y-pv[1], 0);
					int yh = Math.min(y+pv[1], c-1);
					int zl = Math.max(z-pv[2], 0);
					int zh = Math.min(z+pv[2], s-1);

					// set the mean
					for (int xi = xl; xi <= xh; xi++)
						for (int yi = yl; yi <= yh; yi++)
							for (int zi = zl; zi <= zh; zi++){
								meanval += im[xi][yi][zi];
								count++;
							}
					meanval /= count;

					// calculate the standard deviation
					for (int xi = xl; xi <= xh; xi++)
						for (int yi = yl; yi <= yh; yi++)
							for (int zi = zl; zi <= zh; zi++) {
								val = im[xi][yi][zi] - meanval;
								stdval += val*val;
							}
					stdval = (float)Math.sqrt(stdval / count);
					means[x][y][z] = meanval;
					if (stdval == 0)
						stdval = 0.001f;
					stds[x][y][z] = stdval; 

				}
	}

	private void normalizeImages(){
		MSIntensityNormalizer MSIN = new MSIntensityNormalizer(oObs,target);
		for(int i=0;i<numRaters;i++){
			JistLogger.logOutput(JistLogger.INFO, String.format("++ Normalizing Image %d ++", i));
			float[][][] im = snagImage(i);
			im = MSIN.normalizeImage(im);
			placeImage(im,i);
		}
	}

	private float[][][] snagImage(int idx){
		float[][][] im = new float[r][c][s];
		for(int i=0;i<r;i++)
			for(int j=0;j<c;j++)
				for(int k=0;k<s;k++)
					im[i][j][k] = ims[i][j][k][idx];
		return im;
	}

	private void mapSpace(){
		JistLogger.logOutput(JistLogger.INFO, "Re-mapping image to cropped space");
		int rn = croppingRegionX[1] - croppingRegionX[0] + 1;
		int cn = croppingRegionY[1] - croppingRegionY[0] + 1;
		int sn = croppingRegionZ[1] - croppingRegionZ[0] + 1;
		int[][][][] tempLabels = new int[rn][cn][sn][numRaters];
		float[][][][] tempImage = null;
		if(useIntensity)
			tempImage = new float[rn][cn][sn][numRaters];
		for(int i=0;i<rn;i++)
			for(int j=0;j<cn;j++)
				for(int k=0;k<sn;k++)
					for(int l=0;l<numRaters;l++){
						tempLabels[i][j][k][l] = oObs[i+croppingRegionX[0]][j+croppingRegionY[0]][k+croppingRegionZ[0]][l];
						if(useIntensity)
							tempImage[i][j][k][l] = ims[i+croppingRegionX[0]][j+croppingRegionY[0]][k+croppingRegionZ[0]][l];
					}
		r = rn;
		c = cn;
		s = sn;
		oObs = tempLabels;
		ims  = tempImage;
	}

	private void determineConsensusVoxels(){
		JistLogger.logOutput(JistLogger.INFO, "++ Determining Consensus Voxels ++");
		JistLogger.logFlush();
		cons = new boolean[r][c][s];
		//File it up
		for(boolean[][] rows:cons)
			for(boolean[] row:rows)
				Arrays.fill(row, false);

		numVoxels = 0;
		numConsensus = 0;
		HashMap<String,Integer> map;
		//iterate over all labels
		for(int i=0;i<r;i++){
			for(int j=0;j<c;j++){
				for(int k=0;k<s;k++){
					numVoxels++;
					// Map of observed labels by class
					map = new HashMap<String,Integer>();
					boolean isConsensus = true;
					// Check the labels of all raters
					for(int l=0;l<numRaters;l++){
						String cl = classes[l];
						int obsLab = oObs[i][j][k][l];
						if(obsLab > maxLabel)
							maxLabel = obsLab;
						// Put it into to map if there is no key already
						if(!map.containsKey(cl)){
							map.put(cl, obsLab);
						}
						int lab = map.get(cl);
						if(lab != obsLab){
							isConsensus = false;
							break;
						}
					}
					if(isConsensus){
						numConsensus++;
						cons[i][j][k] = true;
					}
				}
			}
		}
		float p = (float) numConsensus / (float) numVoxels;
		JistLogger.logOutput(JistLogger.INFO, String.format("%f of the voxels are consensus", p));
		numToProcess = numVoxels - numConsensus;
		// TODO: Second level consensus determination
	}

	private void determineCroppingRegion(){
		JistLogger.logOutput(JistLogger.INFO,"Determining Cropping Region");
		croppingRegionX = new int[2];
		croppingRegionY = new int[2];
		croppingRegionZ = new int[2];
		croppingRegionX[1] = r;
		croppingRegionY[1] = c;
		croppingRegionZ[1] = s;

		boolean[] arrX = new boolean[r];
		boolean[] arrY = new boolean[c];
		boolean[] arrZ = new boolean[s];
		Arrays.fill(arrX, false);
		Arrays.fill(arrY, false);
		Arrays.fill(arrZ, false);

		//Iterate over voxels and fill in locations that are true
		for(int i=0;i<r;i++){
			for(int j=0;j<c;j++){
				for(int k=0;k<s;k++){
					for(int l=0;l<numRaters;l++){
						if(oObs[i][j][k][l] != 0){
							arrX[i] = true;
							arrY[j] = true;
							arrZ[k] = true;
						}
					}
				}
			}
		}
		//Find x lower
		for(int i=0;i<r;i++){
			if(arrX[i]){
				croppingRegionX[0] = i;
				break;
			}
		}
		//Find x upper
		for(int i=r-1;i>=0;i--){
			if(arrX[i]){
				croppingRegionX[1] = i;
				break;
			}
		}

		//Find y lower
		for(int i=0;i<c;i++){
			if(arrY[i]){
				croppingRegionY[0] = i;
				break;
			}
		}
		//Find y upper
		for(int i=c-1;i>=0;i--){
			if(arrY[i]){
				croppingRegionY[1] = i;
				break;
			}
		}

		//Find x lower
		for(int i=0;i<s;i++){
			if(arrZ[i]){
				croppingRegionZ[0] = i;
				break;
			}
		}
		//Find x upper
		for(int i=s-1;i>=0;i--){
			if(arrZ[i]){
				croppingRegionZ[1] = i;
				break;
			}
		}

		JistLogger.logOutput(JistLogger.INFO, 
				String.format("The cropping region is [%d %d] [%d %d] [%d %d]",
						croppingRegionX[0],croppingRegionX[1],croppingRegionY[0],
						croppingRegionY[1],croppingRegionZ[0],croppingRegionZ[1]));
	}

	private void loadImages(List<ImageData> imgs){
		ims = new float[r][c][s][numRaters];
		ImageData im;

		for(int i=0;i<imgs.size();i++){
			int origLevel = prefs.getDebugLevel();
			prefs.setDebugLevel(JistLogger.SEVERE);
			im = imgs.get(i);
			float[][][] imTemp = loadImage(im);
			im.dispose();
			prefs.setDebugLevel(origLevel);
			placeImage(imTemp,i);
		}

	}

	private void placeImage(float[][][] mat,int idx){
		for(int i=0;i<r;i++)
			for(int j=0;j<c;j++)
				for(int k=0;k<s;k++)
					ims[i][j][k][idx] = mat[i][j][k];
	}

	private float[][][] loadImage(ImageData im){
		float[][][] img = new float[r][c][s];
		for(int i=0;i<r;i++)
			for(int j=0;j<c;j++)
				for(int k=0;k<s;k++)
					img[i][j][k] = im.getFloat(i, j, k);
		return img;
	}

	private void loadLabels(List<ImageData> imgs){
		ImageData im;
		oObs = new int[r][c][s][imgs.size()];
		for(int i=0;i<imgs.size();i++){
			int origLevel = prefs.getDebugLevel();
			prefs.setDebugLevel(JistLogger.SEVERE);
			im = imgs.get(i);
			int[][][] intTemp = loadLabels(im);
			placeLabels(intTemp,i);
			im.dispose();
			prefs.setDebugLevel(origLevel);
		}

	}

	private void placeLabels(int[][][] mat,int idx){
		for(int i=0;i<r;i++)
			for(int j=0;j<c;j++)
				for(int k=0;k<s;k++)
					oObs[i][j][k][idx] = mat[i][j][k];
	}

	private int[][][] loadLabels(ImageData im){
		int[][][] img = new int[r][c][s];
		for(int i=0;i<r;i++)
			for(int j=0;j<c;j++)
				for(int k=0;k<s;k++)
					img[i][j][k] = im.getInt(i, j, k);
		return img;
	}

	private boolean checkImages(List<ImageData> ims1,List<ImageData> ims2){
		boolean correct = true;
		ImageData im;
		for(int i=0;i<ims1.size();i++){
			im = ims1.get(i);
			if(!checkImage(im))
				correct = false;
			if(useIntensity){
				im = ims2.get(i);
				if(!checkImage(im))
					correct = false;
			}
		}

		return correct;
	}

	private boolean checkImage(ImageData im){
		if(im.getRows() == r && im.getCols() == c && im.getSlices() == s)
			return true;
		else{
			JistLogger.logError(JistLogger.WARNING, 
					String.format("The expected image dimensions are [%d,%d,%d] but an image has dimensions [%d,%d,%d].",
							r,c,s,im.getRows(),im.getCols(),im.getSlices()));
			return false;
		}
	}

	public boolean isConsensus(int x,int y,int z){
		return cons[x][y][z];
	}

	private void printStatusBar(int num,int tot){
		String status = "[";
		for(int i=0;i<num;i++)
			status += "=";
		for(int i=0;i<tot-num;i++)
			status += "+";
		status += "]";
		JistLogger.logOutput(JistLogger.INFO, status);
		JistLogger.logFlush();
	}

	public int getR(){ return r;}
	public int getS(){ return s;}
	public int getC(){ return c;}
	public int getNumRaters(){ return numRaters; }
	public int getMaxLabel(){ return maxLabel; }
	public float[] getProbs(int x,int y,int z,int l){ return vals[x][y][z][l]; }
	public short[] getLabs(int x,int y,int z,int l){ return obs[x][y][z][l]; }
	public int getConsensus(int x,int y, int z){ return oObs[x][y][z][0];}
}
