package edu.vanderbilt.masi.algorithms.labelfusion;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import java.io.*;

public abstract class PerformanceParametersBase extends AbstractCalculation {
	
	protected int num_raters;
	protected final float diagval = 0.8f;
	protected final int PERFORMANCE_PARAMETERS_SINGLE_TYPE = 0;
	protected final int PERFORMANCE_PARAMETERS_VECTORIZED_TYPE = 1;
	protected final static double EXP_CONVERGENCE_THRESHOLD = 1e-6f;
	
	public PerformanceParametersBase (int num_raters_in) {
		super();
		setLabel("PerformanceParametersBase");
		
		num_raters = num_raters_in;
	}
	
	public abstract void initialize();
	public abstract void normalize();
	public abstract void normalize(float [] lp);
	public abstract void reset();
	public abstract File toFile(int j, File outdir);
	public abstract ImageData toImage(String name);
	public abstract float get(int j, int s1, int s2);
	public abstract float get_labelnorm(int j, int s);
	public abstract double get(int j, int s, short [] obslabels, float [] obsvals);
	public abstract double get_log(int j, int s, short [] obslabels, float [] obsvals);
	public abstract PerformanceParametersBase getPerformanceParameters(int i);
	public abstract void copy(PerformanceParametersBase theta2);
	public abstract float get_convergence_factor(PerformanceParametersBase theta2);
	public abstract void add(int j, int s1, int s2, float val);
	public abstract void add(int lvl, int j, int s1, int s2, float val);
	public abstract int get_type();
	
	protected static double get_exponential_labelnorm(double [] vec, int veclength) {
		return(PerformanceParametersBase.get_exponential_labelnorm(vec, veclength, 0.3));
	}
	
	protected static double get_exponential_labelnorm(double [] vec,
													 int veclength,
													 double initial_val) {

		
		double x1, x2, prev_x1, prev_x2;
		double convergence_value = Double.MAX_VALUE;
		double prev_convergence_value = Double.MAX_VALUE;
		double oy1, oy2, y1, y2, val, estval;
		double m, b;
		double min_x, max_x;
		
		if (veclength == 1)
			return(1f);
		
		val = 1;
		x1 = 0.2;
		x2 = 0.4;
		prev_x1 = initial_val - 0.1;
		prev_x2 = initial_val + 0.1;
		min_x = 0;
		max_x = 1;
		
		while (convergence_value > EXP_CONVERGENCE_THRESHOLD) {
			
			// set the y-values
			oy1 = 0; oy2 = 0;
			for (int l = 0; l < veclength; l++) {
				oy1 += Math.pow(vec[l], x1);
				oy2 += Math.pow(vec[l], x2);
			}
			y1 = Math.log(oy1);
			y2 = Math.log(oy2);
			
			// estimate the value
			m = (y2 - y1) / (x2 - x1);
			b = -m*x1 + y1;
			val = -b / m;
			
			// get the error
			estval = 0;
			for (int l = 0; l < veclength; l++)
				estval += Math.pow(vec[l], val);
			convergence_value = Math.abs(estval-1);
			
			// set the appropriate bounds
			if (estval < 1) {
				if (max_x > val)
					max_x = val;
			} else {
				if (min_x < val)
					min_x = val;
			}
			if (oy1 < 1) {
				if (max_x > x1)
					max_x = x1;
			} else {
				if (min_x < x1)
					min_x = x1;
			}
			if (oy2 < 1) {
				if (max_x > x2)
					max_x = x2;
			} else {
				if (min_x < x2)
					min_x = x2;
			}
			
			// adjust the bounds
			if (convergence_value < prev_convergence_value) {
				prev_convergence_value = convergence_value;
			
				// update the x-values
				if (estval < 1) {
					x1 = Math.max(val - convergence_value, min_x);
					x2 = Math.min(val, max_x);
				} else {
					x1 = Math.max(val, min_x);
					x2 = Math.min(val + convergence_value, max_x);
				}
			} else {
				x1 = Math.max(prev_x1 - prev_convergence_value, min_x);
				x2 = Math.min(prev_x2 + prev_convergence_value, max_x);
			}
			
			// set the previous searching values
			prev_x1 = x1;
			prev_x2 = x2;
		}

		return(val);
	}
}
