package edu.vanderbilt.masi.plugins.asl;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.model.file.FileInfoBase;

public class PluginComputeCBF extends ProcessingAlgorithm{
	private static final String cvsversion = "$Revision: 1.7 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "") .replace(" ", "");
	ParamVolume aslLabel;
	ParamVolume aslControl;
	ParamVolume m0;
	ParamOption field;
	ParamOption asltype;
	ParamFloat labelDur;
	ParamFloat postLabelDuration;
	ParamFloat TI;
	ParamFloat TI1;
	ParamFloat sliceTimePCASL;
	ParamFloat sliceTimeQUIPPS;
	
	ParamVolume cbf;
	ParamVolume meanControl;
	ParamVolume meanLabel;
	ParamVolume dm;
	ParamVolume dmVol;
	
	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		// TODO Auto-generated method stub
				inputParams.setName("ComputeCBF");
				inputParams.setLabel("ComputeCBF");
				inputParams.setPackage("MASI");
				inputParams.setCategory("ASL");

				AlgorithmInformation info = getAlgorithmInformation();
				info.setWebsite("http://masi.vuse.vanderbilt.edu");
				info.add(new AlgorithmAuthor("Steve Damon", "stephen.m.damon@vanderbilt.edu", "http://masi.vuse.vanderbilt.edu"));
				info.setVersion(revnum);
				info.setEditable(false);
				info.add(new Citation("Alsop DC et. al., \"Recommended implementation of arterial spin-labeled perfusion MRI for clinical applications: A consensus of the ISMRM perfusion study group and the European consortium for ASL in dementia.\" MRI 2014;DOI: 10.1002/mrm.25197"));	

				info.setStatus(DevelopmentStatus.BETA);
			
				
				//Baseparams
				ParamCollection baseParams = new ParamCollection("Data");
				//Base params: asl, m0, field strength, and computation type
				baseParams.add(aslControl = new ParamVolume("4D control aslData"));
				baseParams.add(aslLabel = new ParamVolume("4D label asl Data "));
				baseParams.add(m0 = new ParamVolume("M0 MAP"));
				//options selections
				List<String> fs = new ArrayList<String>(2);
				fs.add("3.0T");
				fs.add("1.5T");
				List<String> at = new ArrayList<String>(2);
				at.add("PCASL");
				at.add("QUIPPS II PASL");
				baseParams.add(field = new ParamOption("Field Strength",fs));
				baseParams.add(asltype = new ParamOption("aslData Type",at));
				inputParams.add(baseParams);
				
				//Just the Pcasl Params - defaults are for adults
				ParamCollection pcaslParams = new ParamCollection("PCASL Params");
				pcaslParams.add(labelDur = new ParamFloat("Label Duration - time in [ms]",1800));
				pcaslParams.add(postLabelDuration = new ParamFloat("Post Label delay - time in [ms]",1800));
				pcaslParams.add(sliceTimePCASL = new ParamFloat("Slice time - time in [ms]",30));
				sliceTimePCASL.setMandatory(true);
				labelDur.setMandatory(true);
				postLabelDuration.setMandatory(true);
				inputParams.add(pcaslParams);
				
				//Just the PASL Params - defaults for adults
				ParamCollection paslParams = new ParamCollection("PASL Params");
				paslParams.add(TI = new ParamFloat("Inversion Time (TI) - time in [ms]",1800));
				paslParams.add(TI1 = new ParamFloat("Inversion Time 1 (TI1) - time in [ms]",800));
				labelDur.setMandatory(true);
				postLabelDuration.setMandatory(true);
				inputParams.add(paslParams);
					
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		// TODO Auto-generated method stub
		//outputParams.add(cbf= new ParamVolume("Perfusion map [ml/100g/min]"));
		outputParams.add(cbf= new ParamVolume("CBF Volume"));
		outputParams.add(meanControl= new ParamVolume("Mean of non-labeled signal"));
		outputParams.add(meanLabel= new ParamVolume("Mean of the labeled-signal"));
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		
			ImageData controlData =   aslControl.getImageData();
			ImageData labelData =  aslLabel.getImageData();
			ImageData m0Data =  m0.getImageData();
						
			if (controlData.getCols()!=m0Data.getCols()){
				JistLogger.logError(JistLogger.SEVERE, "ERROR: Mismatched dimensions in the column direction");
			}else if (controlData.getRows()!=m0Data.getRows()){
				JistLogger.logError(JistLogger.SEVERE, "ERROR: Mismatched dimensions in the row direction");
			}else if (controlData.getSlices()!=m0Data.getSlices()){
				JistLogger.logError(JistLogger.SEVERE, "ERROR: Mismatched dimensions in the column direction");
			}
			//Set the param values
			float T1b=0;		//T1 of blood;
			float I=(float) 0.9;	//blood-brain partition coefficient
			float a=0;		//labeling efficiency
			
			//T1b
			if (field.getValue().equals("3.0T")){
				JistLogger.logOutput(JistLogger.FINE,"Using 1650ms for T1 of blood");
				T1b=(float) 1650;
			}else if (field.getValue().equals("1.5T")){
				JistLogger.logOutput(JistLogger.FINE,"Using 1350ms for T1 of blood");
				T1b=(float) 1350;
			}
			
			//alpha
			if (asltype.getValue().equals("PCASL")){
				JistLogger.logOutput(JistLogger.FINE,"Using 0.85 for labeling efficiency");
				a=(float) 0.85;
			}else if (asltype.getValue().equals("QUIPPS II PASL")){
				JistLogger.logOutput(JistLogger.FINE,"Using 0.98 for labeling efficiency");
				a=(float) 0.98;
			}
			//Print the calculation params
			JistLogger.logOutput(JistLogger.FINE,"Post Label Delay = "+postLabelDuration.getValue().toString());
			JistLogger.logOutput(JistLogger.FINE,"Label Duration = "+labelDur.getValue().toString());			
			JistLogger.logOutput(JistLogger.FINE,"labeling efficiency = "+a);
			JistLogger.logOutput(JistLogger.FINE,"Lambda = "+I);
			JistLogger.logOutput(JistLogger.FINE,"T1 blood ="+T1b);
			
			//Allocate the memory upfront
			float[][][] cbfVol = new float[controlData.getRows()][controlData.getCols()][controlData.getSlices()];
			float[][][] control = new float[controlData.getRows()][controlData.getCols()][controlData.getSlices()]; 
			float[][][] label = new float[labelData.getRows()][labelData.getCols()][labelData.getSlices()];
			float temp = 0;
			ImageData out = new ImageDataFloat(controlData.getRows(),controlData.getCols(),controlData.getSlices());

			//Mean the control signal
			float meanFact = Float.valueOf(controlData.getComponents());

			for (int i=0;i<controlData.getRows();i++){
				for (int j=0;j<controlData.getCols();j++){
					for (int k=0;k<controlData.getSlices();k++){
						for (int l=0;l<controlData.getComponents();l++){
							temp+=controlData.get(i,j,k,l).floatValue();
						}
						control[i][j][k]=temp/meanFact;
						temp=0;
					}
				}
			}
			
			//Mean the label
			temp = 0;
			meanFact = Float.valueOf(labelData.getComponents());
			for (int i=0;i<labelData.getRows();i++){
				for (int j=0;j<labelData.getCols();j++){
					for (int k=0;k<labelData.getSlices();k++){
						for (int l=1;l<labelData.getComponents();l++){
							temp+=labelData.get(i,j,k,l).floatValue();
						}
						label[i][j][k]=temp/meanFact;
						temp=0;
					}
				}
			}
						
			//PCASL Fit
			if (asltype.getValue().equals("PCASL")){
				//Correct the postlabel delay for slice time
				float[] wz = new float[controlData.getSlices()];
				for (int i=0;i<controlData.getSlices();i++)
					wz[i] = postLabelDuration.getFloat()+sliceTimePCASL.getFloat()/1000*(float) i;
				//make CBF volume - PCASL
				for (int i=0;i<controlData.getRows();i++){
					for (int j=0;j<controlData.getCols();j++){
						for (int k=0;k<controlData.getSlices();k++){
							
							float numerator =  (float) (6000000*I*(control[i][j][k] -label[i][j][k])*Math.exp(wz[k]/T1b));
							float denominator =  (float) (2*a*T1b*m0Data.getFloat(i,j,k)*(1-Math.exp(-1*labelDur.getFloat()/T1b)));
							float cbf = numerator/denominator;
							
							if (Float.isNaN(cbf) || Float.isInfinite(cbf)){
								cbf=0;
							}
							out.set(i,j,k,cbf) ;
							temp=0;
						}
						
					}
				}
			}
			
			//PASL Fit
			else if (asltype.getValue().equals("QUIPPS II PASL")){
				//make CBF volume - QUIPPS II PASL
				for (int i=0;i<controlData.getRows();i++){
					for (int j=0;j<controlData.getCols();j++){
						for (int k=0;k<controlData.getSlices();k++){
							float numerator = (float) (6000000*I*(control[i][j][k] -label[i][j][k])*Math.exp(TI.getDouble()/T1b));
							float denominator = (float) (2*a*TI1.getDouble()*m0Data.getDouble(i,j,k));
							float cbf = numerator/denominator;
							if (Float.isNaN(cbf) || Float.isInfinite(cbf)){
								cbf=0;
							}
							out.set(i,j,k,cbf) ;
							temp=0;
						}
					}
				}
			}
			out.setHeader(m0Data.getHeader());
			out.setName("Perfusion");
			cbf.setValue(out);
		
			ImageDataFloat t = (new ImageDataFloat(control));
			t.setHeader(m0Data.getHeader());
			t.setName("MeanOfControlVols");
			meanControl.setValue(t);
			
			ImageDataFloat t2 = (new ImageDataFloat(label));
			t2.setHeader(m0Data.getHeader());
			t2.setName("MeanOfLabelVols");
			meanLabel.setValue(t2);
	}
}
