package edu.vanderbilt.masi.plugins.classification;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.vanderbilt.masi.algorithms.clasisfication.DistanceFeatureCalculator;
import edu.vanderbilt.masi.algorithms.clasisfication.FeatureCalculationRunner;
import edu.vanderbilt.masi.algorithms.clasisfication.IntensityFeatureCalculator;
import edu.vanderbilt.masi.algorithms.clasisfication.LabelFeatureCalculator;
import edu.jhu.ece.iacl.jist.utility.FileUtil;
import edu.jhu.ece.iacl.jist.utility.JistLogger;

import java.io.*;

public class FeatureCalculatorPlugin extends ProcessingAlgorithm {

	// input parameter
	public ParamVolumeCollection est_vol;
	public ParamVolumeCollection mask;
	public ParamVolumeCollection labels;
	public ParamInteger intensity_radius;
	public ParamInteger label_radius;	
	public ParamBoolean gzip_output;
	public ParamBoolean calculate_distance;

	// output parameters

	public ParamFileCollection feature_file;

	/****************************************************
	 * CVS Version Control
	 ****************************************************/
	private static final String cvsversion = "$Revision: 1.3 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Runs Feature Calculation on Images.";
	private static final String longDescription = "";

	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createInputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.BETA);

		inputParams.setPackage("MASI");
		inputParams.setCategory("Feature Calculation");
		inputParams.setLabel("Feature Calculation");
		inputParams.setName("Feature_Calculation");

		// set the manual observations
		inputParams.add(est_vol = new ParamVolumeCollection("Intensity Image"));
		est_vol.setLoadAndSaveOnValidate(false);
		est_vol.setMandatory(true);

		inputParams.add(labels = new ParamVolumeCollection("Labeled Image"));
		labels.setLoadAndSaveOnValidate(false);
		labels.setMandatory(true);

		inputParams.add(mask = new ParamVolumeCollection("Mask"));
		mask.setLoadAndSaveOnValidate(false);
		mask.setMandatory(true);

		inputParams.add(intensity_radius = new ParamInteger("Radius for Intensity Features"));
		intensity_radius.setLoadAndSaveOnValidate(false);
		intensity_radius.setMandatory(false);
		intensity_radius.setValue(-1);

		inputParams.add(label_radius = new ParamInteger("Radius for Label Features"));
		label_radius.setLoadAndSaveOnValidate(false);
		label_radius.setMandatory(false);
		label_radius.setValue(-1);

		inputParams.add(calculate_distance = new ParamBoolean("Calculate Distance to Label Centers?"));
		calculate_distance.setMandatory(false);
		calculate_distance.setValue(false);

		inputParams.add(gzip_output = new ParamBoolean("Gzip Output File"));
		gzip_output.setMandatory(false);
		gzip_output.setValue(true);

	}

	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createOutputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(feature_file = new ParamFileCollection("Features"));
	}

	protected void execute(CalculationMonitor monitor) throws AlgorithmRuntimeException {
		try {
			ExecuteWrapper wrapper=new ExecuteWrapper();
			monitor.observe(wrapper);
			wrapper.execute(this);

		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	protected class ExecuteWrapper extends AbstractCalculation {
		public void execute(ProcessingAlgorithm alg) throws FileNotFoundException{
			for(int l=0;l<est_vol.size();l++){
				JistLogger.logOutput(JistLogger.INFO, "Starting Image "+l);
				ImageData im = est_vol.getParamVolume(l).getImageData(true);
				ImageData m = labels.getParamVolume(l).getImageData(true);
				ImageData ma = mask.getParamVolume(l).getImageData();
				float[][][] float_im = new float[im.getRows()][im.getCols()][im.getSlices()];
				int[][][] label_im = new int[im.getRows()][im.getCols()][im.getSlices()];
				boolean[][][] bool_mask = new boolean[im.getRows()][im.getCols()][im.getSlices()];
				for(int i = 0; i<m.getRows();i++){
					for(int j = 0;j<m.getCols();j++){
						for(int k = 0;k<m.getSlices();k++){
							label_im[i][j][k] = m.getInt(i,j,k);
							float_im[i][j][k] = im.getFloat(i, j,k);
							bool_mask[i][j][k] = ma.getBoolean(i, j, k);
						}
					}
				}
				JistLogger.logOutput(JistLogger.INFO, "Deallocating Memory");
				m.dispose();
				im.dispose();
				ma.dispose();
				FeatureCalculationRunner FR = new FeatureCalculationRunner();
				if(intensity_radius.getInt()>0){
					FR.addCalculator(new IntensityFeatureCalculator(intensity_radius.getInt()));
					JistLogger.logOutput(JistLogger.INFO, "Added Intensity Radius Calculator");
				}
				if(label_radius.getInt()>0){
					FR.addCalculator(new LabelFeatureCalculator(label_im,label_radius.getInt()));
					JistLogger.logOutput(JistLogger.INFO, "Added Label Count Radius Calculator");
				}
				if(calculate_distance.getValue()){
					FR.addCalculator(new DistanceFeatureCalculator(label_im));
					JistLogger.logOutput(JistLogger.INFO, "Added Distance Calculator");
				}

				FR.calculateFeatures(float_im, label_im, bool_mask);

				JistLogger.logOutput(JistLogger.INFO, "Writing Output");

				File outdir = new File(
						alg.getOutputDirectory() +
						File.separator +
						FileUtil.forceSafeFilename(alg.getAlgorithmName()) +
						File.separator);
				outdir.mkdirs();
				File f;
				if(gzip_output.getValue()) f = new File(outdir, String.format("Feature_File%s.json.gz", l));
				else f = new File(outdir, String.format("Feature_File%s.json", est_vol.getName()));
				FR.writeFile(f);
				feature_file.add(f);

			}
		}
	}

}
