package edu.vanderbilt.masi.plugins.labelfusion;

import java.io.File;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import java.io.FileNotFoundException;

import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFileCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFloat;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.vanderbilt.masi.algorithms.labelfusion.*;
import edu.jhu.ece.iacl.jist.utility.FileUtil;

public abstract class AbstractStatisticalFusionPlugin extends ProcessingAlgorithm {
	
	// Main Parameters parameters
	public transient ParamVolume targetim;
	public transient ParamVolumeCollection obsvols;
	public transient ParamVolumeCollection imsvols;
	public transient ParamFile hierarchy_file;
	public transient ParamOption output_W;
	public transient ParamOption output_theta;
	public transient ParamFloat consensus_thresh;
	public transient ParamOption priortype;
	public transient ParamFloat epsilon;
	public transient ParamInteger maxiter;
	public transient ParamOption probtype;
	
	// Spatially-Varying Performance Level Parameter parameters
	public transient ParamInteger hws_dim1;
	public transient ParamInteger hws_dim2;
	public transient ParamInteger hws_dim3;
	public transient ParamInteger hws_dim4;
	public transient ParamFloat bias;
	public transient ParamBoolean use_scalar_perf;
	
	// weighting parameters
	public transient ParamOption weight_type;
	public transient ParamInteger sv_dim1;
	public transient ParamInteger sv_dim2;
	public transient ParamInteger sv_dim3;
	public transient ParamInteger sv_dim4;
	public transient ParamInteger pv_dim1;
	public transient ParamInteger pv_dim2;
	public transient ParamInteger pv_dim3;
	public transient ParamInteger pv_dim4;
	public transient ParamFloat spatial_sd_dim1;
	public transient ParamFloat spatial_sd_dim2;
	public transient ParamFloat spatial_sd_dim3;
	public transient ParamFloat spatial_sd_dim4;
	public transient ParamFloat intensity_sd;
	public transient ParamFloat global_sel_thresh;
	public transient ParamFloat local_sel_thresh;
	public transient ParamBoolean use_intensity_normalization;
	
	// patch selection parameters
	public transient ParamOption selection_type;
	public transient ParamFloat selection_thresh;
	public transient ParamInteger num_keep;
	
	// output parameters
	public transient ParamVolume labelvol;
	public transient ParamFileCollection theta_text;
	public transient ParamVolume theta_image;
	public transient ParamVolumeCollection labelprob_multiple;
	public transient ParamVolume labelprob_single;
	public transient int priortype_index;
	
	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createInputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected abstract void createInputParameters(ParamCollection inputParams);	
	/*
	 * (non-Javadoc)
	 *
	 * @see edu.jhu.ece.iacl.pipeline.ProcessingAlgorithm#createOutputParameters(edu.jhu.ece.iacl.pipeline.parameter.ParamCollection)
	 */
	protected abstract void createOutputParameters(ParamCollection outputParams);
	
	protected ParamCollection get_main_parameters() {

		ParamCollection mainParams = new ParamCollection("Main");
		
		// set the input rater observations
		mainParams.add(obsvols=new ParamVolumeCollection("Atlas Label Volumes"));
		obsvols.setLoadAndSaveOnValidate(false);
		obsvols.setMandatory(true);
		
		// set the input rater observations
		mainParams.add(targetim=new ParamVolume("Target Image Volume"));
		targetim.setLoadAndSaveOnValidate(false);
		targetim.setMandatory(true);
				
		// set the input rater observations
		mainParams.add(imsvols=new ParamVolumeCollection("Atlas Image Volumes"));
		imsvols.setLoadAndSaveOnValidate(false);
		imsvols.setMandatory(true);
		
		// set the input rater observations
		mainParams.add(hierarchy_file=new ParamFile("Hierarchy txt file"));
		hierarchy_file.setMandatory(false);
		
		// set the probability fusion values
		mainParams.add(probtype = new ParamOption("Probability Dimension ('None' if not probability volumes)", new String[] { "None", "3rd", "4th"}));
		probtype.setValue(0);
		probtype.setMandatory(false);
		
		// set the options for writing the label probabilities
		mainParams.add(output_W = new ParamOption("Output Label Probabilities?", new String[] { "None", "Multiple", "Single"}));
		output_W.setValue(0);
		output_W.setMandatory(false);
		
		// set the options for writing the label probabilities
		mainParams.add(output_theta = new ParamOption("Output Performance Parameters?", new String[] { "None", "Text", "Image"}));
		output_theta.setValue(0);
		output_theta.setMandatory(false);
		
		// set the ignore consensus voxels type
		mainParams.add(consensus_thresh = new ParamFloat("Consensus Threshold (Threshold > 1: No voxels ignored)", 0f, Float.MAX_VALUE));
		consensus_thresh.setValue(0.99);
		consensus_thresh.setMandatory(false);
		
		// set the prior type
		mainParams.add(priortype = new ParamOption("Prior Distribution Type", new String[] { "Global", "Voxelwise", "Adaptive", "Weighted-Voxelwise"}));
		priortype.setValue(1);
		priortype.setMandatory(false);
		
		// set the convergence threshold
		mainParams.add(epsilon = new ParamFloat("Convergence Threshold"));
		epsilon.setValue(1e-4);
		epsilon.setMandatory(false);
		
		// set the maximum number of iterations
		mainParams.add(maxiter = new ParamInteger("Maximum Number of Iterations"));
		maxiter.setValue(100);
		maxiter.setMandatory(false);

		return(mainParams);
	}
	
	protected ParamCollection get_main_parameters_no_intensity() {
		ParamCollection mainParams = new ParamCollection("Main");
		
		// set the input rater observations
		mainParams.add(obsvols=new ParamVolumeCollection("Rater Volumes"));
		obsvols.setLoadAndSaveOnValidate(false);
		obsvols.setMandatory(true);
		
		mainParams.add(hierarchy_file=new ParamFile("Hierarchy txt file"));
		hierarchy_file.setMandatory(false);
		
		// set the probability fusion values
		mainParams.add(probtype = new ParamOption("Probability Dimension ('None' if not probability volumes)", new String[] { "None", "3rd", "4th"}));
		probtype.setValue(0);
		probtype.setMandatory(false);
				
		// set the options for writing the label probabilities
		mainParams.add(output_W = new ParamOption("Output Label Probabilities?", new String[] { "None", "Multiple", "Single"}));
		output_W.setValue(0);
		output_W.setMandatory(false);
		
		// set the options for writing the label probabilities
		mainParams.add(output_theta = new ParamOption("Output Performance Parameters?", new String[] { "None", "Text", "Image"}));
		output_theta.setValue(0);
		output_theta.setMandatory(false);
		
		// set the ignore consensus voxels type
		mainParams.add(consensus_thresh = new ParamFloat("Consensus Threshold (Threshold > 1: No voxels ignored)", 0f, Float.MAX_VALUE));
		consensus_thresh.setValue(0.99);
		consensus_thresh.setMandatory(false);
				
		// set the prior type
		mainParams.add(priortype = new ParamOption("Prior Distribution Type", new String[] { "Global", "Voxelwise", "Adaptive"}));
		priortype.setValue(1);
		priortype.setMandatory(false);
		
		// set the convergence threshold
		mainParams.add(epsilon = new ParamFloat("Convergence Threshold"));
		epsilon.setValue(1e-4);
		epsilon.setMandatory(false);
		
		// set the maximum number of iterations
		mainParams.add(maxiter = new ParamInteger("Maximum Number of Iterations"));
		maxiter.setValue(100);
		maxiter.setMandatory(false);
		
		return(mainParams);
	}
	
	protected ParamCollection get_spatial_parameters() {
		ParamCollection spatialParams = new ParamCollection("Spatial Parameters");
		
		// Set the window radii for each dimension
		spatialParams.add(hws_dim1 = new ParamInteger("Window Radius Dimension 1 (>0: Image Units, <0: Voxels)"));
		hws_dim1.setValue(5);
		hws_dim1.setMandatory(false);
		spatialParams.add(hws_dim2 = new ParamInteger("Window Radius Dimension 2 (>0: Image Units, <0: Voxels)"));
		hws_dim2.setValue(5);
		hws_dim2.setMandatory(false);
		spatialParams.add(hws_dim3 = new ParamInteger("Window Radius Dimension 3 (>0: Image Units, <0: Voxels)"));
		hws_dim3.setValue(5);
		hws_dim3.setMandatory(false);
		spatialParams.add(hws_dim4 = new ParamInteger("Window Radius Dimension 4 (>0: Image Units, <0: Voxels)"));
		hws_dim4.setValue(0);
		hws_dim4.setMandatory(false);
		
		// set the bias amount
		spatialParams.add(bias = new ParamFloat("Bias Amount (Range: [0 1], {0,1} = Fully {local,global})", 0f, 1f));
		bias.setValue(0);
		bias.setMandatory(false);
		
		// set the local performance type
		spatialParams.add(use_scalar_perf = new ParamBoolean("Augment Scalar Performance Model"));
		use_scalar_perf.setValue(false);
		use_scalar_perf.setMandatory(false);
		
		return(spatialParams);
	}
	
	protected ParamCollection get_weighting_parameters() {
		ParamCollection weightParams = new ParamCollection("Correspondence Parameters");
		
		// set the prior type
		weightParams.add(weight_type = new ParamOption("Weighting Type", new String[] { "LNCC", "MSD", "Mixed"}));
		weight_type.setValue(1);
		weight_type.setMandatory(false);
		
		// Set the search volume for each dimension
		weightParams.add(sv_dim1 = new ParamInteger("Search Volume Dimension 1 (>0: Image Units, <0: Voxels)"));
		sv_dim1.setValue(3);
		sv_dim1.setMandatory(false);
		weightParams.add(sv_dim2 = new ParamInteger("Search Volume Dimension 2 (>0: Image Units, <0: Voxels)"));
		sv_dim2.setValue(3);
		sv_dim2.setMandatory(false);
		weightParams.add(sv_dim3 = new ParamInteger("Search Volume Dimension 3 (>0: Image Units, <0: Voxels)"));
		sv_dim3.setValue(3);
		sv_dim3.setMandatory(false);
		weightParams.add(sv_dim4 = new ParamInteger("Search Volume Dimension 4 (>0: Image Units, <0: Voxels)"));
		sv_dim4.setValue(0);
		sv_dim4.setMandatory(false);
		
		// Set the patch volume for each dimension
		weightParams.add(pv_dim1 = new ParamInteger("Patch Volume Dimension 1 (>0: Image Units, <0: Voxels)"));
		pv_dim1.setValue(2);
		pv_dim1.setMandatory(false);
		weightParams.add(pv_dim2 = new ParamInteger("Patch Volume Dimension 2 (>0: Image Units, <0: Voxels)"));
		pv_dim2.setValue(2);
		pv_dim2.setMandatory(false);
		weightParams.add(pv_dim3 = new ParamInteger("Patch Volume Dimension 3 (>0: Image Units, <0: Voxels)"));
		pv_dim3.setValue(2);
		pv_dim3.setMandatory(false);
		weightParams.add(pv_dim4 = new ParamInteger("Patch Volume Dimension 4 (>0: Image Units, <0: Voxels)"));
		pv_dim4.setValue(0);
		pv_dim4.setMandatory(false);
		
		// Set the search volume for each dimension
		weightParams.add(spatial_sd_dim1 = new ParamFloat("Search Volume Standard Deviation Dimension 1 (>0: Image Units, <0: Voxels)"));
		spatial_sd_dim1.setValue(1.5);
		spatial_sd_dim1.setMandatory(false);
		weightParams.add(spatial_sd_dim2 = new ParamFloat("Search Volume Standard Deviation Dimension 2 (>0: Image Units, <0: Voxels)"));
		spatial_sd_dim2.setValue(1.5);
		spatial_sd_dim2.setMandatory(false);
		weightParams.add(spatial_sd_dim3 = new ParamFloat("Search Volume Standard Deviation Dimension 3 (>0: Image Units, <0: Voxels)"));
		spatial_sd_dim3.setValue(1.5);
		spatial_sd_dim3.setMandatory(false);
		weightParams.add(spatial_sd_dim4 = new ParamFloat("Search Volume Standard Deviation Dimension 4 (>0: Image Units, <0: Voxels)"));
		spatial_sd_dim4.setValue(1.5);
		spatial_sd_dim4.setMandatory(false);
		
		// set the difference metric standard deviation
		weightParams.add(intensity_sd = new ParamFloat("Difference Metric Standard Deviation"));
		intensity_sd.setValue(0.25);
		intensity_sd.setMandatory(false);
		
		weightParams.add(global_sel_thresh = new ParamFloat("Global Selection Threshold (Range: [0 1], 0 = No Selection)", 0, 1));
		global_sel_thresh.setValue(0);
		global_sel_thresh.setMandatory(false);
		
		weightParams.add(local_sel_thresh = new ParamFloat("Local Selection Threshold (Range: [0 1], 0 = No Selection)", 0, 1));
		local_sel_thresh.setValue(0.1);
		local_sel_thresh.setMandatory(false);
		
		// set the boolean ignore consensus voxels
		weightParams.add(use_intensity_normalization = new ParamBoolean("Use Intensity Normalization"));
		use_intensity_normalization.setValue(true);
		use_intensity_normalization.setMandatory(false);
		
		// set the Patch selection options
		ParamCollection selParams = new ParamCollection("Patch Selection");
		
		// set the selection type
		selParams.add(selection_type = new ParamOption("Selection Type", new String[] { "SSIM", "Jaccard", "None"}));
		selection_type.setValue(1);
		selection_type.setMandatory(false);
		
		selParams.add(selection_thresh = new ParamFloat("Selection Type Threshold", 0, 1));
		selection_thresh.setValue(0.05);
		selection_thresh.setMandatory(false);
		
		selParams.add(num_keep = new ParamInteger("Number of Patches to Keep (-1 = all patches)"));
		num_keep.setValue(150);
		num_keep.setMandatory(false);
		
		ParamCollection fullWeightParams = new ParamCollection("Weighting Parameters");
		fullWeightParams.add(weightParams);
		fullWeightParams.add(selParams);
		
		return(fullWeightParams);
	}
	
	protected void set_output_parameters() {
		// handle the label estimate output
		outputParams.add(labelvol = new ParamVolume("Label Volume"));
		
		// handle the performance level parameters
		outputParams.add(theta_text = new ParamFileCollection("Performance Level Parameters File Collection"));
		theta_text.setLoadAndSaveOnValidate(false);
		theta_text.setMandatory(false);
		
		outputParams.add(theta_image = new ParamVolume("Performance Level Parameters Volume"));
		theta_image.setLoadAndSaveOnValidate(false);
		theta_image.setMandatory(false);
		
		// handle the label probabilities output
		outputParams.add(labelprob_multiple = new ParamVolumeCollection("Label Probabilities Collection"));
		labelprob_multiple.setLoadAndSaveOnValidate(false);
		labelprob_multiple.setMandatory(false);
		
		outputParams.add(labelprob_single = new ParamVolume("Label Probabilities Volume"));
		labelprob_single.setLoadAndSaveOnValidate(false);
		labelprob_single.setMandatory(false);
	}
	
	protected class AbstractExecuteWrapper extends AbstractCalculation {
		public void execute(ProcessingAlgorithm alg) throws FileNotFoundException{

			// get the observation object
			ObservationBase obs = get_observation_type();
			
			// get the statistical fusion object
			StatisticalFusionBase sf = get_statistical_fusion_type(obs, alg);
			
			// run the statistical fusion algorithm and set the output estimate
			labelvol.setValue(sf.run());
			
			// write the performance parameters if desired
			write_performance_parameters(obs, sf, alg);
							
			// write the label probabilities if desired
			write_label_probabilities(obs, sf, alg);
		}
		
		private ObservationBase get_observation_type() {
			
			// determine the observation type
			ObservationBase obs;
			
			boolean usePartial = true;
			
			JistLogger.logOutput(JistLogger.INFO, "*** Determining appropriate type for the observation object. ***");
			
			if (targetim == null || imsvols == null)
				usePartial = false;
			
			// check to see if the target image was specified
			if (usePartial && targetim.getValue() == null) {
				usePartial = false;
				JistLogger.logOutput(JistLogger.INFO, "No target image specified");
			}

			// check to see if they specified the right number of atlas images
			if (usePartial && obsvols.getParamVolumeList().size() != imsvols.getParamVolumeList().size()) {
				usePartial = false;
				JistLogger.logOutput(JistLogger.INFO, "Number of atlas images does not match number of atlas labels");
			}
			
			// set the prior type
			priortype_index = priortype.getIndex();
						
			if (usePartial) {
				
				JistLogger.logOutput(JistLogger.INFO, "-> Using ObservationVolumePartial.");
			
				// convert everything to voxel units
				int [] sv = new int [4];
				sv[0] = sv_dim1.getInt();
				sv[1] = sv_dim2.getInt();
				sv[2] = sv_dim3.getInt();
				sv[3] = sv_dim4.getInt();
				
				int [] pv = new int [4];
				pv[0] = pv_dim1.getInt();
				pv[1] = pv_dim2.getInt();
				pv[2] = pv_dim3.getInt();
				pv[3] = pv_dim4.getInt();
				
				float [] sp_stdevs = new float [4];
				sp_stdevs[0] = spatial_sd_dim1.getFloat();
				sp_stdevs[1] = spatial_sd_dim2.getFloat();
				sp_stdevs[2] = spatial_sd_dim3.getFloat();
				sp_stdevs[3] = spatial_sd_dim4.getFloat();
				
				// Create the observation structures
				obs = new ObservationVolumePartial(targetim, obsvols, imsvols,
										           weight_type.getIndex(),
										           sv, pv, sp_stdevs,
										           intensity_sd.getFloat(),
										           selection_type.getIndex(),
										           selection_thresh.getFloat(),
										           num_keep.getInt(),
										           consensus_thresh.getFloat(),
										           probtype.getIndex(),
										           use_intensity_normalization.getValue());
				
				
				// if we're using atlas selection, construct that matrix here
				float global_selection_threshold = global_sel_thresh.getFloat();
				float local_selection_threshold = local_sel_thresh.getFloat();
				
				if (global_selection_threshold > 1)
					global_selection_threshold = 0;
				if (local_selection_threshold > 1)
					local_selection_threshold = 0;
				
				if (global_selection_threshold > 0 || local_selection_threshold > 0)
					obs.create_atlas_selection_matrix(global_selection_threshold,
													  local_selection_threshold);
				
				// if we're using a weighted voxelwise prior, we have to create it now
				if (priortype_index == StatisticalFusionBase.PRIORTYPE_WEIGHTED_VOXELWISE)
					obs.create_weighted_prior();
				
				// normalize the observations
				obs.normalize_all();
				
				// update the final consensus estimate
				obs.update_consensus();
			} else if (probtype.getIndex()==0) {
				
				JistLogger.logOutput(JistLogger.INFO, "-> Using ObservationVolume.");
				
				// make sure we have an appropriate prior type.
				if (priortype_index == StatisticalFusionBase.PRIORTYPE_WEIGHTED_VOXELWISE) {
					JistLogger.logOutput(JistLogger.INFO, "You selected Weighted-Voxelwise prior with no intensity information -- using Voxelwise.");
					priortype_index = StatisticalFusionBase.PRIORTYPE_VOXELWISE;
				}
				
				// set the ObservationVolume
				obs = new ObservationVolume(obsvols, consensus_thresh.getFloat());
			} else {
				JistLogger.logOutput(JistLogger.INFO, "-> Using ObservationVolumeProbability.");
				
				// make sure we have an appropriate prior type.
				if (priortype_index == StatisticalFusionBase.PRIORTYPE_WEIGHTED_VOXELWISE) {
					JistLogger.logOutput(JistLogger.INFO, "You selected Weighted-Voxelwise prior with no intensity information -- using Voxelwise.");
					priortype_index = StatisticalFusionBase.PRIORTYPE_VOXELWISE;
				}
				
				// set the ObservationVolumeProbability
				obs = new ObservationVolumeProbability(obsvols, consensus_thresh.getFloat(), probtype.getIndex());
			}
			
			JistLogger.logOutput(JistLogger.INFO, "");
			
			return(obs);
		}
		
		protected StatisticalFusionBase get_statistical_fusion_type(ObservationBase obs, ProcessingAlgorithm alg) {

			StatisticalFusionBase sf;
			String algname = alg.getAlgorithmName();
			
			JistLogger.logOutput(JistLogger.INFO, "*** Determining appropriate type for the statistical fusion object. ***");
			
			boolean useHierarchy = hierarchy_file != null && hierarchy_file.getValue() != null;
			boolean useLocalPerformance = bias != null && (bias.getFloat() != 1f || use_scalar_perf.getValue());
			boolean useScalarTheta = use_scalar_perf != null && use_scalar_perf.getValue();
			
 			// Set the half-window sizes in each dimension in terms of voxels if we're using local performance
			int [] hws = null;
			if (useLocalPerformance) {
				hws = new int [4];
				float [] tempdimres = obs.get_header().getDimResolutions();
				float [] dimres = new float [4];
				int lengthdimres = tempdimres.length;
				for (int i = 0; i < 4; i++)
					dimres[i] = (i < lengthdimres) ? tempdimres[i] : 1;
				
				// set the initial values
				hws[0] = hws_dim1.getInt();
				hws[1] = hws_dim2.getInt();
				hws[2] = hws_dim3.getInt();
				hws[3] = hws_dim4.getInt();
				
				// set the half-window size in mm / voxels
				hws[0] = (hws[0] >= 0) ? Math.round(((float)hws[0]) / dimres[0]) : -hws[0];
				hws[1] = (hws[1] >= 0) ? Math.round(((float)hws[1]) / dimres[1]) : -hws[1];
				hws[2] = (hws[2] >= 0) ? Math.round(((float)hws[2]) / dimres[2]) : -hws[2];
				hws[3] = (hws[3] >= 0) ? Math.round(((float)hws[3]) / dimres[3]) : -hws[3];
				
				// take care of any issues regarding image size
				hws[0] = Math.min(hws[0], (obs.dimx()-1)/2);
				hws[1] = Math.min(hws[1], (obs.dimy()-1)/2);
				hws[2] = Math.min(hws[2], (obs.dimz()-1)/2);
				hws[3] = Math.min(hws[3], (obs.dimv()-1)/2);
			}
			
			if (useHierarchy)
				if (useLocalPerformance)
					if (useScalarTheta)
						sf = new HierarchicalSpatialSTAPLESimple(obs,
   					                                             hws,
   					                                             epsilon.getFloat(),
   					                                             bias.getFloat(),
   					                                             maxiter.getInt(),
   					                                             priortype_index,
   					                                             hierarchy_file.getValue(),
   					                                             String.format("%s_Estimate", algname));
					else
						sf = new HierarchicalSpatialSTAPLE(obs,
         					                               hws,
         					                               epsilon.getFloat(),
         					                               bias.getFloat(),
         					                               maxiter.getInt(),
         					                               priortype_index,
         					                               hierarchy_file.getValue(),
         					                               String.format("%s_Estimate", algname));
				else
					sf = new HierarchicalSTAPLE(obs,
												epsilon.getFloat(),
												maxiter.getInt(),
												priortype_index,
												hierarchy_file.getValue(),
												String.format("%s_Estimate", algname));
			else
				if (useLocalPerformance)
					if(useScalarTheta)
						sf = new SpatialSTAPLESimple(obs,
                                                     hws,
                                                     epsilon.getFloat(),
                                                     bias.getFloat(),
                                                     maxiter.getInt(),
                                                     priortype_index,
                                                     String.format("%s_Estimate", algname));
					else
						sf = new SpatialSTAPLE(obs,
						                       hws,
						                       epsilon.getFloat(),
						                       bias.getFloat(),
						                       maxiter.getInt(),
						                       priortype_index,
						                       String.format("%s_Estimate", algname));
				else
					sf = new STAPLE(obs,
							        epsilon.getFloat(),
							        maxiter.getInt(),
							        priortype_index,
							        String.format("%s_Estimate", algname));
						
			return(sf);
		}
	
		private void write_performance_parameters(ObservationBase obs,
												  StatisticalFusionBase sf,
												  ProcessingAlgorithm alg) {
			if (output_theta.getIndex() == 1) {
				
				File outdir = new File(
						alg.getOutputDirectory() +
						File.separator +
						FileUtil.forceSafeFilename(alg.getAlgorithmName()) +
						File.separator +
						"PerformanceParameters");
				outdir.mkdirs();
				
				PerformanceParametersBase esttheta = sf.get_theta();
				
				JistLogger.logOutput(JistLogger.INFO, "Writing Performance Level Parameters File Collection");
				for (int j = 0; j < obs.num_raters(); j++) {
					// save the performance level parameters
					File f = esttheta.toFile(j, outdir);
					theta_text.add(f);
					theta_text.writeAndFreeNow(alg);
					f = null;
				}
			} else if (output_theta.getIndex() == 2) {
				File outdir = new File(
						alg.getOutputDirectory() +
						File.separator +
						FileUtil.forceSafeFilename(alg.getAlgorithmName()));
				outdir.mkdirs();
				
				PerformanceParametersBase esttheta = sf.get_theta();
				JistLogger.logOutput(JistLogger.INFO, "Writing Performance Level Parameters Volume");
				
				String name = String.format("%s_Performance_Parameters", alg.getAlgorithmName());
				ImageData f = esttheta.toImage(name);
				theta_image.setValue(f);
				theta_image.writeAndFreeNow(outdir);
				f.dispose();
				f = null;
			}
		}
	
		private void write_label_probabilities(ObservationBase obs,
											   StatisticalFusionBase sf,
											   ProcessingAlgorithm alg) {
			if (output_W.getIndex() == 1) {
				
				File outdir = new File(
						alg.getOutputDirectory() +
						File.separator +
						FileUtil.forceSafeFilename(alg.getAlgorithmName()) +
						File.separator +
						"LabelProbabilities");
				outdir.mkdirs();
				
				JistLogger.logOutput(JistLogger.INFO, "Writing Label Probabilities");
				for (short l = 0; l < obs.num_labels(); l++) {
					String name = String.format("%s_Label_Probability_%04d", alg.getAlgorithmName(), obs.get_label_remap(l));
					ImageData f = sf.get_label_prob(l, name);
					f.setHeader(obs.get_header());
					labelprob_multiple.add(f);
					labelprob_multiple.writeAndFreeNow(outdir);
					f.dispose();
					f = null;
				}
			} else if (output_W.getIndex() == 2) {
				
				File outdir = new File(
						alg.getOutputDirectory() +
						File.separator +
						FileUtil.forceSafeFilename(alg.getAlgorithmName()));
				outdir.mkdirs();
				
				JistLogger.logOutput(JistLogger.INFO, "Writing Label Probabilities");
				
				String name = String.format("%s_Label_Probabilities", alg.getAlgorithmName());
				ImageData f = sf.get_all_label_probs(name);
				f.setHeader(obs.get_header());
				labelprob_single.setValue(f);
				labelprob_single.writeAndFreeNow(outdir);
				f.dispose();
				f = null;
			}
		}
	}
}
