package edu.vanderbilt.masi.plugins.labelfusion;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolumeCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamString;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.algorithms.labelfusion.AbstractMultiSetVoting;
import edu.vanderbilt.masi.algorithms.labelfusion.MultiSetLocallyWeightedVote;
import edu.vanderbilt.masi.algorithms.labelfusion.MultiSetMajorityVote;

public class PluginMultiSetVoting extends ProcessingAlgorithm {
	
	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");
	private static final String shortDescription = "Runs the Multi-Set Votings.";
	private static final String longDescription = "Runs Multi-Set voting based algorithms. If image volumes are present locally weighted is used. If not, Majority Vote is used.";
	
	// Inputs
	public ParamVolumeCollection obsVals;
	public ParamFile raterMap;
	public ParamFile initialTheta;
	public ParamString targetClass;
	
	public ParamVolume targetImage;
	public ParamVolumeCollection registeredImages;

	// Outputs
	public ParamVolume segmentation;
	
	@Override
	protected void createInputParameters(ParamCollection inputParams) {
		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("https://masi.vuse.vanderbilt.edu/");
		info.setAffiliation("MASI - Vanderbilt");
		info.add(new AlgorithmAuthor("Andrew Plassard","andrew.j.plassard@vanderbilt.edu","https://masi.vuse.vanderbilt.edu/index.php/MASI:Andrew_Plassard"));
		info.setDescription(shortDescription);
		info.setLongDescription(shortDescription + longDescription);
		info.setVersion(revnum);
		info.setEditable(true);
		info.setStatus(DevelopmentStatus.BETA);
		
		inputParams.setPackage("MASI");
		inputParams.setCategory("Label Fusion");
		inputParams.setName("MS-Voting");
		inputParams.setLabel("Multi-Set Voting");
		
		inputParams.add(obsVals=new ParamVolumeCollection("Registered Labels"));
		obsVals.setMandatory(true);
		inputParams.add(raterMap=new ParamFile("Rater Map"));
		raterMap.setMandatory(true);
		inputParams.add(initialTheta=new ParamFile("Initial Theta"));
		initialTheta.setMandatory(true);
		inputParams.add(targetClass=new ParamString("Target Class"));
		targetClass.setMandatory(true);
		
		inputParams.add(targetImage = new ParamVolume("Target Image"));
		targetImage.setMandatory(false);
		inputParams.add(registeredImages=new ParamVolumeCollection("Registered Images"));
		registeredImages.setMandatory(false);
		
	}

	@Override
	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(segmentation = new ParamVolume("Segmentation"));
		
	}

	@Override
	protected void execute(CalculationMonitor monitor)
			throws AlgorithmRuntimeException {
		AbstractMultiSetVoting sf=null;
		ImageData im = targetImage.getImageData();
		if(im != null){
			JistLogger.logOutput(JistLogger.WARNING, "We have an image volume");
			sf = new MultiSetLocallyWeightedVote(obsVals,raterMap.getValue(),initialTheta.getValue(),targetClass.getValue(),targetImage,registeredImages);
		}else{
			JistLogger.logOutput(JistLogger.WARNING, "We do not have an image volume");
			sf = new MultiSetMajorityVote(obsVals,raterMap.getValue(),initialTheta.getValue(),targetClass.getValue());
		}
		sf.run();
		segmentation.setValue(sf.getSegmentation());
	}

}
