<?xml version="1.0"?>
<!-- 	
 Copyright (c) 2003-2006 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
                                        
@ProposedRating Red (hyzheng)
@AcceptedRating Red (cxh)

This file transforms preprocessed HSIF files into Ptolemy MoML files.

The major steps include:
	1. Variables are mapped into Input/Outputs.
	2. HybridAutomata are mapped into Modal models.
	3. Locations are mapped into States and Refinements in Modal models.
       4. DiffEquations inside locations are mapped into differential equations.
	5. Expressions inside locations are mapped into assertions (Expression + ThrowModelError)
       in Refinements.

The transformed files show hybrid systems with a hierarchical structure as 
CT-ModalModel-CTEmbedded.

    Future work: 1. Export to HSIF
    
@author Haiyang Zheng
@version $Id: HSIF.xsl,v 1.38 2006/08/21 23:16:36 cxh Exp $ 
@since HyVisual 2.2
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
                         xmlns:xalan="http://xml.apache.org/xslt" version="1.0">

	<!-- DOCTYPE element includes public ID and system ID -->
	<xsl:output omit-xml-declaration="no" method="xml" doctype-system="http://ptolemy.eecs.berkeley.edu/xml/dtd/MoML_1.dtd"
		    doctype-public="-//UC Berkeley//DTD MoML 1//EN" indent="yes" />
	
	<!-- index every node via attribute _id -->
	<xsl:key name="nodeID" match="*" use="@_id"/>

    <!-- features of the XSLT 2.0 language -->
	<xsl:decimal-format name="comma" decimal-separator="," grouping-separator="."/>
	
	<!-- time function -->
	<xsl:variable name="now" xmlns:Date="/java.util.Date">
		<xsl:value-of select="Date:toString(Date:new())"/>
	</xsl:variable>
	
	<!-- configuration -->
	<xsl:param name="author">Ptolemy II</xsl:param>
	<xsl:preserve-space elements="*"/>

	<!-- ==========================================================
          root element
          Generate comment on author and time. Apply the according template. 					    
         ========================================================== -->
	<xsl:template match="/">
		<xsl:comment>
			Generated by <xsl:value-of select="$author"/> at <xsl:value-of select="$now"/>.
		</xsl:comment>
		<xsl:apply-templates/>
	</xsl:template>

<!-- ==========================================================
     DNHA element
     Construct a CT domain as top level, iterates the HybridAutomata,
     variables and parameters in HSIF to build Modal models,
     input/output ports and parameters in Ptolemy.
     Construct the relations based on all the ports and link ports.
     ========================================================== -->

<xsl:template match="DNHA">
    <xsl:element name="entity">

        <!-- At DNHA level, the CT domain is chosen as top level -->
        <!-- The director is a CT director. -->
        <xsl:call-template name="composite">
            <xsl:with-param name="name" select="@name"/>
            <xsl:with-param name="class" select="'ptolemy.actor.TypedCompositeActor'"/>
            <xsl:with-param name="type" select="'CT'"/>
        </xsl:call-template>

        <!-- Hybrid Automata are mapped into Modal Models -->
        <xsl:apply-templates select="HybridAutomaton">
        </xsl:apply-templates>

        <!-- Global parameters are associated with top level: CT domain-->
        <xsl:for-each select="IntegerParameter|RealParameter|BooleanParameter|Parameter">
            <xsl:call-template name="parameter">
            </xsl:call-template>
        </xsl:for-each>

        <!-- Global variables are mapped into I/O ports -->
        <xsl:for-each select="IntegerVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.actor.TypedIOPort'"/>
                <xsl:with-param name="dataType" select="'int'"/>
                <xsl:with-param name="environment" select = "'DNHA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="RealVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.actor.TypedIOPort'"/>
                <xsl:with-param name="dataType" select="'double'"/>
                <xsl:with-param name="environment" select = "'DNHA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="BooleanVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.actor.TypedIOPort'"/>
                <xsl:with-param name="dataType" select="'boolean'"/>
                <xsl:with-param name="environment" select = "'DNHA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <!-- Construct and link the relations based on I/O ports -->
        <xsl:for-each select="IntegerVariable|RealVariable|BooleanVariable">
            <xsl:element name="relation">
                <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
            </xsl:element>
            <xsl:element name="link">
                <xsl:attribute name="port"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="relation"><xsl:value-of select="@name"/></xsl:attribute>
            </xsl:element>
        </xsl:for-each>

        <!-- Consturct the relations based on I/O ports of Hybrid Automata -->
        <xsl:for-each select="HybridAutomaton">
            <xsl:variable name="prefix"><xsl:value-of select="@name"/></xsl:variable>
            <xsl:for-each select="IntegerVariable|RealVariable|BooleanVariable">
                <xsl:element name="relation">
                    <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
                </xsl:element>
            </xsl:for-each>
        </xsl:for-each>

        <!-- Link the relations and the I/O ports of Hybrid Automata -->
        <xsl:for-each select="HybridAutomaton">
            <xsl:variable name="prefix"><xsl:value-of select="@name"/></xsl:variable>
            <xsl:for-each select="IntegerVariable|RealVariable|BooleanVariable">
                <xsl:element name="link">
                    <xsl:attribute name="port"><xsl:value-of select="concat($prefix, '.', @name)"/></xsl:attribute>
                    <xsl:attribute name="relation"><xsl:value-of select="@name"/></xsl:attribute>
                </xsl:element>
            </xsl:for-each>
        </xsl:for-each>

    </xsl:element>
</xsl:template>

<!-- ==========================================================
     HybridAutomaton
     Construct a Modal model for a Hybrid Automaton
     including a FSM _Controller.
     Iterates the local variables, parameters, the locations and
     the transitions in HSIF to build ports, parameters,
     states, refinements and transitions in Ptolemy.
     Construct the relations and link the according ports.
     ========================================================== -->

<xsl:template match="HybridAutomaton">

    <xsl:element name="entity">

        <!-- At HybridAutomaton level, an Modal model is chosen. -->
        <!-- Construct the Modal model director. -->
        <xsl:call-template name="composite">
            <xsl:with-param name="name" select="@name"/>
            <xsl:with-param name="class" select="'ptolemy.domains.fsm.modal.ModalModel'"/>
            <xsl:with-param name="type" select="'Modal'"/>
        </xsl:call-template>

        <!-- parameters -->
        <xsl:for-each select="IntegerParameter|RealParameter|BooleanParameter|Parameter">
            <xsl:call-template name="parameter"/>
        </xsl:for-each>

        <!-- I/O ports -->
        <xsl:for-each select="IntegerVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.ModalPort'"/>
                <xsl:with-param name="dataType" select="'int'"/>
                <xsl:with-param name="environment" select = "'HA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="RealVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.ModalPort'"/>
                <xsl:with-param name="dataType" select="'double'"/>
                <xsl:with-param name="environment" select = "'HA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="BooleanVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.ModalPort'"/>
                <xsl:with-param name="dataType" select="'boolean'"/>
                <xsl:with-param name="environment" select = "'HA'"/>
            </xsl:call-template>
        </xsl:for-each>

        <!-- _Controller associated with the Modal Model -->
        <xsl:element name="entity">
            <!-- attributes of entity -->
            <xsl:attribute name="name">_Controller</xsl:attribute>
            <xsl:attribute name="class">ptolemy.domains.fsm.modal.ModalController</xsl:attribute>
            <xsl:call-template name="_Controller">
            </xsl:call-template>
         </xsl:element>

        <!-- Refinements from Discrete States -->
        <xsl:apply-templates select="Location" mode="refinement"/>

        <!-- link _Controller, Refinements -->
        <xsl:for-each select="IntegerVariable|RealVariable|BooleanVariable">
            <xsl:call-template name="relation">
                <xsl:with-param name="variableName" select="@name"/>
            </xsl:call-template>
        </xsl:for-each>

    </xsl:element>

</xsl:template>

<!-- ==========================================================
     Composite Entity
     The compositor with different directors.
     ========================================================== -->

<xsl:template name="composite">
    <xsl:param name="name" select="'Default Name'"/>
    <xsl:param name="class" select="'Default Class'"/>
    <xsl:param name="type" select="'Default Type'"/>
	<!-- attributes of entity -->
	<xsl:attribute name="name"><xsl:value-of select="$name"/></xsl:attribute>
	<xsl:attribute name="class"><xsl:value-of select="$class"/></xsl:attribute>

    <xsl:choose>
        <xsl:when test="$type='CT'"><xsl:call-template name="CTDirector"/></xsl:when>
        <xsl:when test="$type='CTEmbedded'"><xsl:call-template name="CTEmbeddedDirector"/></xsl:when>
        <xsl:when test="$type='Modal'"><xsl:call-template name="ModalDirector"/></xsl:when>
    </xsl:choose>
</xsl:template>

<!-- ==========================================================
     Directors
     Different directors for different domains.
     ========================================================== -->

<!-- CT MixedDirector -->
<xsl:template name="CTDirector">
    <xsl:element name="property">
        <!-- attributes of entity -->
        <xsl:attribute name="name">CT Director</xsl:attribute>
        <xsl:attribute name="class">ptolemy.domains.ct.kernel.CTMixedSignalDirector</xsl:attribute>
        <property name="stopTime" class="ptolemy.data.expr.Parameter" value="10.0"></property>
        <property name="maxStepSize" class="ptolemy.data.expr.Parameter" value="0.3"></property>
    </xsl:element>

    <!-- Help on how to use the model -->
    <property name="annotation1" class="ptolemy.kernel.util.Attribute">
        <property name="_location" class="ptolemy.kernel.util.Location" value="10.0, 70.0">
        </property>
        <property name="_iconDescription" class="ptolemy.kernel.util.SingletonConfigurableAttribute">
            <configure>
           <svg>
           <text x="20" style="font-size:14; font-family:sanserif; fill:blue" y="20">To run the model, follow these steps:
1. To automatically layout the actors, 
   type Ctrl-t
     or
   use the Graph -> Automatic Layout menu choice.

2. Connect a TimedPlotter from the Actors -> Sinks
   palette in the upper left tree widget to one of the
   relations.

3. To run the model, choose View -> Run Window.
   Feel free to modify the parameters in the run window.

For more help, choose Help from the upper menu bar.</text>
           </svg>
           </configure>
        </property>
        <property name="_hideName" class="ptolemy.data.expr.Parameter">
        </property>
    </property>

</xsl:template>

<!-- CT EmbeddedDirector -->
<xsl:template name="CTEmbeddedDirector">
    <xsl:element name="property">
        <!-- attributes of entity -->
        <xsl:attribute name="name">CT Embedded Director</xsl:attribute>
        <xsl:attribute name="class">ptolemy.domains.ct.kernel.CTEmbeddedDirector</xsl:attribute>
    </xsl:element>
</xsl:template>

<!-- Modal Director -->
<xsl:template name="ModalDirector">
<!-- The director for modal model is automatically configured by the ModalModel java class. -->
    <!-- _tableau -->
    <xsl:element name="property">
        <xsl:attribute name="name">_tableauFactory</xsl:attribute>
        <xsl:attribute name="class">ptolemy.vergil.fsm.modal.ModalTableauFactory</xsl:attribute>
    </xsl:element>
</xsl:template>

<!-- ==========================================================
     Controller in Modal Model incluidng States and Transitions
     A FSM in control of the transitions between states.
     Iterate the variables to build input ports and output
     ports. The ports are refinement ports.
     Iterate the states and transitions in HSIF to build FSM.
     ========================================================== -->

<!-- Controller for Modal Model -->
<xsl:template name="_Controller">

     <property name="annotation" class="ptolemy.kernel.util.Attribute">
        <property name="_controllerFactory" class="ptolemy.vergil.basic.NodeControllerFactory"/>
        <property name="_editorFactory" class="ptolemy.vergil.toolbox.AnnotationEditorFactory"/>
        <property name="_library" class="ptolemy.moml.LibraryAttribute">
           <configure>
                <entity name="state library" class="ptolemy.kernel.CompositeEntity">
                   <input source="ptolemy/configs/basicUtilities.xml"></input>
                   <entity name="state" class="ptolemy.domains.fsm.kernel.State">
                      <property name="_centerName" class="ptolemy.kernel.util.Attribute"></property>
                      <property name="_controllerFactory" class="ptolemy.vergil.fsm.modal.HierarchicalStateControllerFactory"></property>
                   </entity>
                </entity>
           </configure>
        </property>
     </property>

    <xsl:element name="property">
        <!-- attributes of property -->
        <xsl:attribute name="name">initialStateName</xsl:attribute>
        <xsl:attribute name="class">ptolemy.kernel.util.StringAttribute</xsl:attribute>
        <xsl:attribute name="value">
           <xsl:for-each select="Location">
               <xsl:if test="@initial='true'">
                    <xsl:value-of select="@name"/>
               </xsl:if>
           </xsl:for-each>
         </xsl:attribute>
    </xsl:element>

    <!-- I/O ports (RefinementPorts in _Controller) -->
    <xsl:for-each select="IntegerVariable">
        <xsl:call-template name="variable">
            <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
            <xsl:with-param name="dataType" select="'int'"/>
            <xsl:with-param name="environment" select = "'controller'"/>
        </xsl:call-template>
    </xsl:for-each>

    <xsl:for-each select="RealVariable">
        <xsl:call-template name="variable">
            <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
            <xsl:with-param name="dataType" select="'double'"/>
            <xsl:with-param name="environment" select = "'controller'"/>
        </xsl:call-template>
    </xsl:for-each>

    <xsl:for-each select="BooleanVariable">
        <xsl:call-template name="variable">
            <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
            <xsl:with-param name="dataType" select="'boolean'"/>
            <xsl:with-param name="environment" select = "'controller'"/>
        </xsl:call-template>
    </xsl:for-each>

    <!-- Discrete States -->
    <xsl:apply-templates select="Location" mode="state"/>

    <!-- Guard Expressions and Set Expressions -->
    <xsl:apply-templates select="Transition" mode="info"/>

    <!-- Transitions -->
    <xsl:apply-templates select="Transition" mode="link"/>

</xsl:template>

<!-- States of FSM from Discrete States -->
<xsl:template match="Location" mode="state">
    <xsl:element name="entity">
        <!-- attributes of entity -->
        <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
        <xsl:attribute name="class">ptolemy.domains.fsm.kernel.State</xsl:attribute>
        <xsl:element name="property">
            <xsl:attribute name="name">refinementName</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.util.StringAttribute</xsl:attribute>
            <xsl:attribute name="value"><xsl:value-of select="@name"/></xsl:attribute>
        </xsl:element>
        <property name="_controllerFactory" class="ptolemy.vergil.fsm.modal.HierarchicalStateControllerFactory">
        </property>
        <xsl:element name="port">
            <xsl:attribute name="name">incomingPort</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.ComponentPort</xsl:attribute>
        </xsl:element>
        <xsl:element name="port">
            <xsl:attribute name="name">outgoingPort</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.ComponentPort</xsl:attribute>
        </xsl:element>
    </xsl:element>
</xsl:template>

<!-- Transition information -->
<xsl:template match="Transition" mode="info">
    <xsl:variable name="stateID" select="@src_end_"/>
    <xsl:variable name="nextStateID" select="@dst_end_"/>
    <xsl:variable name="nextState" select="key('nodeID',$nextStateID)/@name"/>
    <xsl:element name="relation">
        <!-- attributes of relation -->
        <xsl:attribute name="name">
            <xsl:value-of select="concat(@_id, $stateID, $nextStateID)"/>
        </xsl:attribute>
        <xsl:attribute name="class">ptolemy.domains.fsm.kernel.Transition</xsl:attribute>
        <!-- attributes of guard Expression -->
        <xsl:element name="property">
            <xsl:attribute name="name">guardExpression</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.util.StringAttribute</xsl:attribute>
            <xsl:attribute name="value"><xsl:apply-templates select="Expr" mode="expr"/></xsl:attribute>
        </xsl:element>
        <!-- attributes of Output Action -->
        <xsl:element name="property">
            <xsl:attribute name="name">outputActions</xsl:attribute>
            <xsl:attribute name="class">ptolemy.domains.fsm.kernel.OutputActionsAttribute</xsl:attribute>
            <xsl:attribute name="value"></xsl:attribute>
        </xsl:element>
        <!-- attributes of Set Action -->
        <xsl:element name="property">
            <xsl:attribute name="name">setActions</xsl:attribute>
            <xsl:attribute name="class">ptolemy.domains.fsm.kernel.CommitActionsAttribute</xsl:attribute>

            <xsl:variable name="setActionString">
                <xsl:variable name="srcState" select="key('nodeID', $stateID)/@name"/>
                <xsl:variable name="dstState" select="key('nodeID', $nextStateID)/@name"/>

                <!-- Default setActions to ensure continuity of integrator states. -->
                <!-- This part always appears first because the following actions may over write part of it. -->
                <xsl:call-template name="defaultSetAction">
                    <xsl:with-param name="stateName" select="$dstState"/>
                </xsl:call-template>

                <!-- It turns out that exit actions of current state
                can be regarded as the entry actions of next state. -->
                <xsl:apply-templates select="key('nodeID', $stateID)/Action[@__child_as='exitAction']">
                    <xsl:with-param name="stateName" select="$dstState"/>
                </xsl:apply-templates>

                <xsl:apply-templates select="Action|key('nodeID', $nextStateID)/Action[@__child_as='entryAction']">
                    <xsl:with-param name="stateName" select="$dstState"/>
                </xsl:apply-templates>

            </xsl:variable>
            
            <xsl:variable name="stringLength" select="string-length($setActionString) - 1"/>

            <xsl:attribute name="value"><xsl:value-of select="substring($setActionString, 1, $stringLength)"/></xsl:attribute>

        </xsl:element>
        <xsl:element name="property">
            <xsl:attribute name="name">reset</xsl:attribute>
            <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
            <xsl:attribute name="value">true</xsl:attribute>
        </xsl:element>
        <xsl:element name="property">
            <xsl:attribute name="name">preemptive</xsl:attribute>
            <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
            <xsl:attribute name="value">false</xsl:attribute>
        </xsl:element>
    </xsl:element>

</xsl:template>

<!-- Transitions link states -->
<xsl:template match="Transition" mode="link">
    <xsl:variable name="src"><xsl:value-of select="@src_end_"/></xsl:variable>
    <xsl:variable name="dst"><xsl:value-of select="@dst_end_"/></xsl:variable>
    <xsl:variable name="relationName">
        <xsl:value-of select="concat(@_id, $src, $dst)"/>
    </xsl:variable>
    <xsl:for-each select="../Location[@_id=$dst]">
        <xsl:element name="link">
            <xsl:variable name="stateName"><xsl:value-of select="@name"/></xsl:variable>
            <xsl:attribute name="port"><xsl:value-of select="concat($stateName, '.incomingPort')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$relationName"/></xsl:attribute>
        </xsl:element>
    </xsl:for-each>
    <xsl:for-each select="../Location[@_id=$src]">
        <xsl:element name="link">
            <xsl:variable name="stateName"><xsl:value-of select="@name"/></xsl:variable>
            <xsl:attribute name="port"><xsl:value-of select="concat($stateName, '.outgoingPort')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$relationName"/></xsl:attribute>
        </xsl:element>
    </xsl:for-each>
</xsl:template>

<!-- ==========================================================
     Refinements
     Construct the invariants and flow equations in HSIF.
     Construct ports and relations and link ports.
     ========================================================== -->

<!-- Discrete State as State in FSM-->
<xsl:template match="Location" mode="refinement">
    <xsl:element name="entity">
        <xsl:call-template name="composite">
            <xsl:with-param name="name" select="@name"/>
            <xsl:with-param name="class" select="'ptolemy.domains.fsm.modal.Refinement'"/>
            <xsl:with-param name="type" select="'CTEmbedded'"/>
        </xsl:call-template>
        <xsl:element name="property">
            <xsl:attribute name="name">InitialState</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.util.Attribute</xsl:attribute>
        </xsl:element>
        <xsl:element name="property">
            <xsl:attribute name="name">EntryAction</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.util.Attribute</xsl:attribute>
        </xsl:element>
        <xsl:element name="property">
            <xsl:attribute name="name">ExitAction</xsl:attribute>
            <xsl:attribute name="class">ptolemy.kernel.util.Attribute</xsl:attribute>
        </xsl:element>

        <!-- localize the modal parameters-->
        <xsl:for-each select="../IntegerParameter|../RealParameter|../BooleanParameter|../Parameter">
            <xsl:call-template name="parameter"/>
        </xsl:for-each>

        <!-- I/O port (RefinementPort) -->
        <xsl:for-each select="../IntegerVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
                <xsl:with-param name="dataType" select="'int'"/>
                <xsl:with-param name="environment" select = "'FSM'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="../RealVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
                <xsl:with-param name="dataType" select="'double'"/>
                <xsl:with-param name="environment" select = "'FSM'"/>
            </xsl:call-template>
        </xsl:for-each>

        <xsl:for-each select="../BooleanVariable">
            <xsl:call-template name="variable">
                <xsl:with-param name="portType" select="'ptolemy.domains.fsm.modal.RefinementPort'"/>
                <xsl:with-param name="dataType" select="'boolean'"/>
                <xsl:with-param name="environment" select = "'FSM'"/>
            </xsl:call-template>
        </xsl:for-each>

        <!-- Invariants -->
        <xsl:apply-templates select="Expr" mode="invariant"/>

        <!-- Differential Equations -->
        <xsl:for-each select="DiffEquation">
            <xsl:call-template name="DiffEquation"/>
        </xsl:for-each>

        <!-- Algebraic Equations -->
        <xsl:for-each select="AlgEquation">
            <xsl:call-template name="AlgEquation"/>
        </xsl:for-each>

        <!-- Construct the internal links in refinements. -->

        <!-- Construct and link the relations based on I/O ports -->
        <xsl:for-each select="../IntegerVariable|../RealVariable|../BooleanVariable">
            <xsl:element name="relation">
                <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
            </xsl:element>
            <xsl:element name="link">
                <xsl:attribute name="port"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="relation"><xsl:value-of select="@name"/></xsl:attribute>
            </xsl:element>
        </xsl:for-each>

        <!-- If some inputs of expression are outputs of other integrators in the same location, -->
        <!-- they should be wired. [/descendant::VarRef/@var=$varID] -->
        <xsl:for-each select="DiffEquation/VarRef">
            <xsl:variable name="varName"><xsl:value-of select="key('nodeID',@var)/@name"/></xsl:variable>
            <xsl:variable name="varID"><xsl:value-of select="@var"/></xsl:variable>
            <!--
            <xsl:value-of select="concat('This state contains ', count(../../DiffEquation[descendant::VarRef/@var=$varID]), ' ', $varID)"/>
            -->
            <xsl:for-each select="../../DiffEquation/AExpr[descendant::VarRef/@var=$varID]">
                <xsl:variable name="integrator"><xsl:value-of select="key('nodeID',../VarRef/@var)/@name"/></xsl:variable>
                <xsl:variable name="expressionName"><xsl:value-of select="concat($integrator, 'FlowEquation')"/></xsl:variable>
                <xsl:element name="link">
                    <xsl:attribute name="port"><xsl:value-of select="concat($expressionName, '.', $varName)"/></xsl:attribute>
                    <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
                </xsl:element>
            </xsl:for-each>
            <!--
            <xsl:value-of select="concat('This state contains ', count(../../AlgEquation[descendant::VarRef/@var=$varID]), ' ', $varID)"/>
            -->
            <xsl:for-each select="../../AlgEquation/AExpr[descendant::VarRef/@var=$varID]">
                <xsl:variable name="outputName"><xsl:value-of select="key('nodeID',../VarRef/@var)/@name"/></xsl:variable>
                <xsl:variable name="expressionName"><xsl:value-of select="concat($outputName, 'AlgEquation')"/></xsl:variable>
                <xsl:element name="link">
                    <xsl:attribute name="port"><xsl:value-of select="concat($expressionName, '.', $varName)"/></xsl:attribute>
                    <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
                </xsl:element>
            </xsl:for-each>
        </xsl:for-each>

        <!-- Link the relations to the input ports of Differential Equations -->
        <xsl:for-each select="DiffEquation">
            <xsl:variable name="prefix"><xsl:value-of select="key('nodeID',VarRef/@var)/@name"/></xsl:variable>
            <xsl:for-each select="AExpr/descendant::VarRef">
                <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
                <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>
                <xsl:if test="$counts!=0">
                    <xsl:element name="link">
                        <xsl:attribute name="port"><xsl:value-of select="concat($prefix, 'FlowEquation.', $varName)"/></xsl:attribute>
                        <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
                    </xsl:element>
                </xsl:if>
            </xsl:for-each>
        </xsl:for-each>

        <!-- Link the relations to the input ports of Algebraic Equations -->
        <xsl:for-each select="AlgEquation">
            <xsl:variable name="prefix"><xsl:value-of select="key('nodeID',VarRef/@var)/@name"/></xsl:variable>
            <xsl:for-each select="AExpr/descendant::VarRef">
                <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
                <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>
                <xsl:if test="$counts!=0">
                    <xsl:element name="link">
                        <xsl:attribute name="port"><xsl:value-of select="concat($prefix, 'AlgEquation.', $varName)"/></xsl:attribute>
                        <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
                    </xsl:element>
                </xsl:if>
            </xsl:for-each>
        </xsl:for-each>

        <!-- Link the relations and the input ports of Invariants-->
        <!-- Link the output port of Invariants and the input of throwModelError actor -->
        <xsl:for-each select="Expr">
            <xsl:variable name="name">invariant</xsl:variable>
            <xsl:for-each select="descendant::VarRef">
                <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
                <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>

		<xsl:value-of select="$varName"/>
		<xsl:value-of select="' BBB '"/>
		<xsl:value-of select="$counts"/>

                <xsl:if test="$counts!=0">
                    <xsl:element name="link">
                        <xsl:attribute name="port"><xsl:value-of select="concat($name, '.', $varName)"/></xsl:attribute>
                        <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
                    </xsl:element>
                </xsl:if>
                <xsl:element name="relation">
                    <xsl:attribute name="name">relationBetweenInvariantAndThrowModelError</xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
                </xsl:element>
                <xsl:element name="link">
                    <xsl:attribute name="port"><xsl:value-of select="concat($name, '.', 'output')"/></xsl:attribute>
                    <xsl:attribute name="relation">relationBetweenInvariantAndThrowModelError</xsl:attribute>
                </xsl:element>
                <xsl:element name="link">
                    <xsl:attribute name="port"><xsl:value-of select="concat('throwModelError', '.', 'input')"/></xsl:attribute>
                    <xsl:attribute name="relation">relationBetweenInvariantAndThrowModelError</xsl:attribute>
                </xsl:element>

            </xsl:for-each>
        </xsl:for-each>

    </xsl:element>
</xsl:template>

<!-- ========================================================== -->
<!-- Link _Controller, Refinements via relations		-->
<!-- ========================================================== -->

<xsl:template name="relation">
    <xsl:param name="variableName" select="'Default VariableName'"/>
    <xsl:element name="relation">
        <xsl:attribute name="name"><xsl:value-of select="concat($variableName, 'Relation')"/></xsl:attribute>
        <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
    </xsl:element>
    <xsl:element name="link">
        <xsl:attribute name="port"><xsl:value-of select="$variableName"/></xsl:attribute>
        <xsl:attribute name="relation"><xsl:value-of select="concat($variableName, 'Relation')"/></xsl:attribute>
    </xsl:element>
    <xsl:element name="link">
        <xsl:attribute name="port"><xsl:value-of select="concat('_Controller.', $variableName)"/></xsl:attribute>
        <xsl:attribute name="relation"><xsl:value-of select="concat($variableName, 'Relation')"/></xsl:attribute>
    </xsl:element>
    <xsl:for-each select="../Location">
        <xsl:element name="link">
           <xsl:attribute name="port"><xsl:value-of select="concat(@name, '.', $variableName)"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="concat($variableName, 'Relation')"/></xsl:attribute>
        </xsl:element>
    </xsl:for-each>
</xsl:template>

<!-- ========================================================== -->
<!-- Parameters 						-->
<!-- ========================================================== -->
<xsl:template name="parameter">
    <xsl:element name="property">
        <!-- attributes of property -->
        <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
        <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
        <xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
    </xsl:element>
</xsl:template>

<xsl:template match="RealParameter">
    <xsl:element name="property">
        <!-- attributes of property -->
        <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
        <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
        <xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
    </xsl:element>
</xsl:template>

<!-- ========================================================== -->
<!-- Variables 							-->
<!-- ========================================================== -->
<xsl:template name="variable">
    <xsl:param name="portType" select="'Default PortType'"/>
    <xsl:param name="environment" select = "'Default Environment'"/>
    <xsl:param name="dataType" select = "'Default DataType'"/>
    <xsl:choose>
        <!--xsl:when test="@kind='Controlled'"-->
        <xsl:when test="@kind='Output'">
            <xsl:element name="port">
                <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="class"><xsl:value-of select="$portType"/></xsl:attribute>
                <xsl:if test="$environment='controller'">
                    <xsl:element name="property">
                        <xsl:attribute name="name">input</xsl:attribute>
                    </xsl:element>
                </xsl:if>
                <xsl:element name="property">
                    <xsl:attribute name="name">output</xsl:attribute>
                </xsl:element>
                <xsl:call-template name="value"/>
                <!--property name="_type" class="ptolemy.actor.TypeAttribute" value="double"-->
                <xsl:element name="property">
                    <xsl:attribute name="name">_type</xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypeAttribute</xsl:attribute>
                    <xsl:attribute name="value"><xsl:value-of select="$dataType"/></xsl:attribute>
                </xsl:element>
            </xsl:element>
        </xsl:when>
        <xsl:when test="@kind='Input'">
            <xsl:element name="port">
                <xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
                <xsl:attribute name="class"><xsl:value-of select="$portType"/></xsl:attribute>
                <xsl:element name="property">
                    <xsl:attribute name="name">input</xsl:attribute>
                </xsl:element>
                <xsl:element name="property">
                    <xsl:attribute name="name">_type</xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypeAttribute</xsl:attribute>
                    <xsl:attribute name="value"><xsl:value-of select="$dataType"/></xsl:attribute>
                </xsl:element>
                <xsl:call-template name="value"/>
            </xsl:element>
        </xsl:when>
    </xsl:choose>

</xsl:template>

<!-- ========================================================== -->
<!-- Values 							-->
<!-- ========================================================== -->
<xsl:template name="value">
    <xsl:choose>
        <xsl:when test="name()='IntegerVariable'">
            <xsl:element name="property">
                <xsl:attribute name="name">minValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@minValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">maxValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@maxValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">initialMinValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@initialMinValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">initialMaxValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@initialMaxValue"/></xsl:attribute>
            </xsl:element>
        </xsl:when>
        <xsl:when test="name()='RealVariable'">
            <xsl:element name="property">
                <xsl:attribute name="name">minValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@minValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">maxValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@maxValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">initialMinValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@initialMinValue"/></xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">initialMaxValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@initialMaxValue"/></xsl:attribute>
            </xsl:element>
        </xsl:when>
        <xsl:when test="name()='BooleanVariable'">
            <xsl:element name="property">
                <xsl:attribute name="name">defaultValue</xsl:attribute>
                <xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="@defaultValue"/></xsl:attribute>
            </xsl:element>
        </xsl:when>
    </xsl:choose>
</xsl:template>

<!-- ==========================================================
     Expressions, Invariants, DiffEquations, AlgEquations, UpdateActions
     ========================================================== -->

<!-- Expression -->
<xsl:template match="Expr" mode="expr">
    <xsl:apply-templates select="LExpr"/>
</xsl:template>

<xsl:template match="LExpr">
    <xsl:apply-templates select="RExpr"/>
    <xsl:for-each select="LExprR">
        <xsl:text> </xsl:text>
        <xsl:variable name="temp">
            <xsl:value-of select="@logicOp"/>
        </xsl:variable>
        <xsl:choose>
            <xsl:when test="$temp='and'"><xsl:text>&amp;&amp;</xsl:text></xsl:when>
            <xsl:when test="$temp='or'"><xsl:text>||</xsl:text></xsl:when>
            <xsl:when test="$temp=' and '"><xsl:text>&amp;&amp;</xsl:text></xsl:when>
            <xsl:when test="$temp=' or '"><xsl:text>||</xsl:text></xsl:when>
            <!--xsl:otherwise><xsl:value-of select="$temp"/></xsl:otherwise-->
        </xsl:choose>
        <xsl:text> </xsl:text>
        <xsl:apply-templates select="LExpr|RExpr"/>
    </xsl:for-each>
</xsl:template>

<xsl:template match="RExpr">
    <xsl:apply-templates select="AExpr"/>
    <xsl:for-each select="RExprR">
        <xsl:value-of select="@relOp"/>
        <xsl:apply-templates select="RExpr|AExpr"/>
    </xsl:for-each>
</xsl:template>

<xsl:template match="AExpr">
    <xsl:apply-templates select="MExpr"/>
    <xsl:for-each select="AExprR">
        <xsl:value-of select="@addOp"/>
        <xsl:apply-templates select="AExpr"/>
    </xsl:for-each>
</xsl:template>

<xsl:template match="MExpr">
    <xsl:for-each select="Const|VarRef|ParRef">
        <xsl:for-each select="@unOp">
            <xsl:variable name="temp"><xsl:value-of select="."/></xsl:variable>
            <xsl:choose>
                <xsl:when test="$temp!='NOP'"><xsl:value-of select="$temp"/></xsl:when>
            </xsl:choose>
        </xsl:for-each>
        <xsl:value-of select="@value|key('nodeID',@var)/@name|@name"/>
    </xsl:for-each>
    <xsl:for-each select="FunctionCall">
        <xsl:for-each select="@unOp">
            <xsl:variable name="temp"><xsl:value-of select="."/></xsl:variable>
            <xsl:choose>
                <xsl:when test="$temp!='NOP'"><xsl:value-of select="$temp"/></xsl:when>
            </xsl:choose>
        </xsl:for-each>
        <xsl:variable name="tempName" select="@name"/>
        <xsl:value-of select='translate($tempName,"ABCDEFGHIJKLMNOPQRSTUVWXYZ","abcdefghijklmnopqrstuvwxyz")'/>
        <xsl:text>(</xsl:text>
        <xsl:for-each select="ExprArg">
            <xsl:apply-templates select="Expr" mode="expr"/>
            <xsl:if test="position()!=last()">,</xsl:if>
        </xsl:for-each>
        <xsl:text>)</xsl:text>
    </xsl:for-each>
    <xsl:apply-templates select="ParExpr"/>
    <xsl:for-each select="MExprR">
        <xsl:value-of select="@mulOp"/>
        <xsl:apply-templates select="MExpr"/>
    </xsl:for-each>
</xsl:template>

<xsl:template match="ParExpr">
        <xsl:variable name="temp">
            <xsl:value-of select="@unOp"/>
        </xsl:variable>
        <xsl:choose>
            <xsl:when test="$temp='NOT'"><xsl:text>!</xsl:text></xsl:when>
            <xsl:when test="$temp=' NOT '"><xsl:text>!</xsl:text></xsl:when>
            <!--xsl:otherwise><xsl:value-of select="$temp"/></xsl:otherwise-->
        </xsl:choose>
    <xsl:text>(</xsl:text>
    <xsl:apply-templates select="Expr" mode="expr"/>
    <xsl:text>)</xsl:text>
</xsl:template>

<!-- Invariants -->
<xsl:template match="Expr" mode="invariant">
    <xsl:element name="entity">
        <xsl:attribute name="name">invariant</xsl:attribute>
        <xsl:attribute name="class">ptolemy.actor.lib.Expression</xsl:attribute>
        <xsl:element name="property">
            <xsl:attribute name="name">expression</xsl:attribute>
            <!--<xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute>-->
            <xsl:attribute name="value">
                <xsl:text>!(</xsl:text>
                <xsl:apply-templates select="." mode="expr"/>
                <xsl:text>)</xsl:text>
            </xsl:attribute>
        </xsl:element>
        <xsl:element name="port">
            <xsl:attribute name="name">output</xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
            <xsl:element name="property">
                <xsl:attribute name="name">output</xsl:attribute>
            </xsl:element>
            <xsl:element name="property">
                <xsl:attribute name="name">_type</xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypeAttribute</xsl:attribute>
                <xsl:attribute name="value"><xsl:value-of select="'boolean'"/></xsl:attribute>
            </xsl:element>
        </xsl:element>

        <xsl:for-each select="descendant::VarRef">
            <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
            <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>

	    <!--
	    <xsl:value-of select="$varName"/>
            <xsl:value-of select="' AAA '"/>
            <xsl:value-of select="$counts"/>
	    -->

            <xsl:if test="$counts!=0">
                <xsl:element name="port">
                    <xsl:attribute name="name"><xsl:value-of select="$varName"/></xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                    <xsl:element name="property">
                        <xsl:attribute name="name">input</xsl:attribute>
                    </xsl:element>
                </xsl:element>
            </xsl:if>
        </xsl:for-each>
    </xsl:element>

    <xsl:element name="entity">
        <xsl:attribute name="name">throwModelError</xsl:attribute>
        <xsl:attribute name="class">ptolemy.actor.lib.ThrowModelError</xsl:attribute>
        <xsl:element name="property">
            <xsl:attribute name="name">message</xsl:attribute>
            <xsl:attribute name="value">
                <xsl:text>AssertionModelError</xsl:text>
            </xsl:attribute>

        </xsl:element>
    </xsl:element>

</xsl:template>

<!-- Default setActions to ensure the continuity of integrator states. -->
<!-- While the states of integrators could be modified by other following actions. -->

<xsl:template name="defaultSetAction">
    <xsl:param name="stateName" select="'Default StateName'"/>
    <xsl:for-each select="../IntegerVariable[@kind='Output']|../RealVariable[@kind='Output']|../BooleanVariable[@kind='Output']">
        <xsl:variable name="temp"><xsl:value-of select="@name"/></xsl:variable>
        <xsl:variable name="stateID"><xsl:value-of select="@_id"/></xsl:variable>
        <xsl:variable name="counts" select="count(../Location/DiffEquation/VarRef[@var=$stateID])"/>
        <xsl:if test="$counts!=0">
            <xsl:choose>
                <xsl:when test="$temp!=''"><xsl:value-of select="concat($stateName, '.', $temp, '.initialState=', $temp)"/></xsl:when>
            </xsl:choose>
            <xsl:value-of select="';'"/>
        </xsl:if>
    </xsl:for-each>
</xsl:template>

<!-- UpdateActions -->
<xsl:template match="Action">
    <xsl:param name="stateName" select="'Default StateName'"/>
    <xsl:for-each select="VarRef">
        <xsl:variable name="index" select="position()"/>
        <xsl:variable name="id" select="@var"/>
        <xsl:variable name="temp"><xsl:value-of select="key('nodeID', @var)/@name"/></xsl:variable>
        <xsl:variable name="count" select="count(../../../IntegerVariable[@_id = $id]|../../../RealVariable[@_id = $id]|../../../BooleanVariable[@_id = $id])"/>
        <xsl:choose>
            <xsl:when test="$temp!=''">
                <xsl:if test="$count=0"><xsl:value-of select="$temp"/></xsl:if>
                <xsl:if test="$count!=0"><xsl:value-of select="concat($stateName, '.', $temp, '.initialState')"/></xsl:if>
            </xsl:when>
        </xsl:choose>
        <xsl:value-of select="'='"/>
        <xsl:apply-templates select="../Expr[$index]" mode="expr"/>
        <xsl:value-of select="';'"/>
    </xsl:for-each>
</xsl:template>

<!-- DiffEquation -->
<xsl:template name="DiffEquation">
    <!-- integrator(s) -->
    <xsl:for-each select="VarRef">
        <xsl:element name="entity">
            <xsl:variable name="initialValue"><xsl:value-of select="key('nodeID',@var)/@initialMinValue"/></xsl:variable>
            <xsl:attribute name="name"><xsl:value-of select="key('nodeID',@var)/@name"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.domains.ct.lib.Integrator</xsl:attribute>
            <property name="initialState" class="ptolemy.data.expr.Parameter">
                <xsl:attribute name="value">
                    <xsl:if test="$initialValue!=''"><xsl:value-of select="$initialValue"/></xsl:if>
                    <xsl:if test="$initialValue=''"><xsl:value-of select="0.0"/></xsl:if>
                </xsl:attribute>
            </property> 
            <xsl:element name="port">
                <xsl:attribute name="name">output</xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                <xsl:element name="property">
                    <xsl:attribute name="name">output</xsl:attribute>
                </xsl:element>
            </xsl:element>
            <xsl:element name="port">
                <xsl:attribute name="name">input</xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                <xsl:element name="property">
                    <xsl:attribute name="name">input</xsl:attribute>
                </xsl:element>
            </xsl:element>
        </xsl:element>
    </xsl:for-each>

    <!-- flowEquation(s) -->
    <xsl:for-each select="AExpr">
        <xsl:element name="entity">
            <xsl:variable name="integrator"><xsl:value-of select="key('nodeID',../VarRef/@var)/@name"/></xsl:variable>
            <xsl:attribute name="name"><xsl:value-of select="concat($integrator, 'FlowEquation')"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.lib.Expression</xsl:attribute>
            <xsl:element name="property">
                <xsl:attribute name="name">expression</xsl:attribute>
                <!--xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute-->
                <xsl:attribute name="value"><xsl:apply-templates select="."/></xsl:attribute>
            </xsl:element>
            <xsl:element name="port">
                <xsl:attribute name="name">output</xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                <xsl:element name="property">
                    <xsl:attribute name="name">output</xsl:attribute>
                </xsl:element>
                <xsl:element name="property">
                    <xsl:attribute name="name">_type</xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypeAttribute</xsl:attribute>
                    <xsl:attribute name="value"><xsl:value-of select="'double'"/></xsl:attribute>
                </xsl:element>
            </xsl:element>

            <xsl:for-each select="descendant::VarRef">
                <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
                <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>
                <xsl:if test="$counts!=0">
                    <xsl:element name="port">
                        <xsl:attribute name="name"><xsl:value-of select="$varName"/></xsl:attribute>
                        <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                        <xsl:element name="property">
                            <xsl:attribute name="name">input</xsl:attribute>
                        </xsl:element>
                    </xsl:element>
                </xsl:if>
            </xsl:for-each>

        </xsl:element>
    </xsl:for-each>

    <xsl:for-each select="VarRef">
        <xsl:variable name="varName"><xsl:value-of select="key('nodeID',@var)/@name"/></xsl:variable>
        <xsl:variable name="varID"><xsl:value-of select="@var"/></xsl:variable>
        <xsl:element name="relation">
            <xsl:attribute name="name"><xsl:value-of select="$varName"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
        </xsl:element>
        <xsl:element name="relation">
            <xsl:attribute name="name"><xsl:value-of select="concat($varName, 'relationBetweenIntegratorAndExpression')"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="concat($varName, '.input')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="concat($varName, 'relationBetweenIntegratorAndExpression')"/></xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="concat($varName, 'FlowEquation', '.output')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="concat($varName, 'relationBetweenIntegratorAndExpression')"/></xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="$varName"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="concat($varName, '.output')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
        </xsl:element>

    </xsl:for-each>

</xsl:template>

<!-- AlgEquation -->
<xsl:template name="AlgEquation">
    <!-- equation -->
    <xsl:for-each select="AExpr">
        <xsl:element name="entity">
            <xsl:variable name="outputName"><xsl:value-of select="key('nodeID',../VarRef/@var)/@name"/></xsl:variable>
            <xsl:attribute name="name"><xsl:value-of select="concat($outputName, 'AlgEquation')"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.lib.Expression</xsl:attribute>
            <xsl:element name="property">
                <xsl:attribute name="name">expression</xsl:attribute>
                <!--xsl:attribute name="class">ptolemy.data.expr.Parameter</xsl:attribute-->
                <xsl:attribute name="value"><xsl:apply-templates select="."/></xsl:attribute>
            </xsl:element>

            <xsl:element name="port">
                <xsl:attribute name="name">output</xsl:attribute>
                <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                <xsl:element name="property">
                    <xsl:attribute name="name">output</xsl:attribute>
                </xsl:element>
                <xsl:element name="property">
                    <xsl:attribute name="name">_type</xsl:attribute>
                    <xsl:attribute name="class">ptolemy.actor.TypeAttribute</xsl:attribute>
                    <xsl:attribute name="value"><xsl:value-of select="'double'"/></xsl:attribute>
                </xsl:element>
            </xsl:element>

            <xsl:for-each select="descendant::VarRef">
                <xsl:variable name="varName" select="key('nodeID',@var)/@name"/>
                <xsl:variable name="counts" select="count(//DNHA/HybridAutomaton/IntegerVariable[@name=$varName]|//DNHA/HybridAutomaton/RealVariable[@name=$varName]|//DNHA/HybridAutomaton/BooleanVariable[@name=$varName])"/>
                <xsl:if test="$counts!=0">
                    <xsl:element name="port">
                        <xsl:attribute name="name"><xsl:value-of select="$varName"/></xsl:attribute>
                        <xsl:attribute name="class">ptolemy.actor.TypedIOPort</xsl:attribute>
                        <xsl:element name="property">
                            <xsl:attribute name="name">input</xsl:attribute>
                        </xsl:element>
                    </xsl:element>
                </xsl:if>
            </xsl:for-each>

        </xsl:element>
    </xsl:for-each>

    <xsl:for-each select="VarRef">
        <xsl:variable name="varName"><xsl:value-of select="key('nodeID',@var)/@name"/></xsl:variable>
        <xsl:variable name="varID"><xsl:value-of select="@var"/></xsl:variable>
        <xsl:element name="relation">
            <xsl:attribute name="name"><xsl:value-of select="$varName"/></xsl:attribute>
            <xsl:attribute name="class">ptolemy.actor.TypedIORelation</xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="$varName"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
        </xsl:element>
        <xsl:element name="link">
            <xsl:attribute name="port"><xsl:value-of select="concat($varName, 'AlgEquation', '.output')"/></xsl:attribute>
            <xsl:attribute name="relation"><xsl:value-of select="$varName"/></xsl:attribute>
        </xsl:element>

    </xsl:for-each>

</xsl:template>

</xsl:stylesheet>
