package apps;

import data.*;
import misc.*;
import tools.*;

import java.util.*;
import java.util.logging.*;
import java.util.regex.*;
import java.io.*;
import java.text.*;

/**
 * Converts point sets to scheme files, given additional imaging parameters. Outputs V2 scheme files.
 * 
 * @author Philip Cook
 * @version $Id: PointSetToScheme2.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public class PointSetToScheme2 {

    /**
     * logging object
     */
    private static final Logger logger = Logger.getLogger("apps.PointSetToScheme2");



    public static void main(String[] args) {

	CL_Initializer.CL_init(args);

	// need diffusion time and modQ from command line

	boolean flipX = false;
	boolean flipY = false;
	boolean flipZ = false;

	// insist on UK style numbers, no comma decimal points
	Locale.setDefault(Locale.UK);


	// Some scanners define b-value as |g|^2 * \beta
	// where \beta is what they CLAIM the b-value is.
	// If you use normalized gradient directions, you need to increase b accordingly
	// to make the same ADC calculation.
	boolean useGradMod = false;


	// global b value
	double b = 0.0;

	// number of zero measurements to prepend onto scheme
	int M = 0;

	DecimalFormat df = new DecimalFormat("   0.000000;  -0.000000");
	DecimalFormat bValDF = new DecimalFormat("   0.000E00");

	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals("-bvalue")) {
		b = Double.parseDouble(args[i+1]);

		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-usegradmod")) {
		useGradMod = true;
		logger.info("Gradient direction magnitude will be incorporated into b-values");
		CL_Initializer.markAsParsed(i);
	    }
	    else if (args[i].equals("-addzeromeas")) {
		M = Integer.parseInt(args[i+1]);

		CL_Initializer.markAsParsed(i,2);
	    }
	    else if (args[i].equals("-flipx")) {
                flipX = true;
                CL_Initializer.markAsParsed(i);
            }
            else if (args[i].equals("-flipy")) {
                flipY = true;
                CL_Initializer.markAsParsed(i);

            }
            else if (args[i].equals("-flipz")) {
                flipZ = true;
                CL_Initializer.markAsParsed(i);
            }

	}

	CL_Initializer.checkParsing(args);
	
	try {

	    double[][] points = PointSetToScheme.readPoints(CL_Initializer.inputFile, flipX, flipY, flipZ);

	    double[] mod = PointSetToScheme.getPointModulus(points);

	    points = PointSetToScheme.normalizePoints(points);

	    StringBuffer buffer = new StringBuffer();

	    buffer.append("VERSION: 2\n");
	    
	    int N = points.length;

	    // prepend zero measurements
	    for (int i = 0; i < M; i++) {
		for (int j = 0; j < 3; j++) {
		    buffer.append(df.format(0.0));
		}
		buffer.append(bValDF.format(0.0) + "\n");
	    }

	    // set to true if gradient directions are not orientations and 
	    // we are not using their mod to scale the b-value
	    boolean warnAboutGradDirMod = false;

	    for (int i = 0; i < N; i++) {
		for (int j = 0; j < 3; j++) {
		    buffer.append(df.format(points[i][j]));
		}
		
		if (points[i][0] == 0.0 && points[i][1] == 0.0 && points[i][2] == 0.0) {
		    buffer.append(bValDF.format(0.0) + "\n");
		}
		else {
		    if (useGradMod) {
			buffer.append(bValDF.format(b * mod[i] * mod[i]) + "\n");
		    }
		    else {
			
			if ( mod[i] != 0.0 && !(Math.abs(1.0 - mod[i]) < 1E-5) ) {
			    warnAboutGradDirMod = true;
			}
			
			buffer.append(bValDF.format(b) + "\n");
		    }

		}
	    }

	    if (OutputManager.outputFile == null) {
		System.out.print(buffer.toString());
	    }
	    else {
		
		FileOutput out = new FileOutput(OutputManager.outputFile);
		
		out.writeString(buffer.toString());
		
		out.close();
	    }
	    

	    if (warnAboutGradDirMod) {
		logger.warning("Some measurements have non-unit gradient directions. Directions have been " + 
			       "normalized to unit length");
	    }

	}
	catch (IOException e) {
	    throw new LoggedException(e);
	}


	   
    }



}
