package imaging;

import data.*;
import misc.*;
import numerics.*;
import tools.*;

import java.io.*;
import java.util.logging.Logger;
import java.util.zip.*;

/**
 * Meta image header, provides support for a common ITK file format. 
 * Currently only supports 3D binary images (images may have multiple components).
 * <p>
 * Compressed data is read as a ZIP archive, assumed to contain one entry. This feature
 * is untested, so use with caution.
 * 
 * @author Philip Cook
 * @version $Id: MetaImageHeader.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 */
public class MetaImageHeader extends ImageHeader {

    /**
     * Logging object
     */
    private static Logger logger = Logger.getLogger("camino.imaging.MetaImageHeader");


    /** ObjectType, which for us is always Image. */
    public final ObjectType objectType = ObjectType.IMAGE;

    /** NDims, number of dimensions. Always 3. */
    public final int nDims = 3;
    
    /** BinaryData, which is always true. */
    public final boolean binaryData = true;
    
    /** CompressedData, if true, data is compressed with zlib. */
    public boolean compressedData = false;



    /** 
     * BinaryDataByteOrderMSB = !<code>intelByteOrder</code>. 
     * Field named for consistency with AnalyzeHeader. The Meta specification also mentions
     * the seemingly redundant ElementByteOrderMSB. ITK examples contain either field, but 
     * never both. Currently we read either but only write BinaryDataByteOrderMSB.
     * 
     */
    public boolean intelByteOrder = false;

    /** TransformMatrix, a 3x3 rotation matrix. */
    public RealMatrix transformation = RealMatrix.identity(3);
    
    /** Offset */
    public double[] offset = new double[] {0.0, 0.0, 0.0};

    /** CenterOfRotation */
    public double[] rotationCentre = new double[] {0.0, 0.0, 0.0};


    
    /**
     * AnatomicalOrientation field. From the Meta IO 
     * <a href="http://www.itk.org/Wiki/images/2/27/MetaIO-Introduction.pdf">spec</a>:
     * <blockquote>
     * "Specify anatomic ordering of the axis.  Use only [R|L] | 
     * [A|P] | [S|I] per axis.   For example, if the three letter code for (column index, 
     * row index, slice index is) ILP, then the origin is at the superior, right, anterior 
     * corner of the volume, and therefore the axes run from superior to inferior, from 
     * right to left, from anterior to posterior."
     * </blockquote>
     * <p>
     * Camino reads files in the order of left-right, posterior-anterior, inferior-superior.
     * The origin of the image (voxel 0,0,0) is left, posterior, inferior. The orientation 
     * could therefore be described as LPI, but according to the above it's RAS. Because SNAP
     * expects the codes to match the position of the origin, we'll default to LPI.
     *
     */
    public AnatomicalOrientation anatomicalOrientation = AnatomicalOrientation.LPI;


    /** Voxel dimensions. */
    public double[] spacing = new double[] {1.0, 1.0, 1.0};

    /** Number of voxels in each dimension. */
    public int[] dimSize = new int[] {1, 1, 1};


    /**
     * ElementNumberOfChannels, number of components.
     */
    public int channels = 1;


    /**
     * ElementType, ie datatype. 
     */ 
    public DataType dataType = DataType.DOUBLE;


    /**
     * data file. Should be "LOCAL" if the data is in the same file as the header.
     * Note that the variable localDataFile, which should be an absolute path, will be used
     * to read the image within Camino.
     */
    public String dataFile = "LOCAL";


    /**
     * This variable contains the absolute path to the file containing the data (which may be the
     * same as the file containing the header). It is set when readHeader is called. 
     */
    public String localDataFile = "";


    /** 
     * Offset of data in elementDataFile. Set when readHeader is called. 
     */
    public int dataByteOffset = 0;


    
    /**
     * Data type enum. We use a subset of allowed Meta types that correspond to Java data types.
     * UCHAR is an 8 bit unsigned byte, CHAR is an 8 bit signed byte, ie a Java <code>byte</code>. 
     * The SHORT, INT, FLOAT, LONG and DOUBLE types correspond to the Java types of the respective name.
     */
    public enum DataType {
	
	// numeric values defined for consistency with Analyze
	UCHAR("MET_UCHAR", 2),
	CHAR("MET_CHAR", 132),
	SHORT("MET_SHORT", 4),
	USHORT("MET_USHORT", 130),
	INT("MET_INT", 8),
	UINT("MET_UINT", 136),
	FLOAT("MET_FLOAT", 16),
	DOUBLE("MET_DOUBLE", 64),
	LONG("MET_LONG", 199);


        DataType(String name, int i) {
	    typeName = name;
	    index = i;
	}


	public String toString() {
	    return typeName;
	}


	public static DataType getDataType(String s) {

	    for (DataType type : DataType.values()) {
		if (s.equals(type.typeName)) {
		    return type;
		}
	    }
	    
	    // use of enum should make this impossible
	    throw new LoggedException("Unsupported Meta data type " + s);
	}

	private final String typeName;
	private final int index;

    }



    /**
     * Object type enum. Only "Image" is allowed.
     */
    public enum ObjectType {

	// only support image for now, others exist
	IMAGE("Image");
	
	// note that enum.name is set automatically to the instance name, eg IMAGE, 
	// and cannot be modified.
        ObjectType(String s) {
	    typeName = s;
	}

	public String toString() {
	    return typeName;
	}


	public static ObjectType getObjectType(String s) {
	    
	    for (ObjectType type : ObjectType.values()) {
		if (s.equals(type.typeName)) {
		    return type;
		}
	    }

	    // use of enum should make this impossible
	    throw new LoggedException("Unsupported Meta object type: '" + s + "'");
	    
	}	

	private final String typeName;

    }


    /**
     * Supported anatomical orientations [L|R] [A|P] [I|S].
     *
     */
    public enum AnatomicalOrientation {
	// default string value is the variable name
	LAS,
	LAI,
	LPS,
	LPI,
	RAS,
	RAI,
	RPS,
	RPI;

	public static AnatomicalOrientation getAnatomicalOrientation(String s) {
	   
	    for (AnatomicalOrientation orient : AnatomicalOrientation.values()) {
		if (s.equals(orient.name())) {
		    return orient;
		}
	    }
	   
	    // use of enum should make this impossible
	    throw new LoggedException("Unsupported Anatomical Orientation " + s);
	   
	}
       
    }

    


    /**
     * Sets all values to their default.
     *
     */
    public MetaImageHeader() {

	
    }

    

    /**
     * Read the text header from a file. Adds either ".mha" or ".mhd" if omitted.
     *
     */
    public static MetaImageHeader readHeader(String file) throws IOException {

	if (!( file.endsWith(".mha") || file.endsWith(".mhd") )) {

	    // look for file, header first
	    File f = new File(file + ".mha");

	    if (f.exists()) {
		file += ".mha";
	    }
	    else {
		f = new File(file + ".mhd");
		
		if (f.exists()) {
		    file += ".mhd";
		}	    
		else {
		    throw new FileNotFoundException("Can't find file " + file + 
						    ".mha or " + file + ".mhd");
		}
	    }
	}

	DataInputStream din = 
	    new DataInputStream(new BufferedInputStream(new FileInputStream(file), 1024));
	
	MetaImageHeader mh = readHeader(din);

	din.close();

	if (mh.dataFile.equals("LOCAL")) {
	    mh.localDataFile = file;
	}
	else {
	    // attach path to file
	    String prefix = "";

	    String slashie = File.separator;
	    
	    int index = file.lastIndexOf(slashie);
	    
	    if (index > -1) {
		prefix = file.substring(0, index+1);
	    }

	    mh.localDataFile = prefix + mh.dataFile;
	}

	return mh;
    }
    
    
    /**
     * Read the text header from an input stream. 
     *
     */
    public static MetaImageHeader readHeader(DataInput din) throws IOException {

	MetaImageHeader mh = new MetaImageHeader();

	boolean gotDataFile = false;

	//  Unix newline, ie one character
	byte newLine = new String("\n").getBytes()[0];


	// Windows uses \r\n for a new line
	byte cr = new String("\r").getBytes()[0];


	// count how many bytes are in the header
	int hdrBytes = 0;
	

	while (!gotDataFile) {

	    String headerKey = "";
	    String headerValue = "";

	    byte[] lineBytes = new byte[2048];
	    
	    // bytes read from this line
	    int bytesRead = 0;

	    readLine: 
	    while (true) { // EOFException thrown at end of file, which we should not hit
		
		byte b = din.readByte();
		
		lineBytes[bytesRead++] = b; 
		
		if (b == newLine) {
		    hdrBytes += bytesRead;
		    break readLine;
		}
	    }

	    int lineLength = bytesRead - 1; 

 	    // remove \r from line written under Windows
 	    if (lineBytes[bytesRead - 1] == cr) {
 		lineLength -= 1;
 	    }

	    String line = new String(lineBytes, 0, lineLength, "US-ASCII");

	    // split on [spaces]=[spaces] where there might be zero spaces
	    // this actually splits on = and gets rid of spaces
	    // leading and trailing spaces removed first by the call to trim
	    String[] headerKeyValue = line.trim().split("\\s*=\\s*");

	    if (headerKeyValue.length != 2) {
		throw new LoggedException("Header line\n" + line + "\nis not a valid Meta header " +
					  " field. Meta fields are of the form \"Key = Value\"");
	    }

	    mh.setHeaderValue(headerKeyValue[0].trim(), headerKeyValue[1].trim());

	    // ElementDataFile is always last
	    if (headerKeyValue[0].equals("ElementDataFile")) {
		gotDataFile = true;
	    }
	}

	// if data file is local, set dataByteOffset
	if (mh.dataFile.equals("LOCAL")) {
	    mh.dataByteOffset = hdrBytes;
	}

	return mh;
	    
    }

   
    /**
     * Parses header values from their String representation. Warns, but does not throw an Exception,
     * if there is an unrecognized key.
     *
     */
    public void setHeaderValue(String key, String value) {

	if (key.equalsIgnoreCase("ObjectType")) {
	    ObjectType type = ObjectType.getObjectType(value);
	    
	    if (type != ObjectType.IMAGE) {
		throw new LoggedException("Only image type is supported");
	    }

	}
	else if (key.equalsIgnoreCase("NDims")) {
	    int dims = Integer.parseInt(value);
	    
	    if (dims != 3) {
		throw new LoggedException("only 3D images are supported.");
	    }
	}
	else if (key.equalsIgnoreCase("BinaryData")) {
	    boolean binary = Boolean.parseBoolean(value);

	    if (!binary) {
		throw new LoggedException("ASCII data is not supported.");
	    }
	}
	else if (key.equalsIgnoreCase("CompressedData")) {
	    compressedData = Boolean.parseBoolean(value);
	}
	else if (key.equalsIgnoreCase("BinaryDataByteOrderMSB")) {
	    intelByteOrder = !(Boolean.parseBoolean(value));
	}
	else if (key.equalsIgnoreCase("ElementByteOrderMSB")) {
	    // it is unclear what the difference is between ElementByteOrderMSB and BinaryDataByteOrderMSB
	    // if both are specified we go with the later one
	    intelByteOrder = !(Boolean.parseBoolean(value));
	}
	else if (key.equalsIgnoreCase("TransformMatrix")) {
	    String[] values = value.split("\\s+");

	    transformation = new RealMatrix(3,3);

	    for (int i = 0; i < 9; i++) {
		transformation.entries[i / 3][i % 3] = Double.parseDouble(values[i]);
	    }
	}
	else if (key.equalsIgnoreCase("Offset")) {

	    String[] values = value.split("\\s+");

	    offset = new double[3];

	    for (int i = 0; i < 3; i++) {
		offset[i] = Double.parseDouble(values[i]);
	    }
	}
	else if (key.equalsIgnoreCase("Position")) {

	    String[] values = value.split("\\s+");

	    offset = new double[3];

	    for (int i = 0; i < 3; i++) {
		offset[i] = Double.parseDouble(values[i]);
	    }
	}
	else if (key.equalsIgnoreCase("Origin")) {

	    String[] values = value.split("\\s+");

	    offset = new double[3];

	    for (int i = 0; i < 3; i++) {
		offset[i] = Double.parseDouble(values[i]);
	    }
	}
	else if (key.equalsIgnoreCase("CenterOfRotation")) {

	    String[] values = value.split("\\s+");

	    rotationCentre = new double[3];

	    for (int i = 0; i < 3; i++) {
		rotationCentre[i] = Double.parseDouble(values[i]);
	    }
	}
	else if (key.equalsIgnoreCase("AnatomicalOrientation")) {
	    anatomicalOrientation = AnatomicalOrientation.getAnatomicalOrientation(value);
	}
	else if (key.equalsIgnoreCase("ElementSpacing")) {
	    
	    String[] values = value.split("\\s+");

	    spacing = new double[3];

	    for (int i = 0; i < 3; i++) {
		spacing[i] = Double.parseDouble(values[i]);
	    }
	    
	}
	else if (key.equalsIgnoreCase("DimSize")) {
	    
	    String[] values = value.split("\\s+");

	    dimSize = new int[3];

	    for (int i = 0; i < 3; i++) {
		dimSize[i] = Integer.parseInt(values[i]);
	    }
	    
	}
	else if (key.equalsIgnoreCase("ElementNumberOfChannels")) {
	    channels = Integer.parseInt(value);
	}
	else if (key.equalsIgnoreCase("ElementType")) {
	    dataType = DataType.getDataType(value);
	}
	else if (key.equalsIgnoreCase("ElementDataFile")) {
	    dataFile = value;
	}
	else {
	    logger.warning("Unrecognized header field " + key + " ignored");
	}

    }
	


    /**
     * Writes header to a file. Expects either a .mha or .mhd extension, which is
     * added if omitted. The Meta convention is to store everything in a .mhd file,
     * or use two files: header.mha and data.mhd.
     *
     */
    public void writeHeader(String file) throws IOException {

	if (!file.endsWith(".mha") && !file.endsWith(".mhd")) {
	    
	    if (dataFile.equalsIgnoreCase("LOCAL")) {
		file = file + ".mhd";
	    }
	    else {
		file = file + ".mha";
	    }
	    
	}

	DataOutputStream dout = 
	    new DataOutputStream(new BufferedOutputStream(new FileOutputStream(file), 1024));

	writeHeader(dout);

	dout.close();
    }


    public void writeHeader(DataOutput dout) throws IOException {
        
	dout.write(new String("ObjectType = " + objectType + "\n").getBytes("US-ASCII"));
	dout.write(new String("NDims = " + nDims + "\n").getBytes("US-ASCII"));
	dout.write(new String("BinaryData = " + titleCaseBool(binaryData) +
				   "\n").getBytes("US-ASCII"));
	
	dout.write(new String("CompressedData = " + titleCaseBool(compressedData) +
				   "\n").getBytes("US-ASCII"));
	
	dout.write(new String("BinaryDataByteOrderMSB = " + titleCaseBool(!intelByteOrder) + 
				   "\n").getBytes("US-ASCII"));

	dout.write(new String("TransformMatrix = " + flatMatrixString(transformation) + 
				   "\n").getBytes("US-ASCII"));

	dout.write(new String("Offset = " + offset[0] + " " + offset[1] + " " + offset[2] + 
			      "\n").getBytes("US-ASCII"));
	
	dout.write(new String("CenterOfRotation = " + rotationCentre[0] + " " + rotationCentre[1] + " " +
			      rotationCentre[2] + "\n").getBytes("US-ASCII"));
	
	dout.write(new String("AnatomicalOrientation = " + anatomicalOrientation + 
			      "\n").getBytes("US-ASCII"));
	
	String voxelDimString = "ElementSpacing = " + spacing[0] + " " + spacing[1] + " " + 
	    spacing[2] + "\n"; 
	
	dout.write(voxelDimString.getBytes("US-ASCII"));
	
	String dataDimString = "DimSize = " + dimSize[0] + " " + dimSize[1] + " " + dimSize[2] + "\n"; 
	
	dout.write(new String(dataDimString).getBytes("US-ASCII"));
	
	dout.write(new String("ElementNumberOfChannels = " + channels + "\n").getBytes("US-ASCII"));
	dout.write(new String("ElementType = " + dataType + "\n").getBytes("US-ASCII"));
	dout.write(new String("ElementDataFile = " + dataFile + "\n").getBytes("US-ASCII"));
	
    }


    /**
     * Gets the Camino data type string
     */
    public String caminoDataTypeString() {

	if (dataType == DataType.UCHAR) {
	    return "char";
	}
	if (dataType == DataType.CHAR) {
	    return "byte";
	}
	if (dataType == DataType.SHORT) {
	    return "short";
	}
	if (dataType == DataType.INT) {
	    return "int";
	}
	if (dataType == DataType.FLOAT) {
	    return "float";
	}
	if (dataType == DataType.DOUBLE) {
	    return "double";
	}
	if (dataType == DataType.USHORT) {
	    return "ushort";
	}
	if (dataType == DataType.UINT) {
	    return "uint";
	}
	if (dataType == DataType.LONG) {
	    return "long";
	}
	

	throw new LoggedException("File does not have a supported Camino data type");

    }


    
    /**
     * @return "True" if b, "False" if !b.
     */
    private static String titleCaseBool(boolean b) {
	if (b) {
	    return "True";
	}
	else {
	    return "False";
	}
    }
    

    /**
     * @return a String representation of the matrix <code>m</code>, with all
     * rows concatenated into a single line.
     */
    private static String flatMatrixString(RealMatrix m) {
	String line = "";

	double[][] matrix = m.entries;

	for (int r = 0; r < m.rows(); r++) {
	    for (int c = 0; c < m.columns(); c++) {
		line += matrix[r][c] + " ";
	    }
	}

	return line.trim();
    }


    // ImageHeader implementation methods

    public int xDataDim() {
	return dimSize[0];
    }

    public int yDataDim() {
	return dimSize[1]; 
    }

    public int zDataDim() {
	return dimSize[2];
    }

    public int[] getDataDims() {
	return new int[] {dimSize[0], dimSize[1], dimSize[2]};
    }

    public double xVoxelDim() {
	return spacing[0];
    }

    public double yVoxelDim() {
	return spacing[1];
    }

    public double zVoxelDim() {
	return spacing[2];
    }

    public double[] getVoxelDims() {
	return new double[] {spacing[0], spacing[1], spacing[2]};
    }

    public int components() {
	return channels > 0 ? channels : 1;
    }

    public double[] getOrigin() {
	return new double[] {offset[0], offset[1], offset[2]};
    }


    /**
     * @return a VoxelOrderDataSource for the image data.
     *
     */
    public DataSource getImageDataSource() {

	if (compressedData) {
	    
	    logger.warning("Compressed Meta IO is untested, use with caution");

	    try {
		// set up an input stream, then skip header bytes
		FileInputStream fin = new FileInputStream(localDataFile);
		
		int bytesSkipped = 0;
		
		while (bytesSkipped < dataByteOffset) {
		    fin.skip(dataByteOffset - bytesSkipped);
		}
		
		// now buffer on the I/O side
		ZipInputStream zin = new ZipInputStream(new BufferedInputStream(fin, ExternalDataSource.FILEBUFFERSIZE/2));
		
		// assume there is one entry (ie one compressed file in the ZIP archive)
		zin.getNextEntry();
		
		// and buffer on the decompressed side for speed
		EndianNeutralDataInputStream dataIn = 
		    new EndianNeutralDataInputStream(new BufferedInflaterInputStream(zin, 
										     ExternalDataSource.FILEBUFFERSIZE/2), 
						     intelByteOrder);
		
		
		return new VoxelOrderDataSource(dataIn, channels, caminoDataTypeString());

	    }
	    catch (IOException e) {
		throw new LoggedException(e);
	    }
	}
	else {
	    VoxelOrderDataSource dataSource = 
		new VoxelOrderDataSource(localDataFile, channels, 
					 caminoDataTypeString(), intelByteOrder, dataByteOffset);
	    
	    return dataSource;
	}
    }
  


    /**
     * Reads a data array, assuming voxel-ordered data for multi-channel data.
     *
     * @return a 4D array with extent [xDataDim][yDataDim][zDataDim][components()].
     */
    public double[][][][] readVolumeData() {
	
	int xDataDim = dimSize[0];
	int yDataDim = dimSize[1];
	int zDataDim = dimSize[2];

	double[][][][] data = new double[xDataDim][yDataDim][zDataDim][channels];
	
	try {
	    
	    DataSource din = getImageDataSource();
	    
	    for (int k = 0; k < zDataDim; k++) { 
		for (int j = 0; j < yDataDim; j++) {
		    for (int i = 0; i < xDataDim; i++) {
			double[] nextVoxel = din.nextVoxel();

			for (int c = 0; c < channels; c++) {
			    data[i][j][k][c] = nextVoxel[c];
			}
		    }
		}
	    }
	    
	}
	catch(DataSourceException e) {
	    throw new LoggedException(e);
	} 

	return data;
	
    }



    public double[][][] readVolume(int index) {
	

	if (!( (index >= 0) && (index < components()) )) {
	    throw new LoggedException("Attempted to read non-existent component " + index);
	}


	int xDataDim = dimSize[0];
	int yDataDim = dimSize[1];
	int zDataDim = dimSize[2];

	double[][][] data = new double[xDataDim][yDataDim][zDataDim];
	
	try {
	    
	    DataSource din = getImageDataSource();
	    
	    for (int k = 0; k < zDataDim; k++) { 
		for (int j = 0; j < yDataDim; j++) {
		    for (int i = 0; i < xDataDim; i++) {
			double[] nextVoxel = din.nextVoxel();
			
			data[i][j][k] = nextVoxel[index];
			
		    }
		}
	    }
	    
	}
	catch(DataSourceException e) {
	    throw new LoggedException(e);
	} 

	return data;
	
    }
      



    
    
}
