package imaging;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>General interface classes representing acquisition schemes
 * 
 * <dt>Description:
 * 
 * <dd>Set and get methods for all the acquisition parameters and
 * various data operations.
 * 
 * </dl>
 * 
 * @author Danny Alexander
 * @version $Id: Scheme.java,v 1.4 2009/03/27 01:26:41 bennett Exp $
 *  
 */
public interface Scheme {

    public static final int [] gradXYZ = {0,1,2};

    public static final int [] gradXZY = {0,2,1};

    public static final int [] gradYXZ = {1,0,2};

    public static final int [] gradYZX = {1,2,0};

    public static final int [] gradZXY = {2,0,1};

    public static final int [] gradZYX = {2,1,0};

    /**
     * Returns the number of measurements in each voxel.
     * 
     * @return The number of measurements in each voxel.
     */
    public int numMeasurements();

    /**
     * Returns the number of diffusion weighted measurements in each voxel.
     * 
     * @return The number of diffusion weighted measurements in each voxel.
     */
    public int numDWMeasurements();
    public boolean isIgnored(int j);
    
    /**
     * Returns the number of q=0 measurements in each voxel.
     * 
     * @return The number of q=0 measurements in each voxel.
     */
    public int numZeroMeasurements();


    /**
     * Returns one wavenumber from the list.
     * 
     * @param i
     *            The index of the wavenumber to return.
     * 
     * @return The i-th wavenumber.
     */
    public double[] getQ(int i);


    /**
     * Returns the modulus of one wavenumber from the list.
     * 
     * @param i
     *            The index of the wavenumber to return.
     * 
     * @return |q| of the i-th wavenumber.
     */
    public double getModQ(int i);


    /**
     * Returns the mean non-zero radial wavenumber.
     * 
     * @return The mean non-zero |q|.
     */
    public double getMeanNonZeroModQ();


    /**
     * Returns the diffusion time for one measurement.
     * 
     * @param i The index of the measurement.
     *
     * @return The diffusion time.
     */
    public double getDiffusionTime(int i);


    /**
     * Returns the diffusion times for all measurements.
     * 
     * @return Array of diffusion times.
     */
    public double[] getDiffusionTimes();


    /**
     * Returns the diffusion times for all measurements with nonzero q. 
     * 
     * 
     * @return <code>taus</code>, array of diffusion times, where <code>taus[i]</code> corresponds
     * to element <code>i</code> in the array returned from <code>normalizeData</code>.
     */
    public double[] getNonZeroQ_DiffusionTimes();


    /**
     * Sets the diffusion time for one measurement.
     * 
     * @param diffusionTime The new diffusion time.
     * 
     * @param i The index of the measurement.
     */
    public void setDiffusionTime(double diffusionTime, int i);


    /**
     * Returns the list of normalized non-zero qs. Zero qs are removed
     * and the remaining qs are normalized by the mean non-zero |q|.
     * 
     * @return {{q1x, q1y, q1z}, {q2x, q2y, q2z}, ..., {q(N-M)x, q(N-M)y,
     *         q(N-M)z}}
     */
    public double[][] getNormNonZeroQs();
 

    /**
     * Returns the list of non-zero qs. Zero qs at the start of the
     * list are removed and the list contains the remaining qs.
     * 
     * @return {{q1x, q1y, q1z}, {q2x, q2y, q2z}, ..., {q(N-M)x, q(N-M)y,
     *         q(N-M)z}}
     */
    public double[][] getNonZeroQs();


    /**
     * Normalizes a voxel set of measurements acquired with this
     * sequence by the geometric mean of the q=0 measurements.  If the
     * mean is zero, the normalization does nothing.
     * 
     * @params data Unnormalized set of measurements.
     * 
     * @return Normalized measurements. Only data with q != 0 are returned.
     */
    public double[] normalizeData(double[] data);


    /**
     * Computes the geometric mean of the q=0 measurements.
     * 
     * @params data Unnormalized set of measurements.
     * 
     * @return The geometric mean.
     */
    public double geoMeanZeroMeas(double[] data);


    /**
     * Gets the duration of gradient blocks for the i-th measurement.
     *
     * @param i The index of the measurement.
     *
     * @return delta
     */
    public double getDelta(int i);


    /**
     * Sets the duration of gradient blocks for the i-th measurement.
     * 
     * @param delta gradient block duration
     *
     * @param i The index of the measurement.
     */
    public void setDelta(double delta, int i);


    /**
     * Gets the separation of gradient block onsets for the i-th
     * measurement.
     *
     * @param i The index of the measurement.
     *
     * @return DELTA
     */
    public double getDELTA(int i);


    /**
     * Sets the duration of gradient block onsets for the i-th
     * measurement.
     * 
     * @param DELTA gradient block separation
     *
     * @param i The index of the measurement.
     */
    public void setDELTA(double delta, int i);


    /** 
     * Gets the strength of the gradient for the i-th measurement.
     * 
     * @param i The index of the measurement.
     *
     * @return gradient strength
     */
    public double getModG(int i);


    /**
     * Sets new value for the gradient strength for the i-th measurement.
     * 
     * @param modG gradient strength
     *
     * @param i The index of the measurement.
     */
    public void setModG(double modG, int i);


    /** 
     * Gets time of first gradient block onset for the i-th measurement.
     * 
     * @param i The index of the measurement.
     *
     * @return time of start of block
     */
    public double getHalfP90(int i);


    /** 
     * Sets time of first gradient block onset for the i-th measurement.
     * 
     * @param t1 block start time in ms
     *
     * @param i The index of the measurement.
     */
    public void setHalfP90(double halfP90, int i);


    /** 
     * Gets the number of measurements at the end of the sequence that
     * are ignored in every voxel.
     * 
     * @return the number of measurements
     */
    public int getIgnore();


    /** 
     * Sets the number of measurements at the end of the sequence that
     * are ignored in every voxel.
     * 
     * @param i The number of measurements to ignore.
     */
    public void setIgnore(int i);

    
    /**
     * tells the calling function what type of scheme object this is
     * 
     * @return 0 for V0, 1 for V1 etc
     */
    public int getVersion();


    /**
     * Get a scheme composed of a subset of the measurements in this scheme.
     *
     * @param indices integer indices of the measurements to include in the subset.
     */
    public Scheme getSubsetScheme(int[] indices);

    /**
     * Negates the X component of the gradient directions
     */
    public void flipX();

    /**
     * Negates the Y component of the gradient directions
     */
    public void flipY();

    /**
     * Negates the Z component of the gradient directions
     */
    public void flipZ();

    /**
     * @param order the new order of the gradients
     *
     * Reorders the gradient directions
     */
    public void gradOrder(int [] order);

	public int numIgnoredMeasurements();    
}
