package inverters;

import optimizers.*;
import misc.*;

/**
 * <dl>
 * 
 * <dt>Purpose:
 * 
 * <dd>Non-linear least squares fitter of a three tensor model with one prolate
 * axisymmetric tensor to DW-MR data.
 * 
 * <dt>Description:
 * 
 * <dd>This class implements the abstract methods of MarquardtChiSqFitter to
 * provide a Levenburg-Marquardt algorithm for fitting a three tensor model with
 * one prolate axisymmetric tensor to DW-MR measurements.
 * 
 * </dl>
 * 
 * $Id: ThreeTensorOneAxiSymFitter.java,v 1.1 2008/12/08 17:48:43 bennett Exp $
 * 
 * @author Danny Alexander
 *  
 */
public class ThreeTensorOneAxiSymFitter extends ThreeTensorCholFitter {

    /**
     * Default constructor does nothing.
     */
    public ThreeTensorOneAxiSymFitter() {
    }

    /**
     * The constructor requires a list of independent values (indepVals) and
     * associated dependent values (depVals) - these are the data. Together with
     * the number of unweighted acquisitions that are made (nob0s), which is
     * required to estimate the noise levels of each data item. The b-value used
     * in the imaging sequence is also required.
     * 
     * @param indepVals
     *            The matrix of wavenumbers q without the zero wavenumbers.
     * 
     * @param depVals
     *            The normalized measurements.
     * 
     * @param diffusionTimes
     *            The array of diffusion times.
     * 
     * @param nob0s
     *            The number of q=0 measurements.
     */
    public ThreeTensorOneAxiSymFitter(double[][] indepVals, double[] depVals,
            double[] diffusionTimes, int nob0s) throws MarquardtMinimiserException {
        noParams = 18;
        dt1StartIndex = 1;
        dt2StartIndex = 5;
        dt3StartIndex = 11;
        mixPar1Index = 17;
        mixPar2Index = 18;
        initialize(indepVals, depVals, diffusionTimes, nob0s);
    }

    /**
     * Overridden as there is no need to compute the axisymmetric diffusion
     * tensor.
     */
    protected double[] initParams1() {
        double traceD1 = 2.1E-9;
        double[] params = new double[4];
        for (int i = 0; i < 3; i++) {
            params[i] = 0.0;
        }
        params[3] = Math.sqrt(traceD1 / 3.0);

        return params;
    }

    /**
     * Overridden as there is no need to compute the axisymmetric diffusion
     * tensor.
     */
    protected double[] fAndBetaToParams1(double[] fBeta) {
        double[] params = new double[4];

        params[0] = fBeta[0];
        params[1] = fBeta[1];
        params[2] = fBeta[2];
        params[3] = Math.sqrt(fBeta[3]);

        return params;
    }

    /**
     * Overridden to replace the first DT by the axisymmetric representation.
     */
    protected void insertTensorParamDerivs(double[] derivs, double[] atry, double[] q,
            double d1Contrib, double d2Contrib, double d3Contrib, double p1, double p2, double tau) {

        double s = atry[dt1StartIndex + 3];
        double sModQSq = s * (q[0] * q[0] + q[1] * q[1] + q[2] * q[2]);

        double qDotF1 = q[0] * atry[dt1StartIndex] + q[1] * atry[dt1StartIndex + 1]
                + q[2] * atry[dt1StartIndex + 2];

        double qDotRow21 = q[0] * atry[dt2StartIndex] + q[1] * atry[dt2StartIndex + 1]
                + q[2] * atry[dt2StartIndex + 2];
        double qDotRow22 = q[1] * atry[dt2StartIndex + 3] + q[2]
                * atry[dt2StartIndex + 4];
        double qDotRow23 = q[2] * atry[dt2StartIndex + 5];

        double qDotRow31 = q[0] * atry[dt3StartIndex] + q[1] * atry[dt3StartIndex + 1]
                + q[2] * atry[dt3StartIndex + 2];
        double qDotRow32 = q[1] * atry[dt3StartIndex + 3] + q[2]
                * atry[dt3StartIndex + 4];
        double qDotRow33 = q[2] * atry[dt3StartIndex + 5];

        insertAxiSymDerivs(derivs, tau, dt1StartIndex, q, sModQSq, qDotF1, p1, d1Contrib);
        insertCholDerivs(derivs, tau, dt2StartIndex, q, qDotRow21, qDotRow22, qDotRow23,
                p2, d2Contrib);
        insertCholDerivs(derivs, tau, dt3StartIndex, q, qDotRow31, qDotRow32, qDotRow33,
                1 - p1 - p2, d3Contrib);

    }

    /**
     * Overridden to use the axisymmetric representation.
     */
    protected DT getDT1(double[] a) {
        return getDT_AxiSym(a, dt1StartIndex);
    }

    /**
     * Overridden to use the axisymmetric representation.
     */
    protected double[] dt1ToParams(DT dt1) {
        return getAxiSymParams(dt1);
    }

}
