/* StepGeneratorFactory.java created on 28-Nov-2005
 * (simulation)
 * 
 * author: Matt Hall (m.hall@cs.ucl.ac.uk)
 * 
 */
package simulation.dynamics;

import imaging.Scheme;
import java.util.logging.Logger;

import misc.LoggedException;

import simulation.DiffusionSimulation;
import simulation.SimulationParams;

/**
 * Camino fibre reconstruction and tracking toolkit
 * 
 * StepGeneratorFactory (simulation)
 * build step generators from given list of types.
 * 
 * the simulation step generator is a singleton. if 
 *
 * @author Matt Hall (m.hall@cs.ucl.ac.uk)
 *
 */
public class StepGeneratorFactory {

    /** identifier for fixed length step generator */
    public static final int FIXEDLENGTH=1;
    
    /** identifier for spatially variable */
    public static final int SPATIALLYVARYING=2;
    
    /** logging object */
    private static Logger logger=Logger.getLogger("simulation.StepGeneratorFactory");
    
    /** current active step generator singleton */
    private static StepGenerator stepGen= null;
    
    
    /** factory method */
    public static final StepGenerator getStepGenerator(SimulationParams simParams){
    
    	int type= simParams.getStepType();
    	
        if(type==FIXEDLENGTH){
        	if(StepGeneratorFactory.stepGen==null){
        		logger.info("instantiating fixed length step generator");
        		
        		StepGenerator stepGen= new FixedLengthStepGenerator(simParams);
        		StepGeneratorFactory.stepGen=stepGen;
        		
        		return stepGen;
        	}
        	else{
        		if(stepGen.getType()==FIXEDLENGTH){
        			return stepGen;
        		}
        		else{
        			throw new LoggedException("step generator singleton is of type "+stepGen.getType()+
        										" but fixed length generator was requested from factory.");
        		}
        	}
        }
        else{
            String errMess=new String("unknown diffusion simulation step generator type code "+ type);
            
            logger.severe(errMess);
            throw new RuntimeException(errMess);
        }
    }
        
    
    /** 
     * parameters array generator
     *  
     * @param type the type of step generator being manufactured
     * @param simParams the simulation parameters
     * @param imParams the imaging parameters
     * 
     * @return the appropriate array of parameters for the desired
     *         step generator given the simulation and imaging
     *         parameters for the current simualtion
     */
    public static final double[] getStepParamsArray(int type, SimulationParams simParams, Scheme imParams){
        
        if(type==FIXEDLENGTH){
        	// step length is sqrt((2d)D dt) where d is dimensionality of system, 
        	//D diffusivity and dt timestep duration 
            double stepLength= Math.sqrt(6.0*DiffusionSimulation.DIFF_CONST * simParams.getDt());
            
            return new double[] {stepLength};
        }
        else{
            logger.warning("unknown step generator type "+type+" paramters array " +
            		"for fixed length step generator will be returned instead.");
            
            return getStepParamsArray(FIXEDLENGTH, simParams, imParams);
            
        }
        
    }
    
    
    
    }
